# CLAUDE.md - Bongo Profile Package

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Overview

The Bongo Profile package provides user profile management functionality for the admin area of Laravel applications. Authenticated employees can view and update their personal information including contact details, password, and avatar image.

**Package**: `bongo/profile`
**Namespace**: `Bongo\Profile`
**Version**: 3.0+ (PHP 8.2+, Laravel 10+)

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Comprehensive architecture documentation including class diagrams, data flow, and system design
- **[.cursorrules](.cursorrules)** - Detailed Cursor AI instructions with coding conventions and common tasks
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions with code templates and patterns

## Quick Reference

### Key Files

| File | Purpose | Key Classes/Functions |
|------|---------|---------------------|
| `src/ProfileServiceProvider.php` | Service provider | `ProfileServiceProvider` extends `AbstractServiceProvider` |
| `src/Http/Controllers/Backend/ProfileController.php` | Main profile operations | `index()`, `edit()`, `update()` |
| `src/Http/Controllers/Backend/ProfileImageController.php` | Avatar upload handler | `upload()` returns JSON |
| `src/Routes/backend.php` | Route definitions | Routes auto-registered with `auth` + `employee` middleware |
| `src/Views/backend/index.blade.php` | Profile view page | Displays user details with avatar |
| `src/Views/backend/edit.blade.php` | Profile edit form | Form wrapper with CSRF protection |
| `src/Views/backend/partials/form/details.blade.php` | Edit form fields | Name, contact, password, avatar fields |
| `src/Views/backend/partials/tab_detail.blade.php` | View tab content | Read-only field display |
| `src/Translations/en/backend.php` | English translations | Translation keys for admin area |
| `tests/TestCase.php` | Base test case | Orchestra Testbench setup |

### Available Commands

```bash
# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse

# Install dependencies
composer install

# Update dependencies
rm -f composer.lock && composer update -W
```

## Architecture Summary

### Service Provider Pattern

```php
class ProfileServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'profile';
}
```

The `$module` property enables automatic bootstrapping:
- **Routes**: Loaded from `src/Routes/backend.php` with `backend.*` prefix and `['web', 'auth', 'employee']` middleware
- **Views**: Registered with `profile::` namespace from `src/Views/`
- **Translations**: Registered with `profile::` namespace from `src/Translations/`

### Route Structure

| Method | URI | Name | Controller Action |
|--------|-----|------|------------------|
| GET | `/admin/profile` | `backend.profile.index` | View profile |
| GET | `/admin/profile/edit` | `backend.profile.edit` | Edit form |
| POST | `/admin/profile/update` | `backend.profile.update` | Update profile |
| POST | `/admin/profile/image` | `backend.profile.image` | Upload avatar (AJAX) |

### Controller Classes

```
ProfileController
├── index()            → Returns view('profile::backend.index')
├── edit()             → Returns view('profile::backend.edit')
└── update($request)   → Validates, updates user, redirects with flash message

ProfileImageController
└── upload($request)   → Processes avatar via AvatarImage, returns JSON
```

### Dependencies

- **bongo/framework**: `AbstractServiceProvider`, `AbstractController`
- **bongo/user**: `User` model
- **bongo/image**: `AvatarImage` service, `StoreImageRequest` validation

### Key Helper Functions

- `user()` - Get authenticated user instance
- `trans()` - Translate string (e.g., `trans('profile::backend.update_success')`)
- `route()` - Generate route URL (e.g., `route('backend.profile.index')`)

## Code Style Summary

### Validation Pattern

```php
$request->validate([
    'first_name' => 'nullable|max:70',
    'last_name' => 'nullable|max:70',
    'email' => "required|string|email:rfc,dns|unique:{$userTable},email,".user()->id.',id,deleted_at,NULL|max:50',
    'password' => 'sometimes|nullable|string|min:8|confirmed|regex:/^(?=.*?[A-Z])(?=.*?[a-z])(?=.*?[0-9])(?=.*?[#?!@$%^&*-]).{6,}$/',
]);
```

**Password Requirements**:
- Minimum 8 characters
- At least 1 uppercase, 1 lowercase, 1 number, 1 special character

### Password Update Pattern

```php
if ($request->has('password') && !empty($request->get('password'))) {
    user()->password = Hash::make($request->get('password'));
}
user()->save();
```

### Redirect with Flash Message

```php
return redirect()
    ->route('backend.profile.index')
    ->success(trans('profile::backend.update_success'));
```

### Image Upload Pattern

```php
public function upload(StoreImageRequest $request): JsonResponse
{
    $files = request()->file('files');

    try {
        $user = (new AvatarImage(user(), $files[0]))->save();
        return response()->json([$user->getAvatar()]);
    } catch (Exception $e) {
        return response()->json(['error' => $e->getMessage()], 500);
    }
}
```

### View Patterns

```php
// Extend layout
@extends('framework::backend.layouts.app')

// Content header with actions
@component('blade::content_header')
    {{ $title }}
    @slot('actions')
        @backButton(['name' => 'Back'])
        @editButton(['name' => 'Edit', 'url' => route('backend.profile.edit')])
    @endslot
@endcomponent

// Content body
@component('blade::content_body')
    <!-- Content -->
@endcomponent

// Form fields
@input(['name' => 'first_name', 'value' => $user->first_name])
@static(['name' => 'Status', 'value' => ucfirst($user->status)])

// File uploader
<file-uploader
    url="{{ route('backend.profile.image') }}"
    name="user_avatar"
    height="100"
></file-uploader>
```

## Common Tasks

### Adding a New Field

1. **Add to edit form** (`src/Views/backend/partials/form/details.blade.php`):
```php
@input(['name' => 'new_field', 'value' => $user->new_field])
```

2. **Add to view tab** (`src/Views/backend/partials/tab_detail.blade.php`):
```php
@static(['name' => 'New Field', 'value' => $user->new_field])
```

3. **Add validation** (`ProfileController::update()`):
```php
'new_field' => 'nullable|max:100',
```

4. **Add translation** (`src/Translations/en/backend.php`):
```php
'new_field' => 'New Field',
```

### Adding a New Route

Add to `src/Routes/backend.php`:
```php
Route::get('custom', [ProfileController::class, 'custom'])->name('custom');
```

Result: Route becomes `backend.profile.custom` at `/admin/profile/custom`

### Modifying Password Requirements

Edit regex in `ProfileController::update()`:
```php
'password' => 'sometimes|nullable|string|min:12|confirmed|regex:/your-regex/'
```

### Customising Avatar Processing

Override `ProfileImageController::upload()`:
```php
$user = (new AvatarImage(user(), $files[0]))
    ->setWidth(200)
    ->save();
```

## Testing

### Base Test Class

```php
namespace Bongo\Profile\Tests;

class YourTest extends TestCase
{
    public function test_feature()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user)
            ->get(route('backend.profile.index'));

        $response->assertOk();
    }
}
```

### Running Tests

```bash
vendor/bin/phpunit
vendor/bin/phpunit --filter test_name
```

## Security Considerations

- **Authentication**: All routes require `auth` middleware
- **Authorization**: All routes require `employee` middleware (staff role)
- **CSRF**: All forms include `{{ csrf_field() }}`
- **Password Hashing**: Uses `Hash::make()` (bcrypt)
- **Strong Passwords**: Enforced via regex validation
- **Email Validation**: Includes DNS validation and uniqueness check
- **Soft Deletes**: Email uniqueness excludes soft deleted users

## Integration Points

### User Model Requirements

The package expects `Bongo\User\Models\User` to have:
- Properties: `first_name`, `last_name`, `email`, `password`, `telephone`, `mobile`, `title`, `status`
- Methods: `getAvatar()`, `fill()`, `save()`

### Image Service Integration

Uses `Bongo\Image\Services\AvatarImage`:
```php
$user = (new AvatarImage($user, $uploadedFile))->save();
```

### Blade Components

Uses components from `bongo/blade`:
- `@component('blade::content_header')`, `@component('blade::card')`
- `@input()`, `@static()`, `@label()`
- `@backButton()`, `@editButton()`, `@saveButton()`
- `<file-uploader>`, `<tabs>`, `<tab>`

## Package Structure

```
default/profile/
├── src/
│   ├── Http/Controllers/Backend/
│   │   ├── ProfileController.php              # Main CRUD operations
│   │   └── ProfileImageController.php         # Avatar upload
│   ├── Routes/backend.php                     # Admin routes
│   ├── Translations/en/backend.php            # Translations
│   ├── Views/backend/                         # Blade templates
│   └── ProfileServiceProvider.php             # Service provider
├── tests/TestCase.php                         # Base test class
├── composer.json                              # Package metadata
├── phpunit.xml                                # PHPUnit config
└── README.md                                  # Package readme
```

## Extending AbstractServiceProvider

The package extends `Bongo\Framework\Providers\AbstractServiceProvider`:

**What it provides**:
- Automatic route registration from `src/Routes/backend.php`
- Automatic view registration with `profile::` namespace
- Automatic translation registration with `profile::` namespace
- Automatic middleware application (`web`, `auth`, `employee` for backend routes)

**Configuration**:
```php
protected string $module = 'profile';  // Used for namespaces and paths
```

**Optional arrays** (not used in this package):
```php
protected array $middlewares = [];     // Custom middleware registration
protected array $commands = [];        // Artisan command registration
protected array $listeners = [];       // Event listener registration
protected array $subscribers = [];     // Event subscriber registration
protected array $composers = [];       // View composer registration
```

## Translation Keys

**Namespace**: `profile::backend`

| Key | Value |
|-----|-------|
| `index` | Profile |
| `edit` | Edit |
| `update` | Update |
| `update_success` | Profile successfully updated! |
| `update_failed` | Unable to update profile! |

Usage: `trans('profile::backend.update_success')`

## Common Issues

### Email Validation Fails
Ensure user table name is correct:
```php
$userTable = (new User())->getTable();
```

### Avatar Upload Fails
Check:
- `bongo/image` package installed
- User model has `getAvatar()` method
- File upload permissions correct
- `StoreImageRequest` validation passes

### Routes Not Registered
Verify:
- `ProfileServiceProvider` registered in application
- `$module` property set to `'profile'`
- Routes file at `src/Routes/backend.php`

## Related Documentation

- **ARCHITECTURE.md**: Comprehensive architecture documentation with class diagrams and data flow
- **.cursorrules**: Detailed coding conventions and development guidelines
- **.github/copilot-instructions.md**: Code templates and patterns for GitHub Copilot
- **README.md**: Package installation and basic usage

## Key Conventions

1. Controllers extend `AbstractController`
2. Service provider extends `AbstractServiceProvider` with `$module = 'profile'`
3. Routes use explicit controller action syntax: `[Controller::class, 'method']`
4. Views use `profile::` namespace
5. Translations use `profile::backend.` prefix
6. Forms include CSRF protection: `{{ csrf_field() }}`
7. Type hints on method parameters and return types
8. Password hashing with `Hash::make()`
9. Helper function `user()` for authenticated user
10. Flash messages via `->success()` method on redirect
