# GitHub Copilot Instructions - Bongo Profile Package

## Project Overview

This Laravel package provides user profile management functionality for the admin area. It allows authenticated employees to view and update their personal details, including name, contact information, password, and avatar image.

**Package Name**: `bongo/profile`
**Namespace**: `Bongo\Profile`
**Requirements**: PHP 8.2+, Laravel 10+
**Dependencies**: `bongo/framework`, `bongo/user`, `bongo/image`

## Key Classes and Relationships

### Service Provider
```php
Bongo\Profile\ProfileServiceProvider extends Bongo\Framework\Providers\AbstractServiceProvider
```
- Sets `$module = 'profile'` for automatic bootstrapping
- Inherits route, view, translation, and migration registration

### Controllers

#### ProfileController
```php
Bongo\Profile\Http\Controllers\Backend\ProfileController extends Bongo\Framework\Http\Controllers\AbstractController
```

**Methods**:
- `index()` - Display user profile view
- `edit()` - Display profile edit form
- `update(Request $request)` - Update user profile and password

**Key Features**:
- Uses `user()` helper to get authenticated user
- Validates email uniqueness (excluding current user and soft deleted)
- Hashes passwords with `Hash::make()`
- Enforces strong password requirements via regex
- Returns redirect with success flash message

#### ProfileImageController
```php
Bongo\Profile\Http\Controllers\Backend\ProfileImageController extends Bongo\Framework\Http\Controllers\AbstractController
```

**Methods**:
- `upload(StoreImageRequest $request): JsonResponse` - Handle avatar upload

**Key Features**:
- Uses `Bongo\Image\Services\AvatarImage` for image processing
- Returns JSON response with avatar URL
- Handles exceptions with 500 status code

### Routes
All routes registered in `src/Routes/backend.php`:

```php
// Prefix: /admin/profile
// Name: backend.profile.*
// Middleware: ['web', 'auth', 'employee']

GET  /admin/profile          → ProfileController@index (backend.profile.index)
GET  /admin/profile/edit     → ProfileController@edit (backend.profile.edit)
POST /admin/profile/update   → ProfileController@update (backend.profile.update)
POST /admin/profile/image    → ProfileImageController@upload (backend.profile.image)
```

### Views
Located in `src/Views/backend/`:

- `index.blade.php` - Profile view page with avatar, contact info, and tabs
- `edit.blade.php` - Profile edit form wrapper
- `partials/form/details.blade.php` - Edit form fields (name, contact, password, avatar)
- `partials/tab_detail.blade.php` - View tab displaying user details

### Translations
Located in `src/Translations/en/backend.php`:

Keys: `index`, `edit`, `update`, `update_success`, `update_failed`, `show`, `delete`, `delete_success`, `delete_failed`

## Code Style Templates

### Controller Method Template
```php
public function methodName(Request $request)
{
    // Get current user
    $user = user();

    // Validate request
    $request->validate([
        'field' => 'required|max:255',
    ]);

    // Perform operation
    $user->update($request->only(['field']));

    // Redirect with message
    return redirect()
        ->route('backend.profile.index')
        ->success(trans('profile::backend.update_success'));
}
```

### View Template (Index/View)
```php
@extends('framework::backend.layouts.app')

@section('content')
    @component('blade::content_header')
        {{ $user->email }}
        @slot('actions')
            @backButton(['name' => 'Back', 'url' => route('backend.dashboard.index')])
            @editButton(['name' => 'Edit', 'url' => route('backend.profile.edit')])
        @endslot
    @endcomponent

    @component('blade::content_body')
        <!-- Content here -->
    @endcomponent
@stop
```

### View Template (Edit/Form)
```php
@extends('framework::backend.layouts.app')

@section('content')
    <form method="POST" action="{{ route('backend.profile.update') }}">
        {{ csrf_field() }}

        @component('blade::content_header')
            {{ trans('profile::backend.edit') }}
            @slot('actions')
                @backButton(['name' => trans('profile::backend.index'), 'url' => route('backend.profile.index')])
                @saveButton(['name' => trans('profile::backend.update')])
            @endslot
        @endcomponent

        @component('blade::content_body')
            @include('profile::backend.partials.form.details')
        @endcomponent
    </form>
@stop
```

### Form Field Templates
```php
<!-- Text Input -->
@input(['name' => 'field_name', 'value' => $user->field_name, 'required' => 'true'])

<!-- Static Display -->
@static(['name' => 'Field Name', 'value' => $user->field_name])

<!-- Password Input -->
@input(['label' => 'New Password', 'type' => 'password', 'name' => 'password', 'value' => null])

<!-- File Upload Component -->
<file-uploader
    url="{{ route('backend.profile.image') }}"
    name="user_avatar"
    height="100"
></file-uploader>
```

### Validation Template
```php
$request->validate([
    'first_name' => 'nullable|max:70',
    'last_name' => 'nullable|max:70',
    'email' => "required|string|email:rfc,dns|unique:{$userTable},email,".user()->id.',id,deleted_at,NULL|max:50',
    'password' => 'sometimes|nullable|string|min:8|confirmed|regex:/^(?=.*?[A-Z])(?=.*?[a-z])(?=.*?[0-9])(?=.*?[#?!@$%^&*-]).{6,}$/',
]);
```

### Image Upload Template
```php
public function upload(StoreImageRequest $request): JsonResponse
{
    $files = request()->file('files');

    try {
        $user = (new AvatarImage(user(), $files[0]))->save();
        return response()->json([$user->getAvatar()]);
    } catch (Exception $e) {
        return response()->json(['error' => $e->getMessage()], 500);
    }
}
```

## Common Patterns

### Accessing Current User
```php
// Get authenticated user
$user = user();

// Access properties
$user->first_name
$user->last_name
$user->email
$user->getAvatar()
```

### Password Update Pattern
```php
// Only update password if provided
if ($request->has('password') && !empty($request->get('password'))) {
    user()->password = Hash::make($request->get('password'));
}
user()->save();
```

### Email Validation with Uniqueness
```php
$userTable = (new User())->getTable();

$request->validate([
    'email' => "required|string|email:rfc,dns|unique:{$userTable},email,".user()->id.',id,deleted_at,NULL|max:50',
]);
```

### Flash Message Pattern
```php
return redirect()
    ->route('backend.profile.index')
    ->success(trans('profile::backend.update_success'));
```

### Translation Pattern
```php
trans('profile::backend.edit')          // Edit
trans('profile::backend.update_success') // Profile successfully updated!
```

### Blade Component Patterns
```php
// Content Header with Actions
@component('blade::content_header')
    {{ $title }}
    @slot('actions')
        @backButton(['name' => 'Back'])
        @editButton(['name' => 'Edit', 'url' => $url])
    @endslot
@endcomponent

// Card Component
@component('blade::card', ['cardClass' => 'mb-4'])
    <!-- Card content -->
@endcomponent

// Tab Component
<tabs class-list="border-t border-gray-100">
    <tab name="Details" :selected="true">
        @include('profile::backend.partials.tab_detail')
    </tab>
</tabs>
```

### Route Definition Pattern
```php
Route::prefix('profile')
    ->as('profile.')
    ->group(function () {
        Route::get('/', [ProfileController::class, 'index'])->name('index');
        Route::get('edit', [ProfileController::class, 'edit'])->name('edit');
        Route::post('update', [ProfileController::class, 'update'])->name('update');
        Route::post('image', [ProfileImageController::class, 'upload'])->name('image');
    });
```

## Testing Patterns

### Test Case Base
```php
namespace Bongo\Profile\Tests;

class YourTest extends TestCase
{
    public function test_feature()
    {
        // Your test implementation
    }
}
```

### Orchestra Testbench Usage
The `TestCase` class:
- Extends `Orchestra\Testbench\TestCase`
- Registers `ProfileServiceProvider` automatically
- Uses test database connection

## Key Conventions

1. **Controllers** extend `AbstractController` from bongo/framework
2. **Service Provider** sets `protected string $module = 'profile'`
3. **Routes** use explicit controller action syntax: `[Controller::class, 'method']`
4. **Views** use `profile::` namespace prefix
5. **Translations** use `profile::backend.` prefix
6. **Form Fields** use Blade directives from bongo/blade package
7. **CSRF Protection** on all POST forms via `{{ csrf_field() }}`
8. **Type Hints** on controller method parameters and return types
9. **Password Hashing** always use `Hash::make()`, never plain text
10. **User Helper** use `user()` helper instead of `Auth::user()`

## Dependencies and Relationships

```
bongo/profile
├── depends on: bongo/framework (AbstractServiceProvider, AbstractController)
├── depends on: bongo/user (User model)
├── depends on: bongo/image (AvatarImage service, StoreImageRequest)
└── used by: Laravel applications via service provider registration
```

## Security Considerations

- All routes protected by `auth` and `employee` middleware
- CSRF tokens on all forms
- Passwords hashed with bcrypt via `Hash::make()`
- Email validation includes DNS checking
- Strong password requirements enforced
- Email uniqueness checked (excluding soft deleted and current user)
- Exception handling in image uploads returns safe JSON errors

## Extension Points

To extend functionality:

1. **Add fields**: Update controller validation, views, and translations
2. **Change password rules**: Modify regex in `ProfileController::update()`
3. **Customise avatar processing**: Extend `AvatarImage` service or override upload method
4. **Add new routes**: Add to `src/Routes/backend.php` following naming pattern
5. **Add tabs**: Add new `<tab>` component in `index.blade.php` with corresponding partial
6. **Customise validation**: Override validation rules in `update()` method
7. **Add middleware**: Define in service provider `$middlewares` property
