# Architecture Documentation - Bongo Profile Package

## Overview

The Bongo Profile package provides user profile management functionality for the admin area of Laravel applications. It follows Laravel package development best practices and integrates with the Bongo framework architecture.

**Purpose**: Enable authenticated employees to view and update their personal profile information, including contact details, password, and avatar image.

**Version**: 3.0+ (PHP 8.2+, Laravel 10+)

## Directory Structure

```
default/profile/
├── .github/
│   └── copilot-instructions.md         # GitHub Copilot AI instructions
├── src/
│   ├── Http/
│   │   └── Controllers/
│   │       └── Backend/                # Admin area controllers
│   │           ├── ProfileController.php           # Main profile operations (index, edit, update)
│   │           └── ProfileImageController.php      # Avatar image upload handler
│   ├── Routes/
│   │   └── backend.php                 # Backend routes (auto-registered with auth + employee middleware)
│   ├── Translations/
│   │   └── en/
│   │       └── backend.php             # English translations for admin area
│   ├── Views/
│   │   └── backend/
│   │       ├── index.blade.php         # Profile view page (display mode)
│   │       ├── edit.blade.php          # Profile edit page (form wrapper)
│   │       └── partials/
│   │           ├── form/
│   │           │   └── details.blade.php       # Edit form fields
│   │           └── tab_detail.blade.php        # View tab content
│   └── ProfileServiceProvider.php      # Service provider (extends AbstractServiceProvider)
├── tests/
│   └── TestCase.php                    # Base test case (Orchestra Testbench)
├── .cursorrules                        # Cursor AI instructions
├── .editorconfig                       # Editor configuration
├── .gitignore                          # Git ignore rules
├── .php_cs                             # PHP-CS-Fixer configuration (legacy)
├── .styleci.yml                        # StyleCI configuration
├── composer.json                       # Package dependencies and metadata
├── phpunit.xml                         # PHPUnit configuration
├── LICENSE                             # MIT License
└── README.md                           # Package documentation
```

### Key Directory Purposes

- **src/Http/Controllers/Backend/**: Admin-only controllers requiring authentication and employee role
- **src/Routes/**: Route definitions auto-registered by AbstractServiceProvider
- **src/Translations/**: Localisation files (namespace: `profile::`)
- **src/Views/**: Blade templates (namespace: `profile::`)
- **tests/**: PHPUnit tests using Orchestra Testbench

## System Architecture

### High-Level Component Diagram

```
┌─────────────────────────────────────────────────────────────────┐
│                      Laravel Application                         │
│  ┌────────────────────────────────────────────────────────────┐ │
│  │           ProfileServiceProvider Registration              │ │
│  │  (Auto-registers routes, views, translations via boot())   │ │
│  └────────────────────────────────────────────────────────────┘ │
└─────────────────────────────────────────────────────────────────┘
                               │
                               ▼
┌─────────────────────────────────────────────────────────────────┐
│                   Middleware Stack (Backend)                     │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────────────────┐  │
│  │     web      │→ │     auth     │→ │      employee        │  │
│  │  (sessions,  │  │ (requires    │  │  (requires staff     │  │
│  │   CSRF)      │  │  login)      │  │   role)              │  │
│  └──────────────┘  └──────────────┘  └──────────────────────┘  │
└─────────────────────────────────────────────────────────────────┘
                               │
                               ▼
┌─────────────────────────────────────────────────────────────────┐
│                         Route Layer                              │
│  /admin/profile         → ProfileController@index   (GET)        │
│  /admin/profile/edit    → ProfileController@edit    (GET)        │
│  /admin/profile/update  → ProfileController@update  (POST)       │
│  /admin/profile/image   → ProfileImageController@upload (POST)   │
└─────────────────────────────────────────────────────────────────┘
                               │
                               ▼
┌─────────────────────────────────────────────────────────────────┐
│                      Controller Layer                            │
│  ┌───────────────────────┐  ┌────────────────────────────────┐  │
│  │  ProfileController    │  │  ProfileImageController        │  │
│  │  - index()            │  │  - upload()                    │  │
│  │  - edit()             │  │    (JSON response)             │  │
│  │  - update()           │  └────────────────────────────────┘  │
│  │    (with validation)  │                                      │
│  └───────────────────────┘                                      │
└─────────────────────────────────────────────────────────────────┘
                               │
                               ▼
┌─────────────────────────────────────────────────────────────────┐
│                       Service Layer                              │
│  ┌───────────────────────────────────────────────────────────┐  │
│  │  Bongo\Image\Services\AvatarImage                         │  │
│  │  - Processes uploaded avatar images                       │  │
│  │  - Saves to user model                                    │  │
│  │  - Returns updated user with avatar URL                   │  │
│  └───────────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────────┘
                               │
                               ▼
┌─────────────────────────────────────────────────────────────────┐
│                         Model Layer                              │
│  ┌───────────────────────────────────────────────────────────┐  │
│  │  Bongo\User\Models\User (from bongo/user)                 │  │
│  │  - Eloquent model for users table                         │  │
│  │  - Properties: first_name, last_name, email, etc.         │  │
│  │  - Methods: getAvatar(), fill(), save()                   │  │
│  └───────────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────────┘
                               │
                               ▼
┌─────────────────────────────────────────────────────────────────┐
│                         Database Layer                           │
│  ┌───────────────────────────────────────────────────────────┐  │
│  │  users table                                               │  │
│  │  - id, first_name, last_name, email, password              │  │
│  │  - telephone, mobile, title, status                        │  │
│  │  - avatar (file path)                                      │  │
│  │  - deleted_at (soft deletes)                               │  │
│  └───────────────────────────────────────────────────────────┘  │
└─────────────────────────────────────────────────────────────────┘
```

## Request Flow Diagrams

### Profile View Request Flow

```
User (authenticated employee)
    │
    │ GET /admin/profile
    ▼
ProfileController@index()
    │
    │ 1. Get authenticated user via user() helper
    │ 2. Load view with user data
    ▼
View: profile::backend.index
    │
    │ Displays:
    │ - User avatar
    │ - Name, status
    │ - Contact info (telephone, mobile, email)
    │ - Details tab with read-only fields
    │
    │ Actions:
    │ - Back to dashboard
    │ - Edit profile button
    ▼
Rendered HTML → User
```

### Profile Update Request Flow

```
User (authenticated employee)
    │
    │ 1. GET /admin/profile/edit
    ▼
ProfileController@edit()
    │
    │ Load edit form with user data
    ▼
View: profile::backend.edit
    │
    │ Form fields:
    │ - first_name, last_name
    │ - telephone, mobile, email
    │ - password, password_confirmation
    │ - avatar upload component
    │
    │ 2. User submits form
    │ POST /admin/profile/update
    ▼
ProfileController@update(Request $request)
    │
    │ 3. Validate request
    │    ├─ first_name: nullable|max:70
    │    ├─ last_name: nullable|max:70
    │    ├─ email: required|unique (DNS validation)
    │    └─ password: min:8|confirmed|regex (strong password)
    │
    │ 4. Update user model
    │    ├─ Fill with request data (except password)
    │    └─ Hash password if provided
    │
    │ 5. Save user
    │
    │ 6. Redirect with flash message
    ▼
Redirect to /admin/profile
    │
    │ Flash: "Profile successfully updated!"
    ▼
ProfileController@index()
```

### Avatar Upload Request Flow

```
User (authenticated employee)
    │
    │ 1. On edit page, selects image file
    │ 2. Vue component triggers upload
    │ POST /admin/profile/image (AJAX)
    │ Content-Type: multipart/form-data
    ▼
ProfileImageController@upload(StoreImageRequest $request)
    │
    │ 3. Validate image via StoreImageRequest
    │    (from bongo/image package)
    │
    │ 4. Get uploaded file
    │    $files = request()->file('files');
    │
    │ 5. Process image via AvatarImage service
    │    try {
    │        $user = (new AvatarImage(user(), $files[0]))->save();
    │    } catch (Exception $e) {
    │        return JSON error
    │    }
    │
    │ 6. Return JSON response
    │    return response()->json([$user->getAvatar()]);
    ▼
JSON Response → Vue Component
    │
    │ { "avatar_url": "/storage/avatars/user-123.jpg" }
    │
    │ 7. Component updates preview
    ▼
User sees new avatar
```

## Class Structure and Relationships

### Class Diagram

```
┌────────────────────────────────────────────────────────────────┐
│  Bongo\Framework\Providers\AbstractServiceProvider            │
│  (from bongo/framework)                                        │
│  ──────────────────────────────────────────────────────────   │
│  # $module: string                                             │
│  # $middlewares: array                                         │
│  # $commands: array                                            │
│  # $listeners: array                                           │
│  # $subscribers: array                                         │
│  # $composers: array                                           │
│  ──────────────────────────────────────────────────────────   │
│  + boot(): void                                                │
│  # registerRoutes(): void                                      │
│  # registerViews(): void                                       │
│  # registerTranslations(): void                                │
│  # registerMigrations(): void                                  │
└────────────────────────────────────────────────────────────────┘
                               △
                               │ extends
                               │
┌────────────────────────────────────────────────────────────────┐
│  Bongo\Profile\ProfileServiceProvider                         │
│  ──────────────────────────────────────────────────────────   │
│  # $module: string = 'profile'                                 │
│  ──────────────────────────────────────────────────────────   │
│  (Inherits all auto-registration from parent)                  │
└────────────────────────────────────────────────────────────────┘


┌────────────────────────────────────────────────────────────────┐
│  Bongo\Framework\Http\Controllers\AbstractController          │
│  (from bongo/framework)                                        │
│  ──────────────────────────────────────────────────────────   │
│  (Base controller functionality)                               │
└────────────────────────────────────────────────────────────────┘
                               △
                               │ extends
                               │
                ┌──────────────┴──────────────┐
                │                             │
┌───────────────────────────────┐ ┌───────────────────────────────┐
│  ProfileController            │ │  ProfileImageController       │
│  ─────────────────────────── │ │  ───────────────────────────  │
│  + index()                    │ │  + upload(request): JsonResp  │
│  + edit()                     │ └───────────────────────────────┘
│  + update(request)            │              │
└───────────────────────────────┘              │ uses
                │                              ▼
                │ uses             ┌───────────────────────────────┐
                ▼                  │  AvatarImage                  │
┌───────────────────────────────┐  │  (from bongo/image)           │
│  User                         │  │  ─────────────────────────── │
│  (from bongo/user)            │  │  + __construct(user, file)   │
│  ─────────────────────────── │  │  + save(): User              │
│  + first_name: string         │  └───────────────────────────────┘
│  + last_name: string          │
│  + email: string              │
│  + password: string           │
│  + telephone: string          │
│  + mobile: string             │
│  + title: string              │
│  + status: string             │
│  + getAvatar(): string        │
│  + fill(array): self          │
│  + save(): bool               │
└───────────────────────────────┘
```

### Controller Methods Detail

#### ProfileController

| Method | Parameters | Return Type | Purpose |
|--------|-----------|-------------|---------|
| `index()` | none | `View` | Display profile view page with user details |
| `edit()` | none | `View` | Display profile edit form |
| `update()` | `Request $request` | `RedirectResponse` | Validate and update user profile, handle password changes |

**Key Logic in `update()`**:
1. Get user table name dynamically: `(new User())->getTable()`
2. Validate all fields including unique email check (excluding current user and soft deleted)
3. Update user fields via `fill()` except password
4. Hash and update password only if provided and not empty
5. Save user model
6. Redirect to index with success flash message

#### ProfileImageController

| Method | Parameters | Return Type | Purpose |
|--------|-----------|-------------|---------|
| `upload()` | `StoreImageRequest $request` | `JsonResponse` | Handle avatar image upload via AJAX |

**Key Logic**:
1. Extract uploaded file from request: `request()->file('files')[0]`
2. Create `AvatarImage` service instance with user and file
3. Call `save()` to process and store image
4. Return JSON array with avatar URL: `[$user->getAvatar()]`
5. Catch exceptions and return 500 JSON error response

## Data Flow and Validation

### Validation Rules

#### Profile Update Validation

```php
[
    'first_name' => 'nullable|max:70',
    'last_name' => 'nullable|max:70',
    'telephone' => 'nullable',
    'mobile' => 'nullable',
    'email' => 'required|string|email:rfc,dns|unique:{$userTable},email,{user_id},id,deleted_at,NULL|max:50',
    'password' => 'sometimes|nullable|string|min:8|confirmed|regex:/^(?=.*?[A-Z])(?=.*?[a-z])(?=.*?[0-9])(?=.*?[#?!@$%^&*-]).{6,}$/',
]
```

**Email Validation Breakdown**:
- `required` - Must be provided
- `string` - Must be string type
- `email:rfc,dns` - Must be valid RFC email format with DNS validation
- `unique:{table},email,{id},id,deleted_at,NULL` - Must be unique excluding current user and soft deleted records
- `max:50` - Maximum 50 characters

**Password Validation Breakdown**:
- `sometimes` - Only validate if field is present
- `nullable` - Can be null/empty
- `string` - Must be string type
- `min:8` - Minimum 8 characters
- `confirmed` - Must match `password_confirmation` field
- Regex requires:
  - At least 1 uppercase letter `(?=.*?[A-Z])`
  - At least 1 lowercase letter `(?=.*?[a-z])`
  - At least 1 digit `(?=.*?[0-9])`
  - At least 1 special character `(?=.*?[#?!@$%^&*-])`
  - Minimum 6 characters `.{6,}` (overridden by `min:8`)

#### Avatar Upload Validation

Handled by `StoreImageRequest` from `bongo/image` package:
- File type validation (likely: jpg, png, gif)
- File size validation
- Image dimension validation
- MIME type validation

### Password Handling

```php
// Only update password if provided and not empty
if ($request->has('password') && !empty($request->get('password'))) {
    user()->password = Hash::make($request->get('password'));
}
user()->save();
```

**Security Notes**:
- Uses Laravel's `Hash` facade (bcrypt)
- Never stores plain text passwords
- Only updates if explicitly provided
- Requires confirmation field to prevent typos

## View Architecture

### View Hierarchy

```
framework::backend.layouts.app (main layout from bongo/framework)
    │
    ├─ profile::backend.index (view mode)
    │   ├─ blade::content_header (with actions: back, edit)
    │   └─ blade::content_body
    │       ├─ User info card (avatar, name, contacts)
    │       └─ Tabs component
    │           └─ Details tab
    │               └─ profile::backend.partials.tab_detail
    │
    └─ profile::backend.edit (edit mode)
        ├─ blade::content_header (with actions: back, save)
        └─ blade::content_body
            └─ profile::backend.partials.form.details
                ├─ Name fields (first_name, last_name)
                ├─ Contact fields (telephone, mobile, email)
                ├─ Avatar uploader (file-uploader component)
                └─ Password fields (password, password_confirmation)
```

### Blade Components Used

| Component | Source Package | Purpose |
|-----------|---------------|---------|
| `@component('blade::content_header')` | bongo/blade | Page header with title and action buttons |
| `@component('blade::content_body')` | bongo/blade | Main content wrapper |
| `@component('blade::card')` | bongo/blade | Card container with styling |
| `@backButton()` | bongo/blade | Back navigation button directive |
| `@editButton()` | bongo/blade | Edit action button directive |
| `@saveButton()` | bongo/blade | Save/submit button directive |
| `@input()` | bongo/blade | Form input field directive |
| `@static()` | bongo/blade | Static field display directive |
| `@label()` | bongo/blade | Form label directive |
| `<file-uploader>` | bongo/blade | Vue.js file upload component |
| `<tabs>`, `<tab>` | bongo/blade | Tab navigation components |

### Translation Keys

**Namespace**: `profile::backend`

| Key | English Value | Usage |
|-----|--------------|-------|
| `index` | Profile | Page title |
| `show` | View | View action label |
| `edit` | Edit | Edit action label |
| `update` | Update | Update action label |
| `update_success` | Profile successfully updated! | Success flash message |
| `update_failed` | Unable to update profile! | Error flash message |
| `delete` | Delete | Delete action label (unused) |
| `delete_success` | Profile successfully deleted! | Success flash (unused) |
| `delete_failed` | Unable to delete profile! | Error flash (unused) |

**Usage**:
```php
trans('profile::backend.update_success')
```

## Routes Architecture

### Route Registration

Routes are automatically registered via `AbstractServiceProvider::boot()`:

**File**: `src/Routes/backend.php`
**Auto-applied Middleware**: `['web', 'auth', 'employee']`
**Name Prefix**: `backend.`
**Route Prefix**: `/admin/profile`

### Route Table

| Method | URI | Name | Controller | Action |
|--------|-----|------|-----------|--------|
| GET | `/admin/profile` | `backend.profile.index` | ProfileController | Display profile view |
| GET | `/admin/profile/edit` | `backend.profile.edit` | ProfileController | Display edit form |
| POST | `/admin/profile/update` | `backend.profile.update` | ProfileController | Update profile |
| POST | `/admin/profile/image` | `backend.profile.image` | ProfileImageController | Upload avatar (AJAX) |

### Route Definition Pattern

```php
Route::prefix('profile')
    ->as('profile.')
    ->group(function () {
        Route::get('/', [ProfileController::class, 'index'])->name('index');
        Route::get('edit', [ProfileController::class, 'edit'])->name('edit');
        Route::post('update', [ProfileController::class, 'update'])->name('update');
        Route::post('image', [ProfileImageController::class, 'upload'])->name('image');
    });
```

**Result**: All routes automatically receive:
- `backend.` name prefix → `backend.profile.index`
- `/admin/` URI prefix → `/admin/profile`
- `['web', 'auth', 'employee']` middleware stack

## Dependencies

### Required Packages

```json
{
  "php": ">=8.2",
  "illuminate/contracts": "^10.0",
  "bongo/framework": "^3.0"
}
```

### Development Dependencies

```json
{
  "laravel/pint": "^1.0",              // Code style fixer
  "nunomaduro/collision": "^7.9",      // Error handler
  "nunomaduro/larastan": "^2.0.1",     // Static analysis
  "orchestra/testbench": "^8.0",       // Package testing
  "phpunit/phpunit": "^10.0"           // Unit testing
}
```

### External Class Dependencies

| Class | Package | Purpose |
|-------|---------|---------|
| `Bongo\Framework\Providers\AbstractServiceProvider` | bongo/framework | Base service provider with auto-registration |
| `Bongo\Framework\Http\Controllers\AbstractController` | bongo/framework | Base controller class |
| `Bongo\User\Models\User` | bongo/user | User model with profile data |
| `Bongo\Image\Services\AvatarImage` | bongo/image | Avatar image processing |
| `Bongo\Image\Http\Requests\StoreImageRequest` | bongo/image | Image upload validation |

### Helper Functions Used

| Function | Purpose | Defined In |
|----------|---------|-----------|
| `user()` | Get authenticated user instance | bongo/framework (likely) |
| `trans()` | Translate string | Laravel core |
| `view()` | Create view instance | Laravel core |
| `route()` | Generate route URL | Laravel core |
| `request()` | Get request instance | Laravel core |
| `asset()` | Generate asset URL | Laravel core |

## Extension Points

### Adding New Fields

**1. Add Database Column** (via migration in parent project):
```php
Schema::table('users', function (Blueprint $table) {
    $table->string('new_field')->nullable();
});
```

**2. Add to Edit Form** (`src/Views/backend/partials/form/details.blade.php`):
```php
@input(['name' => 'new_field', 'value' => $user->new_field])
```

**3. Add to View Tab** (`src/Views/backend/partials/tab_detail.blade.php`):
```php
@static(['name' => 'New Field', 'value' => $user->new_field])
```

**4. Add Validation** (`ProfileController::update()`):
```php
$request->validate([
    'new_field' => 'nullable|max:100',
    // ... existing rules
]);
```

**5. Add Translation** (`src/Translations/en/backend.php`):
```php
'new_field' => 'New Field',
```

### Adding New Routes

Add to `src/Routes/backend.php`:
```php
Route::get('custom-action', [ProfileController::class, 'customAction'])
    ->name('custom-action');
```

Result: Route automatically becomes `backend.profile.custom-action` at `/admin/profile/custom-action`

### Adding Middleware

Define in service provider:
```php
class ProfileServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'profile';

    protected array $middlewares = [
        'customMiddleware' => CustomMiddleware::class,
    ];
}
```

Then use in routes:
```php
Route::get('/', [ProfileController::class, 'index'])
    ->middleware('customMiddleware')
    ->name('index');
```

### Adding New Tabs

In `src/Views/backend/index.blade.php`:
```php
<tabs class-list="border-t border-gray-100">
    <tab name="Details" :selected="true">
        @include('profile::backend.partials.tab_detail')
    </tab>
    <tab name="Custom Tab">
        @include('profile::backend.partials.tab_custom')
    </tab>
</tabs>
```

Create partial: `src/Views/backend/partials/tab_custom.blade.php`

### Customising Avatar Processing

Override or extend `ProfileImageController::upload()`:
```php
public function upload(StoreImageRequest $request): JsonResponse
{
    $files = request()->file('files');

    try {
        $user = (new AvatarImage(user(), $files[0]))
            ->setWidth(250)
            ->setHeight(250)
            ->save();

        return response()->json([$user->getAvatar()]);
    } catch (Exception $e) {
        return response()->json(['error' => $e->getMessage()], 500);
    }
}
```

### Changing Password Requirements

Modify regex in `ProfileController::update()`:
```php
'password' => 'sometimes|nullable|string|min:12|confirmed|regex:/your-custom-regex/'
```

## Testing Architecture

### Test Structure

```
tests/
└── TestCase.php    # Base test case
```

### Base Test Case

```php
namespace Bongo\Profile\Tests;

use Bongo\Profile\ProfileServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    public function setUp(): void
    {
        parent::setUp();
        // Setup logic
    }

    protected function getPackageProviders($app): array
    {
        return [ProfileServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app)
    {
        // Environment configuration
        // e.g., database setup, config overrides
    }
}
```

### Writing Tests

**Example Test Structure**:
```php
namespace Bongo\Profile\Tests\Feature;

use Bongo\Profile\Tests\TestCase;
use Bongo\User\Models\User;

class ProfileControllerTest extends TestCase
{
    public function test_authenticated_user_can_view_profile()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user)
            ->get(route('backend.profile.index'));

        $response->assertOk();
        $response->assertViewIs('profile::backend.index');
        $response->assertViewHas('user', $user);
    }

    public function test_user_can_update_profile()
    {
        $user = User::factory()->create();

        $response = $this->actingAs($user)
            ->post(route('backend.profile.update'), [
                'first_name' => 'John',
                'last_name' => 'Doe',
                'email' => 'john@example.com',
            ]);

        $response->assertRedirect(route('backend.profile.index'));
        $this->assertEquals('John', $user->fresh()->first_name);
    }
}
```

### Running Tests

```bash
# Run all tests
vendor/bin/phpunit

# Run with coverage
vendor/bin/phpunit --coverage-html coverage

# Run specific test
vendor/bin/phpunit --filter test_user_can_view_profile
```

## Security Architecture

### Authentication & Authorization

```
Request
    │
    ├─ web middleware
    │   ├─ Session management
    │   ├─ CSRF protection
    │   └─ Cookie encryption
    │
    ├─ auth middleware
    │   └─ Requires authenticated user
    │
    └─ employee middleware
        └─ Requires employee/staff role
```

### CSRF Protection

All POST forms include:
```php
{{ csrf_field() }}
```

Generates hidden input:
```html
<input type="hidden" name="_token" value="...">
```

### Password Security

1. **Hashing**: Uses bcrypt via `Hash::make()`
2. **Validation**: Requires strong passwords (uppercase, lowercase, number, special char)
3. **Confirmation**: Requires `password_confirmation` field match
4. **Optional Update**: Only updates if explicitly provided

### Email Validation

1. **Format**: RFC-compliant email format
2. **DNS**: Validates DNS records exist for domain
3. **Uniqueness**: Prevents duplicate emails (excluding soft deleted)
4. **Current User Exclusion**: Allows user to keep their own email

### Input Validation

- All inputs validated before processing
- Max length constraints prevent overflow
- Type validation ensures correct data types
- Nullable fields allow optional data

### Error Handling

```php
try {
    $user = (new AvatarImage(user(), $files[0]))->save();
    return response()->json([$user->getAvatar()]);
} catch (Exception $e) {
    return response()->json(['error' => $e->getMessage()], 500);
}
```

- Exceptions caught and returned as JSON errors
- HTTP 500 status code for server errors
- Error messages provided for debugging

## Performance Considerations

### Database Queries

- Single user lookup via `user()` helper (likely cached in session)
- No N+1 query issues (no relationships loaded)
- Soft delete check in email uniqueness query

### View Rendering

- Blade template compilation cached in production
- Minimal database queries (only current user)
- No complex computations in views

### Image Processing

- Handled by `AvatarImage` service (from bongo/image)
- Likely includes resizing and optimisation
- Saved to storage (filesystem or cloud)

### Validation

- Validation performed before any database operations
- Early return on validation failure
- DNS validation for email (may add latency)

## Common Patterns and Idioms

### Getting Current User
```php
$user = user();
```

### Loading Views with Data
```php
return view('profile::backend.index', ['user' => user()]);
```

### Redirecting with Flash Messages
```php
return redirect()
    ->route('backend.profile.index')
    ->success(trans('profile::backend.update_success'));
```

### Conditional Password Update
```php
if ($request->has('password') && !empty($request->get('password'))) {
    user()->password = Hash::make($request->get('password'));
}
```

### Dynamic Table Name for Validation
```php
$userTable = (new User())->getTable();
$request->validate([
    'email' => "required|unique:{$userTable},email,{$userId}",
]);
```

### JSON Error Responses
```php
return response()->json(['error' => $e->getMessage()], 500);
```

### Form Request Validation
```php
public function upload(StoreImageRequest $request): JsonResponse
{
    // Request automatically validated before reaching here
}
```

## Integration with Bongo Framework

### AbstractServiceProvider Integration

The package leverages AbstractServiceProvider's automatic bootstrapping:

1. **Config Registration**: None needed (no config file)
2. **Route Registration**: Loads `src/Routes/backend.php` with middleware
3. **View Registration**: Registers `src/Views/` with `profile::` namespace
4. **Translation Registration**: Registers `src/Translations/` with `profile::` namespace
5. **Migration Registration**: None (uses User model from bongo/user)

### AbstractController Integration

Controllers extend AbstractController which provides:
- Base controller functionality
- Response helpers
- Middleware assignment
- Common controller methods

### Framework Dependencies

```
bongo/profile
    ├─ uses: bongo/framework (core)
    ├─ uses: bongo/user (User model)
    ├─ uses: bongo/image (avatar processing)
    └─ uses: bongo/blade (view components)
```

## Future Enhancement Possibilities

1. **Multi-factor Authentication**: Add 2FA setup to profile
2. **Activity Log**: Track profile changes with audit trail
3. **Preferences**: Add user preference settings
4. **Social Links**: Add social media profile links
5. **API Access**: Add RESTful API endpoints
6. **Notification Settings**: Add email/SMS notification preferences
7. **Privacy Settings**: Add privacy control options
8. **Account Deletion**: Add self-service account deletion
9. **Export Data**: Add GDPR-compliant data export
10. **Profile Visibility**: Add public/private profile toggle
