# Bongo Profile

A Laravel package that provides user profile management functionality for the admin area. Allows authenticated employees to view and update their personal information including contact details, password, and avatar image.

## Features

- **Profile Viewing**: Display user profile with avatar, contact information, and personal details
- **Profile Editing**: Update name, contact information, email, and password
- **Avatar Upload**: Upload and manage user avatar images via AJAX
- **Strong Password Validation**: Enforces password complexity requirements
- **Email Validation**: Validates email format with DNS checking and uniqueness
- **Admin-Only Access**: Protected by authentication and employee middleware
- **Flash Messages**: User-friendly success/error notifications
- **Responsive Design**: Mobile-friendly interface using Tailwind CSS

## Requirements

- PHP 8.2+
- Laravel 10+
- [bongo/framework](https://designtecpackages.co.uk) ^3.0
- [bongo/user](https://designtecpackages.co.uk) package (provides User model)
- [bongo/image](https://designtecpackages.co.uk) package (provides avatar processing)

## Installation

### 1. Install via Composer

```bash
composer require bongo/profile
```

### 2. Service Provider Registration

Laravel 11+ will auto-discover the service provider. For Laravel 10 or manual registration, add to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Profile\ProfileServiceProvider::class,
],
```

### 3. Ensure Dependencies

Make sure you have the required Bongo packages installed:

```bash
composer require bongo/framework bongo/user bongo/image
```

### 4. Configure Authentication

Ensure your application has:
- User authentication configured
- `employee` middleware registered (for staff-only access)
- `user()` helper function available (typically from bongo/framework)

## Usage

### Routes

Once installed, the following routes are automatically registered:

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/admin/profile` | `backend.profile.index` | View profile page |
| GET | `/admin/profile/edit` | `backend.profile.edit` | Edit profile form |
| POST | `/admin/profile/update` | `backend.profile.update` | Update profile |
| POST | `/admin/profile/image` | `backend.profile.image` | Upload avatar (AJAX) |

All routes require authentication and the `employee` middleware.

### Accessing Profile Pages

**View Profile**:
```php
<a href="{{ route('backend.profile.index') }}">My Profile</a>
```

**Edit Profile**:
```php
<a href="{{ route('backend.profile.edit') }}">Edit Profile</a>
```

### Password Requirements

Passwords must meet the following criteria:
- Minimum 8 characters
- At least 1 uppercase letter (A-Z)
- At least 1 lowercase letter (a-z)
- At least 1 number (0-9)
- At least 1 special character (#?!@$%^&*-)

### Avatar Upload

The avatar upload component uses Vue.js and AJAX:

```blade
<file-uploader
    url="{{ route('backend.profile.image') }}"
    name="user_avatar"
    height="100"
></file-uploader>
```

Uploaded images are processed by the `Bongo\Image\Services\AvatarImage` service.

## User Model Requirements

Your User model (from `bongo/user`) must have:

**Properties**:
- `first_name` (string, nullable)
- `last_name` (string, nullable)
- `email` (string, required, unique)
- `password` (string, hashed)
- `telephone` (string, nullable)
- `mobile` (string, nullable)
- `title` (string, nullable)
- `status` (string)

**Methods**:
- `getAvatar()` - Returns avatar image path/URL

## Customization

### Validation Rules

To customize validation, extend `ProfileController` and override the `update()` method:

```php
namespace App\Http\Controllers\Backend;

use Bongo\Profile\Http\Controllers\Backend\ProfileController as BaseProfileController;
use Illuminate\Http\Request;

class ProfileController extends BaseProfileController
{
    public function update(Request $request)
    {
        $request->validate([
            'first_name' => 'required|max:70',  // Make required
            'last_name' => 'required|max:70',   // Make required
            'email' => 'required|email|unique:users,email,'.user()->id,
            'password' => 'sometimes|nullable|string|min:12|confirmed',  // Longer password
        ]);

        // ... rest of update logic
    }
}
```

Then update your routes to use the custom controller.

### Views

To customize views, publish them to your application:

```bash
php artisan vendor:publish --tag=profile-views
```

Views will be copied to `resources/views/vendor/profile/`.

### Translations

To customize translations, publish them:

```bash
php artisan vendor:publish --tag=profile-translations
```

Translations will be copied to `resources/lang/vendor/profile/`.

### Password Requirements

To change password requirements, modify the regex in `ProfileController::update()`:

```php
'password' => 'sometimes|nullable|string|min:8|confirmed|regex:/your-custom-regex/'
```

### Avatar Processing

To customize avatar processing, extend `ProfileImageController`:

```php
namespace App\Http\Controllers\Backend;

use Bongo\Profile\Http\Controllers\Backend\ProfileImageController as BaseProfileImageController;
use Bongo\Image\Http\Requests\StoreImageRequest;
use Bongo\Image\Services\AvatarImage;
use Illuminate\Http\JsonResponse;

class ProfileImageController extends BaseProfileImageController
{
    public function upload(StoreImageRequest $request): JsonResponse
    {
        $files = request()->file('files');

        try {
            $user = (new AvatarImage(user(), $files[0]))
                ->setWidth(300)
                ->setHeight(300)
                ->save();

            return response()->json([$user->getAvatar()]);
        } catch (Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}
```

## Development

### Running Tests

```bash
vendor/bin/phpunit
```

### Code Style

Check code style with Laravel Pint:

```bash
vendor/bin/pint --test
```

Fix code style issues:

```bash
vendor/bin/pint
```

### Static Analysis

Run Larastan for static analysis:

```bash
vendor/bin/phpstan analyse
```

## Security

- All routes are protected by `auth` and `employee` middleware
- CSRF protection on all forms
- Passwords are hashed using bcrypt (`Hash::make()`)
- Email validation includes DNS verification
- Strong password requirements enforced
- Email uniqueness checked (excluding soft deleted users)

## API

### Controllers

#### ProfileController

```php
Bongo\Profile\Http\Controllers\Backend\ProfileController
```

**Methods**:
- `index()` - Display profile view page
- `edit()` - Display profile edit form
- `update(Request $request)` - Update user profile

#### ProfileImageController

```php
Bongo\Profile\Http\Controllers\Backend\ProfileImageController
```

**Methods**:
- `upload(StoreImageRequest $request): JsonResponse` - Handle avatar upload

### Helper Functions

- `user()` - Get the currently authenticated user

### Translation Keys

**Namespace**: `profile::backend`

- `index` - "Profile"
- `edit` - "Edit"
- `update` - "Update"
- `update_success` - "Profile successfully updated!"
- `update_failed` - "Unable to update profile!"

Usage:
```php
trans('profile::backend.update_success')
```

## Troubleshooting

### Routes Not Working

Ensure the service provider is registered and Laravel has cached the routes:

```bash
php artisan route:clear
php artisan route:cache
```

### Avatar Upload Fails

Check:
1. The `bongo/image` package is installed
2. Storage directories are writable
3. File upload limits in `php.ini` are sufficient
4. The User model has a `getAvatar()` method

### Validation Fails

Check:
1. Email is unique in the database
2. Password meets complexity requirements
3. Password confirmation field matches

### Middleware Errors

Ensure:
1. The `employee` middleware is registered in your application
2. The authenticated user has the appropriate role/permissions

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation
- **[CLAUDE.md](CLAUDE.md)** - Claude Code guidance
- **[.cursorrules](.cursorrules)** - Cursor AI instructions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions

## License

This package is proprietary software. See [LICENSE](LICENSE) for details.

## Author

**Stuart Elliott**
- Email: stuart.elliott@bespokeuk.com
- Homepage: [https://bespokeuk.com](https://bespokeuk.com)

## Support

For issues and feature requests, please contact the development team.

## Changelog

See commit history for package changes. Future versions will maintain a CHANGELOG.md file.
