<?php

namespace Bongo\Project\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Project\Events\ProjectCategoryCreated;
use Bongo\Project\Events\ProjectCategoryDeleted;
use Bongo\Project\Events\ProjectCategoryUpdated;
use Bongo\Project\Http\Requests\StoreProjectCategoryRequest;
use Bongo\Project\Http\Requests\UpdateProjectCategoryRequest;
use Bongo\Project\Models\ProjectCategory;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class ProjectCategoryController extends AbstractController
{
    protected ProjectCategory $projectCategory;

    public function __construct(ProjectCategory $projectCategory)
    {
        $this->projectCategory = $projectCategory;
    }

    public function index(): View
    {
        return view('project::backend.category.index');
    }

    public function create(): View
    {
        $projectCategory = $this->projectCategory;

        return view('project::backend.category.create', compact('projectCategory'));
    }

    public function store(StoreProjectCategoryRequest $request): RedirectResponse
    {
        $projectCategory = $this->projectCategory->create($request->all());
        $this->syncProjects($projectCategory);
        event(new ProjectCategoryCreated($projectCategory));

        return redirect()
            ->route('backend.project_category.show', $projectCategory->id)
            ->success(trans('project::backend.category_store_success'));
    }

    public function show(ProjectCategory $projectCategory): View
    {
        return view('project::backend.category.show', compact('projectCategory'));
    }

    public function edit(ProjectCategory $projectCategory): View
    {
        return view('project::backend.category.edit', compact('projectCategory'));
    }

    public function update(UpdateProjectCategoryRequest $request, ProjectCategory $projectCategory): RedirectResponse
    {
        $projectCategory->update($request->all());
        $this->syncProjects($projectCategory);
        event(new ProjectCategoryUpdated($projectCategory));

        return redirect()
            ->route('backend.project_category.show', $projectCategory->id)
            ->success(trans('project::backend.category_update_success'));
    }

    public function destroy(ProjectCategory $projectCategory)
    {
        if ($projectCategory->delete()) {
            event(new ProjectCategoryDeleted($projectCategory));
            $redirect = back();

            // If the project was deleted from their detail view,
            // we'll redirect back to the project list view.
            if (route('backend.project_category.show', $projectCategory) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.project_category.index');
            }

            return $redirect
                ->success(trans('project::backend.category_delete_success'));
        }

        return back()
            ->error(trans('project::backend.category_delete_failed'));
    }

    private function syncProjects($projectCategory)
    {
        $projectIds = request('project_ids');
        if (isset($projectIds)) {
            $projectCategory->projects()->sync($projectIds);
        }
    }
}
