<?php

namespace Bongo\Project\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Project\Events\ProjectCreated;
use Bongo\Project\Events\ProjectDeleted;
use Bongo\Project\Events\ProjectUpdated;
use Bongo\Project\Http\Requests\StoreProjectRequest;
use Bongo\Project\Http\Requests\UpdateProjectRequest;
use Bongo\Project\Models\Project;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class ProjectController extends AbstractController
{
    protected Project $project;

    public function __construct(Project $project)
    {
        $this->project = $project;
    }

    public function index(): View
    {
        return view('project::backend.index');
    }

    public function create(): View
    {
        $project = $this->project;

        return view('project::backend.create', compact('project'));
    }

    public function store(StoreProjectRequest $request): RedirectResponse
    {
        $project = $this->project->create($request->all());
        $this->syncProjectCategories($project);
        event(new ProjectCreated($project));

        return redirect()
            ->route('backend.project.show', $project->id)
            ->success(trans('project::backend.store_success'));
    }

    public function show(Project $project): View
    {
        return view('project::backend.show', compact('project'));
    }

    public function edit(Project $project): View
    {
        return view('project::backend.edit', compact('project'));
    }

    public function update(UpdateProjectRequest $request, Project $project): RedirectResponse
    {
        $project->update($request->all());
        $this->syncProjectCategories($project);
        event(new ProjectUpdated($project));

        return redirect()
            ->route('backend.project.show', $project->id)
            ->success(trans('project::backend.update_success'));
    }

    public function destroy(Project $project)
    {
        if ($project->delete()) {
            event(new ProjectDeleted($project));
            $redirect = back();

            // If the project was deleted from their detail view,
            // we'll redirect back to the project list view.
            if (route('backend.project.show', $project) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.project.index');
            }

            return $redirect->success(trans('project::backend.delete_success'));
        }

        return back()->error(trans('project::backend.delete_failed'));
    }

    public function duplicate(Project $project)
    {
        $newProject = $project->duplicate();

        return redirect()
            ->route('backend.project.edit', $newProject->id)
            ->success(trans('project::backend.update_success'));
    }

    private function syncProjectCategories($project)
    {
        $categoryIds = request('category_ids');
        if (isset($categoryIds)) {
            $project->categories()->sync($categoryIds);
        }
    }
}
