# GitHub Copilot Instructions - Bongo Project Package

## Project Overview

This Laravel package provides a complete project portfolio management system with categories, images, and WYSIWYG content editing. It's part of the Bongo framework ecosystem and is a client-specific package for managing project showcases.

### Key Capabilities
- Projects with many-to-many categories
- Image management integration
- Frontend builder for content editing
- SEO metadata management
- Event-driven architecture
- API endpoints with resources
- Related projects functionality
- Project duplication

## Key Classes and Relationships

### Models

#### Project (`Bongo\Project\Models\Project`)
Extends: `Bongo\Framework\Models\AbstractModel`
Implements: `Bongo\Image\Interfaces\Imageable`

**Primary Traits:**
- `HasContent` - Content management
- `HasImages` - Image relationships
- `HasSeo` - SEO metadata
- `HasStatus` - Status scopes
- `HasRelated` - Related projects logic
- `HasUUID` - UUID generation
- `SoftDeletes` - Soft delete support

**Key Relationships:**
```php
public function categories(): BelongsToMany  // project_categories_pivot
```

**Key Methods:**
```php
public function duplicate(): self
public function getSummaryAttribute($dbValue): ?string
public function getRelatedByRandom(int $limit = 4): Collection  // from HasRelated
public function getPrevious(): ?Project  // from HasRelated
public function getNext(): ?Project  // from HasRelated
```

#### ProjectCategory (`Bongo\Project\Models\ProjectCategory`)
Extends: `Bongo\Framework\Models\AbstractModel`
Implements: `Bongo\Image\Interfaces\Imageable`

**Key Relationships:**
```php
public function projects(): BelongsToMany  // project_categories_pivot
```

### Controllers

#### Backend Controllers
All extend `Bongo\Framework\Http\Controllers\AbstractController`

**ProjectController** (`Bongo\Project\Http\Controllers\Backend\ProjectController`):
```php
public function index(): View
public function create(): View
public function store(StoreProjectRequest $request): RedirectResponse
public function show(Project $project): View
public function edit(Project $project): View
public function update(UpdateProjectRequest $request, Project $project): RedirectResponse
public function destroy(Project $project)
public function duplicate(Project $project)
private function syncProjectCategories($project)
```

**ProjectDatatableController** (`Bongo\Project\Http\Controllers\Backend\ProjectDatatableController`):
```php
protected function getBaseQuery(): Builder
protected function setResults()
```

#### Frontend Controllers
**ProjectController** (`Bongo\Project\Http\Controllers\Frontend\ProjectController`):
```php
public function index()  // Redirects to category index
public function show($slug)  // Public project display
public function edit($uuid)  // Builder interface (auth required)
public function update(Request $request, $uuid)  // Builder save
public function image(Request $request, $uuid)  // WYSIWYG upload
```

**ProjectCategoryController** (`Bongo\Project\Http\Controllers\Frontend\ProjectCategoryController`):
```php
public function index()  // Category listing
public function show($slug)  // Category with projects
```

### Events & Listeners

**Events:**
- `ProjectCreated`, `ProjectUpdated`, `ProjectDeleted`
- `ProjectCategoryCreated`, `ProjectCategoryUpdated`, `ProjectCategoryDeleted`

**Listeners:**
- `SetProjectDate` - Sets date on creation (implements ShouldQueue)
- `SetProjectUser` - Sets user_id on creation (implements ShouldQueue)

**External Listeners:**
- `ClearMenuCache` - From bongo/menu
- `UpdateSitemap` - From bongo/sitemap

### Service Provider
**ProjectServiceProvider** (`Bongo\Project\ProjectServiceProvider`):
Extends `Bongo\Framework\Providers\AbstractServiceProvider`

```php
protected string $module = 'project';
protected array $composers = [ /* View composers */ ];
protected array $listeners = [ /* Event listeners */ ];

public function boot(): void
{
    parent::boot();  // Auto-loads routes, views, config, migrations, translations
    AliasLoader::getInstance()->alias('ProjectCategory', ProjectCategory::class);
}
```

## Code Style Templates

### Creating a New Event Listener
```php
<?php

namespace Bongo\Project\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;

class MyListener implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        // Access model via $event->project or $event->projectCategory
    }
}
```

### Creating a New Backend Controller
```php
<?php

namespace Bongo\Project\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Project\Models\Project;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class MyController extends AbstractController
{
    protected Project $project;

    public function __construct(Project $project)
    {
        $this->project = $project;
    }

    public function index(): View
    {
        return view('project::backend.my-view');
    }
}
```

### Creating a New Form Request
```php
<?php

namespace Bongo\Project\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class MyRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => 'required|max:75',
        ];
    }
}
```

### Querying Projects
```php
// Get active projects with relationships
$projects = Project::active()
    ->with('categories', 'images')
    ->orderBy('date', 'desc')
    ->get();

// Get projects by category
$category = ProjectCategory::where('slug', $slug)->first();
$projects = $category->projects()->active()->get();

// Get related projects
$related = $project->getRelatedByRandom(4);

// Navigate between projects
$previous = $project->getPrevious();
$next = $project->getNext();
```

### Creating/Updating Projects
```php
// Create
$project = Project::create([
    'name' => 'Project Name',
    'content' => '<p>Content</p>',
    'status' => Project::ACTIVE,
]);
$project->categories()->sync([1, 2, 3]);
event(new ProjectCreated($project));

// Update
$project->update($request->all());
$project->categories()->sync($request->input('category_ids'));
event(new ProjectUpdated($project));

// Duplicate
$newProject = $project->duplicate();
```

### Working with Images
```php
// Upload and attach image
$image = new Image();
$image->name = $fileName;
$image->title = $project->name;
$image->path = $filePath;
$image->type = Image::WYSIWYG;
$image->ext = $ext;
$image->created_by = user()?->id;
$image->updated_by = user()?->id;
$project->images()->save($image);

// Get project images
$project->images;
$project->images()->where('type', Image::FEATURED)->first();
```

### Frontend Builder Pattern
```php
// Edit route (requires UUID)
Route::get('edit/{uuid}', [ProjectController::class, 'edit'])
    ->middleware(['auth', 'employee'])
    ->name('edit');

// Update with BuilderService
public function update(Request $request, $uuid)
{
    $project = Project::where('uuid', $uuid)->firstOrFail();
    $project->content = $request->get('html');

    $builderService = new BuilderService($project, '/projects/');
    $project->content = $builderService->process();
    $project->save();

    event(new ProjectUpdated($project));

    return redirect()
        ->route('frontend.project.edit', $project->uuid)
        ->success(trans('project::backend.update_success'));
}
```

## Common Patterns

### Status Constants
```php
// Use constants instead of magic strings
$project->status = Project::ACTIVE;
$project->meta_index = Project::INDEX;
$project->transparent_header = Project::ENABLED;
```

### Event Firing
```php
// Always fire events after model changes
$project->save();
event(new ProjectUpdated($project));

// Events registered in ProjectServiceProvider
protected array $listeners = [
    ProjectCreated::class => [
        SetProjectDate::class,
        SetProjectUser::class,
    ],
    ProjectUpdated::class => [
        ClearMenuCache::class,
        UpdateSitemap::class,
    ],
];
```

### Route Naming
```php
// Backend routes: backend.project.*
route('backend.project.index')
route('backend.project.show', $project->id)
route('backend.project_category.edit', $category->id)

// Frontend routes: frontend.project.*
route('frontend.project.show', $project->slug)
route('frontend.project.edit', $project->uuid)
route('frontend.project_category.show', $category->slug)

// API routes: api.project.*
route('api.project.index')
```

### View Rendering
```php
// Backend views
return view('project::backend.index');
return view('project::backend.show', compact('project'));

// Frontend views
return view('project::frontend.show', compact('project'));
return view('project::frontend.category.show', compact('category'));
```

### Session State
```php
// Set selected category for related/navigation context
if (!Session::has('selected_project_category')) {
    Session::put('selected_project_category', $project->getPrimaryCategory());
}

// Used by HasRelated trait for getPrevious/getNext
$selectedCategory = Session::get('selected_project_category');
```

### Datatable Controllers
```php
// Extend AbstractDatatableController
class MyDatatableController extends AbstractDatatableController
{
    protected function getBaseQuery(): Builder
    {
        return $this->model->newQuery();
    }

    protected function setResults()
    {
        $this->results = MyResource::collection($this->query->get());
    }
}
```

## Architecture Notes

### Service Provider Auto-Loading
The `AbstractServiceProvider` automatically loads:
- **Config**: `src/Config/project.php` → `config('project.prefix')`
- **Routes**: `src/Routes/*.php` → Auto-registered with appropriate middleware
- **Views**: `src/Views/` → `project::backend.index`
- **Migrations**: `src/Migrations/` → Auto-discovered
- **Translations**: `src/Translations/` → `trans('project::backend.store_success')`

### Route Middleware
- **api.php**: `auth:sanctum` middleware
- **backend.php**: `auth` + `employee` middleware
- **frontend.php**: Custom middleware (`hasRedirects`, `hasShortCodes`, `minifyHtml`)

### Database Relationships
```
projects (1) ←→ (M) project_categories_pivot ←→ (M) project_categories (1)
projects (1) ←→ (M) images (polymorphic)
```

### Trait Composition
Models use multiple traits from `bongo/framework`:
- `HasContent` - Content/summary methods
- `HasHeaderClass` - Header customisation
- `HasImages` - Image relationships
- `HasKey` - Unique key generation
- `HasSeo` - SEO metadata
- `HasStatus` - Status scopes
- `HasUUID` - UUID generation

### Image Storage
Images stored at: `config('image.public_path').'projects/'.$project->id.'/'`

## Dependencies

- `bongo/framework` - Base classes and traits
- `bongo/image` - Image management
- `bongo/menu` - Menu cache integration
- `bongo/package` - Package management
- `bongo/setting` - Settings management
- `bongo/sitemap` - Sitemap updates

## Testing Notes

- Tests in `tests/Unit/` and `tests/Feature/`
- Use `tests/TestCase.php` as base class
- Test with `vendor/bin/phpunit`

## Important Conventions

1. Always eager load relationships to avoid N+1 queries
2. Fire events after CRUD operations
3. Use scopes for status filtering
4. Use form requests for validation
5. Return resource collections from API endpoints
6. Use UUID for frontend builder routes
7. Check authentication for non-public project views
8. Sync categories separately after create/update
9. Process builder content with BuilderService
10. Set session state for category context
