# Architecture Documentation - Bongo Project Package

## Table of Contents
1. [Directory Structure](#directory-structure)
2. [Class Diagrams](#class-diagrams)
3. [Database Schema](#database-schema)
4. [Lifecycle Flows](#lifecycle-flows)
5. [Trait Reference](#trait-reference)
6. [Extension Points](#extension-points)
7. [How to Add Features](#how-to-add-features)

## Directory Structure

```
bongo/project/
├── src/
│   ├── Config/
│   │   └── project.php                          # Route prefixes configuration
│   ├── Events/
│   │   ├── ProjectCreated.php                   # Fired on project creation
│   │   ├── ProjectUpdated.php                   # Fired on project update
│   │   ├── ProjectDeleted.php                   # Fired on project deletion
│   │   ├── ProjectCategoryCreated.php           # Fired on category creation
│   │   ├── ProjectCategoryUpdated.php           # Fired on category update
│   │   └── ProjectCategoryDeleted.php           # Fired on category deletion
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── Api/
│   │   │   │   ├── ProjectController.php        # API: List projects as JSON
│   │   │   │   └── ProjectCategoryController.php # API: List categories as JSON
│   │   │   ├── Backend/
│   │   │   │   ├── ProjectController.php        # Admin: CRUD operations
│   │   │   │   ├── ProjectDatatableController.php # Admin: Datatable data
│   │   │   │   ├── ProjectImageController.php   # Admin: Image uploads
│   │   │   │   ├── ProjectCategoryController.php # Admin: Category CRUD
│   │   │   │   ├── ProjectCategoryDatatableController.php
│   │   │   │   └── ProjectCategoryImageController.php
│   │   │   └── Frontend/
│   │   │       ├── ProjectController.php        # Public: Show, Builder edit/update
│   │   │       ├── ProjectImageController.php   # Builder: WYSIWYG image upload
│   │   │       ├── ProjectBackgroundImageController.php # Builder: Background images
│   │   │       └── ProjectCategoryController.php # Public: Category listing/detail
│   │   ├── Requests/
│   │   │   ├── StoreProjectRequest.php          # Validation for project creation
│   │   │   ├── UpdateProjectRequest.php         # Validation for project update
│   │   │   ├── StoreProjectCategoryRequest.php
│   │   │   └── UpdateProjectCategoryRequest.php
│   │   ├── Resources/
│   │   │   ├── ProjectResource.php              # API resource transformation
│   │   │   └── ProjectCategoryResource.php
│   │   └── ViewComposers/
│   │       ├── ProjectComposer.php              # Provides project dropdown data
│   │       ├── ProjectCategoryComposer.php      # Provides category dropdown data
│   │       └── RecentProjectsComposer.php       # Provides recent projects widget
│   ├── Listeners/
│   │   ├── SetProjectDate.php                   # Sets date on creation (queued)
│   │   └── SetProjectUser.php                   # Sets user_id on creation (queued)
│   ├── Migrations/
│   │   ├── 2021_01_01_000001_create_projects_table.php
│   │   ├── 2021_01_01_000002_create_project_categories_table.php
│   │   ├── 2021_01_01_000003_create_project_categories_pivot_table.php
│   │   ├── 2021_01_01_000004_change_project_transparent_header_column_to_enum.php
│   │   ├── 2021_01_01_000005_change_project_sticky_header_column_to_enum.php
│   │   ├── 2021_01_01_000006_change_project_category_transparent_header_column_to_enum.php
│   │   ├── 2021_01_01_000007_change_project_category_sticky_header_column_to_enum.php
│   │   └── 2025_01_01_000008_add_schema_column_to_projects_table.php
│   ├── Models/
│   │   ├── Project.php                          # Main project model
│   │   └── ProjectCategory.php                  # Category model
│   ├── Routes/
│   │   ├── api.php                              # API routes (auth:sanctum)
│   │   ├── backend.php                          # Admin routes (auth + employee)
│   │   └── frontend.php                         # Public routes (custom middleware)
│   ├── Seeders/
│   │   ├── DataSeeder.php                       # Sample data seeder
│   │   └── PackageSeeder.php                    # Package registration seeder
│   ├── Traits/
│   │   └── HasRelated.php                       # Related projects functionality
│   ├── Translations/
│   │   └── en/
│   │       └── backend.php                      # English translations
│   ├── Views/
│   │   ├── backend/
│   │   │   ├── index.blade.php                  # Project listing (datatable)
│   │   │   ├── create.blade.php                 # Create form
│   │   │   ├── edit.blade.php                   # Edit form
│   │   │   ├── show.blade.php                   # Detail view
│   │   │   ├── partials/
│   │   │   │   ├── dropdowns/
│   │   │   │   │   └── project.blade.php        # Project dropdown
│   │   │   │   └── form/
│   │   │   │       └── details.blade.php        # Form fields
│   │   │   └── category/                        # Category views (similar structure)
│   │   └── frontend/
│   │       ├── show.blade.php                   # Project detail page
│   │       ├── builder.blade.php                # WYSIWYG builder interface
│   │       ├── category/
│   │       │   ├── index.blade.php              # Category listing
│   │       │   └── show.blade.php               # Category detail with projects
│   │       └── partials/
│   │           ├── project.blade.php            # Project card partial
│   │           ├── category.blade.php           # Category card partial
│   │           ├── related_projects.blade.php   # Related projects widget
│   │           └── next_previous.blade.php      # Navigation widget
│   └── ProjectServiceProvider.php               # Service provider
├── tests/
│   ├── Feature/                                 # Feature tests
│   ├── Unit/                                    # Unit tests
│   └── TestCase.php                             # Base test case
├── .cursorrules                                 # Cursor AI instructions
├── .github/
│   └── copilot-instructions.md                  # GitHub Copilot instructions
├── ARCHITECTURE.md                              # This file
├── CLAUDE.md                                    # Claude Code guidance
├── README.md                                    # Package overview
├── composer.json                                # Package dependencies
└── phpunit.xml                                  # PHPUnit configuration
```

## Class Diagrams

### Model Hierarchy

```
AbstractModel (bongo/framework)
    │
    ├─── Project
    │     Implements: Imageable (bongo/image)
    │     Traits: HasContent, HasHeaderClass, HasImages, HasKey,
    │            HasRelated, HasSeo, HasStatus, HasUUID, SoftDeletes
    │     Relations:
    │       ├─ categories(): BelongsToMany → ProjectCategory
    │       └─ images(): MorphMany → Image (polymorphic)
    │
    └─── ProjectCategory
          Implements: Imageable (bongo/image)
          Traits: HasContent, HasHeaderClass, HasImages, HasKey,
                 HasSeo, HasStatus, HasUUID, SoftDeletes
          Relations:
            ├─ projects(): BelongsToMany → Project
            └─ images(): MorphMany → Image (polymorphic)
```

### Controller Hierarchy

```
AbstractController (bongo/framework)
    │
    ├─── Backend Controllers
    │     ├─ ProjectController
    │     │   Methods: index, create, store, show, edit, update, destroy, duplicate
    │     ├─ ProjectCategoryController
    │     │   Methods: index, create, store, show, edit, update, destroy
    │     ├─ ProjectImageController
    │     │   Methods: upload
    │     └─ ProjectCategoryImageController
    │         Methods: upload
    │
    └─── Frontend Controllers
          ├─ ProjectController
          │   Methods: index, show, edit, update, image
          ├─ ProjectImageController
          │   Methods: upload
          ├─ ProjectBackgroundImageController
          │   Methods: upload
          └─ ProjectCategoryController
              Methods: index, show

AbstractDatatableController (bongo/framework)
    │
    └─── Datatable Controllers
          ├─ ProjectDatatableController
          │   Methods: getBaseQuery, setResults
          └─ ProjectCategoryDatatableController
              Methods: getBaseQuery, setResults
```

### Event & Listener Flow

```
Event (Illuminate\Foundation\Events)
    │
    ├─── ProjectCreated
    │     Listeners:
    │       ├─ SetProjectDate (queued)
    │       └─ SetProjectUser (queued)
    │
    ├─── ProjectUpdated
    │     Listeners:
    │       ├─ ClearMenuCache (bongo/menu)
    │       └─ UpdateSitemap (bongo/sitemap)
    │
    ├─── ProjectDeleted
    │     Listeners:
    │       ├─ ClearMenuCache (bongo/menu)
    │       └─ UpdateSitemap (bongo/sitemap)
    │
    ├─── ProjectCategoryCreated
    │     Listeners: (none registered)
    │
    ├─── ProjectCategoryUpdated
    │     Listeners:
    │       ├─ ClearMenuCache (bongo/menu)
    │       └─ UpdateSitemap (bongo/sitemap)
    │
    └─── ProjectCategoryDeleted
          Listeners:
            ├─ ClearMenuCache (bongo/menu)
            └─ UpdateSitemap (bongo/sitemap)
```

### Service Provider Bootstrap Flow

```
AbstractServiceProvider::boot()
    │
    ├─ registerConfig()           → Loads src/Config/project.php
    ├─ registerRoutes()           → Loads src/Routes/*.php with middleware
    │   ├─ api.php                → Prefix: /api, Middleware: auth:sanctum
    │   ├─ backend.php            → Prefix: /admin, Middleware: auth, employee
    │   └─ frontend.php           → Named: frontend.*, Middleware: custom
    ├─ registerViews()            → Loads src/Views/project/
    ├─ registerMigrations()       → Loads src/Migrations/
    ├─ registerTranslations()     → Loads src/Translations/
    ├─ registerMiddleware()       → Registers middleware (if any)
    ├─ registerCommands()         → Registers commands (if any)
    ├─ registerComposers()        → Registers view composers
    └─ registerEventListeners()   → Registers event listeners

ProjectServiceProvider::boot()
    │
    ├─ parent::boot()             → Executes AbstractServiceProvider::boot()
    └─ Alias 'ProjectCategory'    → Creates facade alias
```

## Database Schema

### Entity Relationship Diagram

```
┌─────────────────────────┐           ┌────────────────────────────────┐
│      projects           │           │  project_categories_pivot      │
├─────────────────────────┤           ├────────────────────────────────┤
│ id (PK)                 │───────────│ project_id (FK)                │
│ uuid (indexed)          │         M │ project_category_id (FK)       │
│ user_id (FK, nullable)  │           └────────────────────────────────┘
│ date                    │                          │
│ name                    │                          │ M
│ key (indexed)           │                          │
│ slug (indexed)          │           ┌──────────────────────────────┐
│ content                 │           │  project_categories          │
│ summary                 │           ├──────────────────────────────┤
│ status (enum)           │           │ id (PK)                      │
│ transparent_header      │           │ uuid (indexed)               │
│ sticky_header           │           │ name                         │
│ css                     │           │ key (indexed)                │
│ js                      │           │ slug (indexed)               │
│ schema                  │           │ content                      │
│ meta_title              │           │ status (enum)                │
│ meta_description        │           │ transparent_header           │
│ meta_index (enum)       │           │ sticky_header                │
│ meta_canonical          │           │ css                          │
│ created_by, updated_by  │           │ js                           │
│ deleted_by, deleted_at  │           │ meta_title                   │
│ timestamps              │           │ meta_description             │
└─────────────────────────┘           │ meta_index (enum)            │
                                      │ meta_canonical               │
         │                            │ created_by, updated_by       │
         │ Polymorphic                │ deleted_by, deleted_at       │
         │                            │ timestamps                   │
         │                            └──────────────────────────────┘
         │                                       │
         │ M                                     │ Polymorphic
         │                                       │
┌─────────────────────────┐                     │ M
│      images             │─────────────────────┘
├─────────────────────────┤
│ id (PK)                 │
│ imageable_id (FK)       │
│ imageable_type          │  ← 'Bongo\Project\Models\Project'
│ name                    │     'Bongo\Project\Models\ProjectCategory'
│ title                   │
│ path                    │
│ type (enum)             │  ← FEATURED, GALLERY, WYSIWYG, etc.
│ ext                     │
│ created_by, updated_by  │
│ timestamps              │
└─────────────────────────┘
```

### Table Details

#### projects
| Column | Type | Nullable | Indexed | Notes |
|--------|------|----------|---------|-------|
| id | increments | No | Primary | Auto-increment |
| uuid | uuid | No | Yes | For frontend builder URLs |
| user_id | unsignedInteger | Yes | Yes | Foreign key to users |
| date | timestamp | Yes | No | Project date |
| name | string | No | No | Project name |
| key | string | Yes | Yes | Unique key for API/integration |
| slug | string | No | Yes | URL-friendly identifier |
| content | longText | Yes | No | Main content (HTML) |
| summary | mediumText | Yes | No | Short description |
| status | enum | No | No | pending, active, inactive |
| transparent_header | tinyInteger | No | No | 0 = disabled, 1 = enabled |
| sticky_header | tinyInteger | No | No | 0 = disabled, 1 = enabled |
| css | text | Yes | No | Custom CSS overrides |
| js | text | Yes | No | Custom JS overrides |
| schema | text | Yes | No | JSON-LD structured data (v3.0+) |
| meta_title | string | Yes | No | SEO title |
| meta_description | text | Yes | No | SEO description |
| meta_index | enum | No | No | index, noindex |
| meta_canonical | string | Yes | No | Canonical URL |
| created_by, updated_by, deleted_by | unsignedInteger | Yes | Yes | Audit trail |
| created_at, updated_at | timestamps | No | No | Laravel timestamps |
| deleted_at | timestamp | Yes | No | Soft delete |

#### project_categories
Similar structure to projects, excluding: user_id, date, schema

#### project_categories_pivot
| Column | Type | Nullable | Indexed | Notes |
|--------|------|----------|---------|-------|
| project_id | unsignedInteger | No | Yes | Foreign key to projects |
| project_category_id | unsignedInteger | No | Yes | Foreign key to project_categories |

## Lifecycle Flows

### Project Creation Flow

```
1. User Request
   └─> POST /admin/projects/store

2. StoreProjectRequest Validation
   └─> Rules: name required, unique slug, max 75 chars

3. Backend\ProjectController::store()
   ├─> Project::create($request->all())
   │   ├─> Name auto-converted to slug via HasKey trait
   │   └─> UUID auto-generated via HasUUID trait
   │
   ├─> syncProjectCategories($project)
   │   └─> $project->categories()->sync($categoryIds)
   │
   └─> event(new ProjectCreated($project))
       ├─> SetProjectDate listener (queued)
       │   └─> Sets date to now() if empty
       │
       └─> SetProjectUser listener (queued)
           └─> Sets user_id to auth()->user()->id

4. Redirect to backend.project.show
   └─> With success message
```

### Project Update Flow

```
1. User Request
   └─> POST /admin/projects/{project}/update

2. UpdateProjectRequest Validation
   └─> Rules: name required, unique slug (except current), max 75 chars

3. Backend\ProjectController::update()
   ├─> $project->update($request->all())
   │
   ├─> syncProjectCategories($project)
   │   └─> $project->categories()->sync($categoryIds)
   │
   └─> event(new ProjectUpdated($project))
       ├─> ClearMenuCache listener
       │   └─> Clears menu cache if project in menu
       │
       └─> UpdateSitemap listener
           └─> Regenerates sitemap with updated project

4. Redirect to backend.project.show
   └─> With success message
```

### Frontend Display Flow

```
1. User Request
   └─> GET /projects/{slug}

2. Frontend\ProjectController::show($slug)
   ├─> Project::where('slug', $slug)->first()
   │
   ├─> Check if project exists and is active
   │   └─> If not: abort(404)
   │
   ├─> Get primary category
   │   └─> If not in session: Session::put('selected_project_category', $primary)
   │
   └─> return view('project::frontend.show', compact('project'))
       ├─> View can access:
       │   ├─> $project->content
       │   ├─> $project->categories
       │   ├─> $project->images
       │   ├─> $project->getRelatedByRandom(4)
       │   ├─> $project->getPrevious()
       │   └─> $project->getNext()
       │
       └─> Middleware applied:
           ├─> hasRedirects (checks for URL redirects)
           ├─> hasShortCodes (processes shortcodes)
           └─> minifyHtml (minifies HTML output)
```

### Builder Edit/Update Flow

```
1. User Request (Authenticated)
   └─> GET /projects/edit/{uuid}

2. Frontend\ProjectController::edit($uuid)
   ├─> Project::where('uuid', $uuid)->firstOrFail()
   │   └─> If not found: abort(404)
   │
   └─> return view('project::frontend.builder', compact('project'))
       └─> Loads WYSIWYG builder interface

3. User Edits Content
   └─> POST /projects/update/{uuid}

4. Frontend\ProjectController::update($uuid)
   ├─> Project::where('uuid', $uuid)->firstOrFail()
   │
   ├─> $project->content = $request->get('html')
   │
   ├─> BuilderService processes content
   │   ├─> Downloads remote images
   │   ├─> Saves to local storage
   │   └─> Updates image URLs in HTML
   │
   ├─> $project->save()
   │
   └─> event(new ProjectUpdated($project))
       ├─> ClearMenuCache
       └─> UpdateSitemap

5. Redirect to frontend.project.edit
   └─> With success message
```

### Related Projects Query Flow

```
1. $project->getRelatedByRandom(4)
   │
   ├─> $project->loadMissing('categories')
   │
   ├─> getRelated() query
   │   └─> Project::active()
   │       ->join('project_categories_pivot')
   │       ->where('projects.id', '!=', $this->id)
   │       ->whereIn('category_id', $this->categories->pluck('id'))
   │
   ├─> ->take(4)
   │   ->inRandomOrder()
   │   ->pluck('projects.id')
   │
   └─> Project::with('images')->find($relatedProjectIds)
```

### Previous/Next Navigation Flow

```
1. $project->getPrevious() or $project->getNext()
   │
   ├─> $project->loadMissing('categories')
   │
   ├─> Get selected category from session
   │   └─> Session::get('selected_project_category', $this->getPrimaryCategory())
   │
   ├─> Query projects in selected category
   │   ├─> For Previous: where('id', '<', $this->id)->orderBy('id', 'DESC')
   │   └─> For Next: where('id', '>', $this->id)->orderBy('id', 'ASC')
   │
   └─> ->active()->with('images')->first()
```

## Trait Reference

### Traits Used by Project & ProjectCategory Models

| Trait | Source | Purpose | Key Methods |
|-------|--------|---------|-------------|
| HasContent | bongo/framework | Content management | `getContent()`, `getSummary()`, `truncateContent($length)` |
| HasHeaderClass | bongo/framework | Header customisation | `isTransparentHeader()`, `isStickyHeader()`, `getHeaderClass()` |
| HasImages | bongo/image | Image relationships | `images()`, `getFeaturedImage()`, `getGalleryImages()` |
| HasKey | bongo/framework | Key generation | `formatKey($name)`, boot hook for auto-key |
| HasRelated | bongo/project (custom) | Related projects | `getRelated()`, `getRelatedByRandom()`, `getPrevious()`, `getNext()` |
| HasSeo | bongo/framework | SEO metadata | `getMetaTitle()`, `getMetaDescription()`, `getMetaCanonical()` |
| HasStatus | bongo/framework | Status scopes | `active()`, `inactive()`, `pending()`, `isActive()` |
| HasUUID | bongo/framework | UUID generation | boot hook for auto-UUID generation |
| SoftDeletes | Laravel | Soft delete | `delete()`, `restore()`, `forceDelete()`, `withTrashed()` |

### Trait Method Details

#### HasContent Methods
```php
getContent(): ?string                    // Returns processed content
getSummary(int $limit = 150): ?string    // Returns or generates summary
truncateContent(int $length = 150): string  // Truncates content to length
```

#### HasHeaderClass Methods
```php
isTransparentHeader(): bool              // Checks if transparent_header == 1
isStickyHeader(): bool                   // Checks if sticky_header == 1
getHeaderClass(): string                 // Returns CSS classes for header
```

#### HasImages Methods (via Imageable)
```php
images(): MorphMany                      // Polymorphic relationship
getFeaturedImage(): ?Image               // Gets first FEATURED type image
getGalleryImages(): Collection           // Gets all GALLERY type images
```

#### HasKey Methods
```php
formatKey(string $name): string          // Formats name to lowercase alphanumeric key
// Boot hook automatically generates key from name on creation
```

#### HasRelated Methods (Custom Trait)
```php
getRelated(): Builder                    // Base query for related projects
getRelatedByRandom(int $limit = 4): Collection  // Random related projects
getPrimaryCategory(): ?ProjectCategory   // First category
getPrevious(): ?Project                  // Previous project in category
getNext(): ?Project                      // Next project in category
```

#### HasSeo Methods
```php
getMetaTitle(): string                   // Returns meta_title or falls back to name
getMetaDescription(): ?string            // Returns meta_description or generates from content
getMetaCanonical(): ?string              // Returns meta_canonical URL
getMetaIndex(): string                   // Returns meta_index (index/noindex)
```

#### HasStatus Methods
```php
// Scopes
active(): Builder                        // where('status', 'active')
inactive(): Builder                      // where('status', 'inactive')
pending(): Builder                       // where('status', 'pending')

// Checkers
isActive(): bool                         // status === 'active'
isInactive(): bool                       // status === 'inactive'
isPending(): bool                        // status === 'pending'
```

#### HasUUID Methods
```php
// Boot hook automatically generates UUID on creation
// No public methods, just automatic behaviour
```

## Extension Points

### Adding a New Field to Projects

**1. Create Migration**
```bash
php artisan make:migration add_my_field_to_projects_table
```

**2. Update Migration**
```php
Schema::table('projects', function (Blueprint $table) {
    $table->string('my_field')->nullable()->after('content');
});
```

**3. Update Model Fillable**
```php
// src/Models/Project.php
protected $fillable = [
    // ... existing fields
    'my_field',
];
```

**4. Update Form Request Validation**
```php
// src/Http/Requests/StoreProjectRequest.php
public function rules(): array
{
    return [
        'name' => 'required|unique:projects,slug,NULL,id,deleted_at,NULL|max:75',
        'my_field' => 'required|string|max:255',
    ];
}
```

**5. Update Views**
```blade
{{-- src/Views/backend/partials/form/details.blade.php --}}
<div class="form-group">
    <label for="my_field">My Field</label>
    <input type="text" name="my_field" value="{{ old('my_field', $project->my_field) }}" class="form-control">
</div>
```

### Adding a New Event Listener

**1. Create Listener Class**
```php
// src/Listeners/MyListener.php
<?php

namespace Bongo\Project\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;

class MyListener implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        $project = $event->project;
        // Your logic here
    }
}
```

**2. Register in Service Provider**
```php
// src/ProjectServiceProvider.php
protected array $listeners = [
    ProjectCreated::class => [
        SetProjectDate::class,
        SetProjectUser::class,
        MyListener::class,  // Add here
    ],
];
```

### Adding a New Route

**1. Add Route to Appropriate File**
```php
// For backend: src/Routes/backend.php
Route::get('my-action/{project}', [ProjectController::class, 'myAction'])
    ->name('my_action');

// For frontend: src/Routes/frontend.php
Route::get('my-public-action/{slug}', [ProjectController::class, 'myPublicAction'])
    ->name('my_public_action');

// For API: src/Routes/api.php
Route::get('my-api-action', [ProjectController::class, 'myApiAction'])
    ->name('my_api_action');
```

**2. Add Controller Method**
```php
// src/Http/Controllers/Backend/ProjectController.php (or Frontend/Api)
public function myAction(Project $project): View
{
    // Your logic here
    return view('project::backend.my-view', compact('project'));
}
```

### Adding a New View Composer

**1. Create Composer Class**
```php
// src/Http/ViewComposers/MyComposer.php
<?php

namespace Bongo\Project\Http\ViewComposers;

use Bongo\Project\Models\Project;
use Illuminate\View\View;

class MyComposer
{
    public function compose(View $view): void
    {
        $data = Project::active()->get();
        $view->with('myData', $data);
    }
}
```

**2. Register in Service Provider**
```php
// src/ProjectServiceProvider.php
protected array $composers = [
    ProjectComposer::class => [
        'project::backend.partials.dropdowns.project',
    ],
    MyComposer::class => [  // Add here
        'project::backend.my-view',
        'project::frontend.my-view',
    ],
];
```

### Adding a New API Resource

**1. Create Resource Class**
```php
// src/Http/Resources/MyResource.php
<?php

namespace Bongo\Project\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;

class MyResource extends JsonResource
{
    public function toArray($request): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            // ... your fields
        ];
    }
}
```

**2. Use in Controller**
```php
// src/Http/Controllers/Api/ProjectController.php
public function index(): JsonResponse
{
    $projects = Project::active()->get();
    return MyResource::collection($projects);
}
```

### Adding Custom Middleware

**1. Create Middleware Class**
```php
// src/Http/Middleware/MyMiddleware.php
<?php

namespace Bongo\Project\Http\Middleware;

use Closure;

class MyMiddleware
{
    public function handle($request, Closure $next)
    {
        // Your logic here
        return $next($request);
    }
}
```

**2. Register in Service Provider**
```php
// src/ProjectServiceProvider.php
protected array $middlewares = [
    'myMiddleware' => MyMiddleware::class,
];
```

**3. Apply to Routes**
```php
// src/Routes/frontend.php
Route::get('{slug}', [ProjectController::class, 'show'])
    ->middleware(['hasRedirects', 'hasShortCodes', 'myMiddleware'])
    ->name('show');
```

## How to Add Features

### Example: Adding Tags to Projects

**1. Create Migration for Tags Table**
```bash
php artisan make:migration create_project_tags_table
```

```php
Schema::create('project_tags', function (Blueprint $table) {
    $table->increments('id');
    $table->string('name');
    $table->string('slug')->index();
    $table->timestamps();
});

Schema::create('project_tag_pivot', function (Blueprint $table) {
    $table->unsignedInteger('project_id');
    $table->unsignedInteger('project_tag_id');
});
```

**2. Create ProjectTag Model**
```php
// src/Models/ProjectTag.php
<?php

namespace Bongo\Project\Models;

use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasKey;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class ProjectTag extends AbstractModel
{
    use HasKey;

    protected $fillable = ['name', 'slug'];

    public function projects(): BelongsToMany
    {
        return $this->belongsToMany(Project::class, 'project_tag_pivot');
    }
}
```

**3. Update Project Model**
```php
// src/Models/Project.php
public function tags(): BelongsToMany
{
    return $this->belongsToMany(ProjectTag::class, 'project_tag_pivot');
}
```

**4. Create Tag Controller**
```php
// src/Http/Controllers/Backend/ProjectTagController.php
// Implement CRUD operations similar to ProjectCategoryController
```

**5. Add Tag Routes**
```php
// src/Routes/backend.php
Route::as('project_tag.')
    ->prefix('project-tags')
    ->group(function () {
        Route::get('/', [ProjectTagController::class, 'index'])->name('index');
        // ... other CRUD routes
    });
```

**6. Update Views to Include Tags**
```blade
{{-- src/Views/backend/partials/form/details.blade.php --}}
<div class="form-group">
    <label for="tag_ids">Tags</label>
    <select name="tag_ids[]" multiple class="form-control">
        @foreach($tags as $tag)
            <option value="{{ $tag->id }}"
                {{ in_array($tag->id, old('tag_ids', $project->tags->pluck('id')->toArray())) ? 'selected' : '' }}>
                {{ $tag->name }}
            </option>
        @endforeach
    </select>
</div>
```

**7. Update Controller to Sync Tags**
```php
// src/Http/Controllers/Backend/ProjectController.php
private function syncProjectTags($project)
{
    $tagIds = request('tag_ids');
    if (isset($tagIds)) {
        $project->tags()->sync($tagIds);
    }
}

public function store(StoreProjectRequest $request): RedirectResponse
{
    $project = $this->project->create($request->all());
    $this->syncProjectCategories($project);
    $this->syncProjectTags($project);  // Add this
    event(new ProjectCreated($project));
    // ...
}
```

**8. Create Tag View Composer**
```php
// src/Http/ViewComposers/ProjectTagComposer.php
<?php

namespace Bongo\Project\Http\ViewComposers;

use Bongo\Project\Models\ProjectTag;
use Illuminate\View\View;

class ProjectTagComposer
{
    public function compose(View $view): void
    {
        $tags = ProjectTag::orderBy('name')->get();
        $view->with('tags', $tags);
    }
}
```

**9. Register Tag Composer**
```php
// src/ProjectServiceProvider.php
protected array $composers = [
    // ... existing composers
    ProjectTagComposer::class => [
        'project::backend.create',
        'project::backend.edit',
    ],
];
```

This completes the architecture documentation for the Bongo Project package. The structure provides clear extension points and patterns for adding new features while maintaining consistency with the existing codebase.
