# Bongo Project Package - Claude Code Guidance

## Overview

The **Bongo Project** package provides a complete project portfolio management system for Laravel applications. It enables administrators to create, manage, and display projects with categories, images, and custom content through both a traditional admin interface and a frontend WYSIWYG builder.

**Package**: `bongo/project` (client-specific package)
**Namespace**: `Bongo\Project`
**PHP**: 8.2+
**Laravel**: 10+

## Documentation Links

- **ARCHITECTURE.md** - Detailed architecture diagrams, database schema, lifecycle flows
- **.cursorrules** - Comprehensive coding conventions and patterns for Cursor AI
- **.github/copilot-instructions.md** - GitHub Copilot instructions and code templates
- **README.md** - Installation and quick start guide

## Commands

From the package directory (`custom/project`):

```bash
# Testing
vendor/bin/phpunit                  # Run all tests
vendor/bin/phpunit --filter MyTest  # Run specific test

# Code Quality
vendor/bin/pint                     # Fix code style issues
vendor/bin/pint --test              # Check code style without fixing
vendor/bin/phpstan analyse          # Run static analysis

# Dependencies
composer install                    # Install dependencies
composer update -W                  # Update dependencies
rm composer.lock && composer update -W  # Force update all
```

## Quick Architecture Reference

### Service Provider Pattern

**ProjectServiceProvider** extends `Bongo\Framework\Providers\AbstractServiceProvider`:

```php
protected string $module = 'project';  // Config/view namespace

protected array $composers = [         // View composers
    ProjectComposer::class => ['project::backend.partials.dropdowns.project'],
];

protected array $listeners = [         // Event listeners
    ProjectCreated::class => [SetProjectDate::class, SetProjectUser::class],
    ProjectUpdated::class => [ClearMenuCache::class, UpdateSitemap::class],
    ProjectDeleted::class => [ClearMenuCache::class, UpdateSitemap::class],
];

public function boot(): void
{
    parent::boot();  // Auto-loads routes, views, config, migrations, translations
    AliasLoader::getInstance()->alias('ProjectCategory', ProjectCategory::class);
}
```

### Route Middleware Patterns

- **api.php** → `/api/projects` with `auth:sanctum` middleware
- **backend.php** → `/admin/projects` with `auth` + `employee` middleware
- **frontend.php** → `/projects` with custom middleware (`hasRedirects`, `hasShortCodes`, `minifyHtml`)

### Model Structure

**Project Model** (`Bongo\Project\Models\Project`):
- Extends: `Bongo\Framework\Models\AbstractModel`
- Implements: `Bongo\Image\Interfaces\Imageable`
- **Traits**: `HasContent`, `HasHeaderClass`, `HasImages`, `HasKey`, `HasRelated`, `HasSeo`, `HasStatus`, `HasUUID`, `SoftDeletes`
- **Relationships**: `categories()` (BelongsToMany), `images()` (MorphMany)
- **Key Methods**: `duplicate()`, `getRelatedByRandom()`, `getPrevious()`, `getNext()`

**ProjectCategory Model** (`Bongo\Project\Models\ProjectCategory`):
- Same structure as Project (minus HasRelated trait)
- **Relationships**: `projects()` (BelongsToMany), `images()` (MorphMany)

### Database Tables

- **projects** - Main project data (68 columns including audit fields)
- **project_categories** - Category data
- **project_categories_pivot** - Many-to-many relationship (project_id, project_category_id)
- **images** - Polymorphic image relationships (imageable_id, imageable_type)

## Key Files Reference

### Core Files

| File | Purpose | Key Elements |
|------|---------|--------------|
| `src/ProjectServiceProvider.php` | Service provider | `$module`, `$composers`, `$listeners`, boot() |
| `src/Models/Project.php` | Main model | Constants, traits, relationships, duplicate() |
| `src/Models/ProjectCategory.php` | Category model | Similar to Project, relationships |
| `src/Config/project.php` | Configuration | Route prefixes |
| `src/Traits/HasRelated.php` | Related logic | getRelated(), getRelatedByRandom(), getPrevious(), getNext() |

### Controllers

| File | Purpose | Key Methods |
|------|---------|-------------|
| `src/Http/Controllers/Backend/ProjectController.php` | Admin CRUD | index, create, store, show, edit, update, destroy, duplicate |
| `src/Http/Controllers/Backend/ProjectDatatableController.php` | Datatable data | getBaseQuery(), setResults() |
| `src/Http/Controllers/Frontend/ProjectController.php` | Public display + Builder | index, show, edit, update, image |
| `src/Http/Controllers/Frontend/ProjectCategoryController.php` | Category display | index, show |
| `src/Http/Controllers/Api/ProjectController.php` | API endpoints | index |

### Form Requests

| File | Purpose | Key Rules |
|------|---------|-----------|
| `src/Http/Requests/StoreProjectRequest.php` | Create validation | name required, unique slug, max 75 |
| `src/Http/Requests/UpdateProjectRequest.php` | Update validation | name required, unique slug (except self), max 75 |

### Events & Listeners

| File | Purpose | When Fired |
|------|---------|------------|
| `src/Events/ProjectCreated.php` | Project created | After Project::create() |
| `src/Events/ProjectUpdated.php` | Project updated | After $project->update() |
| `src/Events/ProjectDeleted.php` | Project deleted | After $project->delete() |
| `src/Listeners/SetProjectDate.php` | Sets date (queued) | On ProjectCreated |
| `src/Listeners/SetProjectUser.php` | Sets user_id (queued) | On ProjectCreated |

### Views

| Directory | Purpose | Key Views |
|-----------|---------|-----------|
| `src/Views/backend/` | Admin interface | index, create, edit, show, partials/form/details |
| `src/Views/backend/category/` | Category admin | index, create, edit, show |
| `src/Views/frontend/` | Public views | show, builder, category/index, category/show |
| `src/Views/frontend/partials/` | Reusable components | project, category, related_projects, next_previous |

### Routes

| File | Prefix | Middleware | Purpose |
|------|--------|------------|---------|
| `src/Routes/backend.php` | `/admin/{prefix}` | auth, employee | Admin CRUD |
| `src/Routes/frontend.php` | `/{prefix}` | hasRedirects, hasShortCodes, minifyHtml | Public + Builder |
| `src/Routes/api.php` | `/api/{prefix}` | auth:sanctum | API endpoints |

## Code Style Summary

### Naming Conventions

- **Models**: Singular PascalCase (`Project`, `ProjectCategory`)
- **Controllers**: PascalCase with suffix (`ProjectController`, `ProjectDatatableController`)
- **Events**: PascalCase past tense (`ProjectCreated`, `ProjectUpdated`)
- **Listeners**: PascalCase imperative (`SetProjectDate`, `SetProjectUser`)
- **Routes**: Snake_case with dots (`backend.project.index`, `frontend.project.show`)
- **Views**: Lowercase with dots (`project::backend.index`, `project::frontend.show`)
- **Config**: Lowercase (`project.prefix`, `project.category_prefix`)

### Return Types

- Controllers: `View`, `RedirectResponse`, `JsonResponse`
- Models: Nullable types (`?string`, `?ProjectCategory`)
- Requests: `array` from `rules()`
- Resources: `array` from `toArray()`
- Listeners: `void` from `handle()`

### Constants Usage

```php
// Status
Project::PENDING      // 'pending'
Project::ACTIVE       // 'active'
Project::INACTIVE     // 'inactive'

// Meta Index
Project::INDEX        // 'index'
Project::NO_INDEX     // 'noindex'

// Boolean flags
Project::ENABLED      // 1
Project::DISABLED     // 0
```

### Query Patterns

```php
// Status scopes
Project::active()->get()
Project::inactive()->get()
Project::pending()->get()

// Eager loading (avoid N+1)
Project::with('categories', 'images')->get()

// Soft deletes
Project::withTrashed()->get()
Project::onlyTrashed()->get()

// Related projects
$project->getRelatedByRandom(4)
$project->getPrevious()
$project->getNext()
```

### Event Firing Pattern

```php
// Always fire events AFTER model changes
$project = Project::create($request->all());
event(new ProjectCreated($project));

$project->update($request->all());
event(new ProjectUpdated($project));

$project->delete();
event(new ProjectDeleted($project));
```

### Validation Rules

```php
// Unique with soft-delete awareness
'name' => "required|unique:{$table},slug,NULL,id,deleted_at,NULL|max:75"

// Unique except current record
'name' => "required|unique:{$table},slug,{$id},id,deleted_at,NULL|max:75"
```

### Image Handling

```php
// Storage path
$filePath = config('image.public_path').'projects/'.$project->id.'/';

// Create image record
$image = new Image();
$image->name = $fileName;
$image->title = $project->name;
$image->path = $filePath;
$image->type = Image::WYSIWYG;  // or FEATURED, GALLERY
$image->ext = $ext;
$image->created_by = user()?->id;
$image->updated_by = user()?->id;
$project->images()->save($image);
```

### Builder Service Pattern

```php
use Bongo\Image\Services\BuilderService;

public function update(Request $request, $uuid)
{
    $project = Project::where('uuid', $uuid)->firstOrFail();
    $project->content = $request->get('html');

    // Process images: download remote, save local, update URLs
    $builderService = new BuilderService($project, '/projects/');
    $project->content = $builderService->process();
    $project->save();

    event(new ProjectUpdated($project));

    return redirect()->route('frontend.project.edit', $project->uuid)
        ->success(trans('project::backend.update_success'));
}
```

## Common Tasks

### Creating a Project

```php
$project = Project::create([
    'name' => 'My Project',
    'content' => '<p>Project content</p>',
    'summary' => 'Short description',
    'status' => Project::ACTIVE,
]);

// Sync categories
$project->categories()->sync([1, 2, 3]);

// Fire event
event(new ProjectCreated($project));
```

### Querying Projects

```php
// Active projects with relationships
$projects = Project::active()
    ->with('categories', 'images')
    ->orderBy('date', 'desc')
    ->get();

// Projects by category
$category = ProjectCategory::where('slug', $slug)->first();
$projects = $category->projects()->active()->get();

// Projects with specific status
$pending = Project::pending()->get();
```

### Working with Images

```php
// Get featured image
$featuredImage = $project->getFeaturedImage();

// Get gallery images
$galleryImages = $project->getGalleryImages();

// Get all images
$allImages = $project->images;
```

### Related Projects

```php
// Get 4 random related projects
$related = $project->getRelatedByRandom(4);

// Navigate between projects
$previous = $project->getPrevious();
$next = $project->getNext();

// Get primary category
$primaryCategory = $project->getPrimaryCategory();
```

### Duplicating a Project

```php
$newProject = $project->duplicate();
// Creates copy with:
// - New UUID
// - Randomised name (original + 6 chars)
// - New slug from name
// - Status = pending
// - Date = now
// - user_id = current user
// - Cloned category relationships
// - Cloned image relationships
```

### SEO Metadata

```php
// Get SEO fields (with fallbacks)
$metaTitle = $project->getMetaTitle();              // meta_title or name
$metaDescription = $project->getMetaDescription();  // meta_description or auto-generated
$metaCanonical = $project->getMetaCanonical();      // meta_canonical URL
$metaIndex = $project->getMetaIndex();              // 'index' or 'noindex'
```

## Configuration

### Config Values (`src/Config/project.php`)

```php
'prefix' => 'projects'                    // Frontend/backend route prefix
'category_prefix' => 'project-categories' // Category route prefix
```

### Route URLs

```php
// Backend
/admin/projects
/admin/projects/create
/admin/projects/{id}/edit
/admin/project-categories
/admin/project-categories/{id}/edit

// Frontend
/projects
/projects/{slug}
/projects/edit/{uuid}          // Builder (auth required)
/project-categories
/project-categories/{slug}

// API
/api/projects
/api/project-categories
```

## Dependencies

- **bongo/framework** - Base classes (AbstractServiceProvider, AbstractModel, AbstractController)
- **bongo/image** - Image management (Imageable interface, HasImages trait, BuilderService)
- **bongo/menu** - Menu integration (ClearMenuCache listener)
- **bongo/package** - Package management
- **bongo/setting** - Settings management
- **bongo/sitemap** - Sitemap updates (UpdateSitemap listener)

## Best Practices

1. **Always fire events** after CRUD operations for cache/sitemap integrity
2. **Use scopes** for status filtering (`active()`, `inactive()`, `pending()`)
3. **Eager load relationships** to avoid N+1 queries (`with('categories', 'images')`)
4. **Check authentication** in frontend show methods for non-public projects
5. **Sync categories** separately after create/update using `syncProjectCategories()`
6. **Use UUID** for frontend builder routes to prevent URL prediction
7. **Set selected category** in session for related project context navigation
8. **Process builder content** with BuilderService to handle image paths
9. **Use form requests** for validation to keep controllers lean
10. **Return resource collections** from API endpoints for consistent structure

## Important Notes

- Projects are **soft-deleted** (use `withTrashed()`, `onlyTrashed()` for queries)
- Slugs auto-generated from names but can be customised
- Keys auto-generated for API/integration purposes
- Header classes (transparent/sticky) stored as tinyIntegers (0/1)
- Date field uses custom `Date` cast from bongo/framework
- Summary auto-generates from content if not provided
- Project duplication creates pending copy with randomised name
- Session state affects related/previous/next queries
- Builder routes require both `auth` and `employee` middleware
- Schema field supports JSON-LD structured data for SEO (added v3.0)

## Troubleshooting

### Common Issues

**Issue**: Related projects not showing
- **Check**: Session has `selected_project_category` set
- **Check**: Projects share at least one category
- **Check**: Related projects are `active` status

**Issue**: Builder not saving content
- **Check**: User is authenticated with `employee` role
- **Check**: UUID is valid and exists
- **Check**: BuilderService is processing images correctly

**Issue**: Images not displaying
- **Check**: Image path configured in `config('image.public_path')`
- **Check**: Storage disk configured correctly
- **Check**: Image relationships properly synced

**Issue**: Duplicate creates too many records
- **Check**: Categories and images sync happening after `replicate()`
- **Check**: Relations cleared before re-loading (`$this->relations = []`)

## Testing

```bash
# Run all tests
vendor/bin/phpunit

# Run specific test
vendor/bin/phpunit --filter testProjectCreation

# Run with coverage
vendor/bin/phpunit --coverage-html coverage/
```

Test structure:
- `tests/Unit/` - Unit tests for models, traits, services
- `tests/Feature/` - Integration tests for controllers, routes
- `tests/TestCase.php` - Base test case with helpers
