<?php

namespace Bongo\Project\Traits;

use Bongo\Project\Models\Project;
use Bongo\Project\Models\ProjectCategory;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Session;

trait HasRelated
{
    public function getRelated()
    {
        $this->loadMissing('categories');

        return Project::query()
            ->active()
            ->join('project_categories_pivot', 'projects.id', '=', 'project_categories_pivot.project_id')
            ->where('projects.id', '!=', $this->id)
            ->whereIn('project_category_id', $this->categories->pluck('id')->toArray());
    }

    public function getRelatedByRandom(int $limit = 4): Collection
    {
        $this->loadMissing('categories');

        // Get a list of active project ids, where they are within the same category as this project
        $relatedProjectIds = $this->getRelated()
            ->take($limit)
            ->inRandomOrder()
            ->pluck('projects.id');

        return Project::with('images')->find($relatedProjectIds);
    }

    public function getPrimaryCategory(): ?ProjectCategory
    {
        return $this->categories()->first();
    }

    public function getPrevious(): ?Project
    {
        $this->loadMissing('categories');

        $selectedCategory = Session::get('selected_project_category', $this->getPrimaryCategory());

        if ($selectedCategory) {
            return $selectedCategory
                ->projects()
                ->active()
                ->with('images')
                ->where('projects.id', '<', $this->id)
                ->orderBy('projects.id', 'DESC')
                ->first();
        }

        return null;
    }

    public function getNext(): ?Project
    {
        $this->loadMissing('categories');

        $selectedCategory = Session::get('selected_project_category', $this->getPrimaryCategory());

        if ($selectedCategory) {
            return $selectedCategory
                ->projects()
                ->active()
                ->with('images')
                ->where('projects.id', '>', $this->id)
                ->orderBy('projects.id', 'ASC')
                ->first();
        }

        return null;
    }
}
