# Bongo Project Package

A Laravel package providing a complete project portfolio management system with categories, image galleries, and WYSIWYG content editing capabilities.

[![Latest Version](https://img.shields.io/badge/version-3.0-blue.svg)](https://github.com/bongo/project)
[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-8892BF.svg)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-%3E%3D10.0-FF2D20.svg)](https://laravel.com)

## Features

- **Project Management**: Create and manage projects with rich content, categories, and metadata
- **Category System**: Organise projects with many-to-many category relationships
- **Image Integration**: Full image gallery support with featured images via `bongo/image`
- **Frontend Builder**: WYSIWYG content editor for authenticated users
- **SEO Optimisation**: Meta titles, descriptions, canonical URLs, and schema.org structured data
- **Status Management**: Pending, active, and inactive status with query scopes
- **Header Customisation**: Per-project transparent and sticky header options
- **Related Projects**: Automatic related projects based on shared categories
- **Navigation**: Previous/next project navigation within category context
- **Project Duplication**: Clone projects with all relationships and images
- **Event System**: Comprehensive event/listener architecture
- **API Support**: RESTful API endpoints with resource transformations
- **Datatable Integration**: Server-side datatables for admin listing
- **Soft Deletes**: Recoverable deletion with audit trail
- **Translations**: Multi-language support

## Requirements

- PHP 8.2 or higher
- Laravel 10 or higher
- `bongo/framework` ^3.0
- `bongo/image` ^3.0
- `bongo/menu` ^3.0
- `bongo/package` ^3.0
- `bongo/setting` ^3.0

## Installation

### Step 1: Install via Composer

Add the private repository to your `composer.json`:

```json
{
    "repositories": [
        {
            "type": "composer",
            "url": "https://designtecpackages.co.uk"
        }
    ]
}
```

Then require the package:

```bash
composer require bongo/project
```

### Step 2: Publish Configuration (Optional)

The package uses auto-discovery for Laravel 5.5+. Configuration is optional as defaults are provided.

```bash
php artisan vendor:publish --provider="Bongo\Project\ProjectServiceProvider" --tag="config"
```

### Step 3: Run Migrations

```bash
php artisan migrate
```

This creates:
- `projects` table
- `project_categories` table
- `project_categories_pivot` table

### Step 4: Seed Package Registration (Optional)

If you're using the `bongo/package` system:

```bash
php artisan db:seed --class="Bongo\Project\Seeders\PackageSeeder"
```

## Configuration

Configuration file: `config/project.php`

```php
return [
    'prefix' => 'projects',                    // Frontend/backend route prefix
    'category_prefix' => 'project-categories', // Category route prefix
];
```

## Usage

### Creating a Project

```php
use Bongo\Project\Models\Project;
use Bongo\Project\Events\ProjectCreated;

$project = Project::create([
    'name' => 'My Amazing Project',
    'content' => '<p>Project details and description</p>',
    'summary' => 'A brief overview of the project',
    'status' => Project::ACTIVE,
    'meta_title' => 'SEO Title',
    'meta_description' => 'SEO description for search engines',
]);

// Sync categories
$project->categories()->sync([1, 2, 3]);

// Fire event
event(new ProjectCreated($project));
```

### Querying Projects

```php
// Get active projects
$activeProjects = Project::active()->get();

// Get projects with relationships (avoid N+1)
$projects = Project::active()
    ->with('categories', 'images')
    ->orderBy('date', 'desc')
    ->get();

// Get projects by category
$category = ProjectCategory::where('slug', 'web-design')->first();
$projects = $category->projects()->active()->get();

// Get projects by status
$pending = Project::pending()->get();
$inactive = Project::inactive()->get();

// Include soft-deleted projects
$all = Project::withTrashed()->get();
```

### Working with Categories

```php
use Bongo\Project\Models\ProjectCategory;

// Create category
$category = ProjectCategory::create([
    'name' => 'Web Design',
    'content' => '<p>Our web design projects</p>',
    'status' => ProjectCategory::ACTIVE,
]);

// Get projects in category
$projects = $category->projects()->active()->get();

// Sync project categories
$project->categories()->sync([1, 2, 3]);
$project->categories()->attach(4);
$project->categories()->detach(2);
```

### Related Projects

```php
// Get 4 random related projects (same categories)
$related = $project->getRelatedByRandom(4);

// Navigate between projects in category
$previous = $project->getPrevious();
$next = $project->getNext();

// Get primary category
$primaryCategory = $project->getPrimaryCategory();
```

### Working with Images

```php
use Bongo\Image\Models\Image;

// Get featured image
$featuredImage = $project->getFeaturedImage();

// Get gallery images
$galleryImages = $project->getGalleryImages();

// Get all images
$allImages = $project->images;

// Attach image
$image = Image::find(1);
$project->images()->save($image);
```

### Duplicating Projects

```php
// Duplicate a project (creates copy with new UUID, randomised name)
$newProject = $project->duplicate();

// The duplicate includes:
// - All project fields (except unique identifiers)
// - Category relationships
// - Image relationships
// - Status set to 'pending'
// - New UUID for builder access
```

### SEO Metadata

```php
// Get SEO fields (with automatic fallbacks)
$metaTitle = $project->getMetaTitle();              // meta_title or name
$metaDescription = $project->getMetaDescription();  // meta_description or auto-generated from content
$metaCanonical = $project->getMetaCanonical();      // meta_canonical URL
$metaIndex = $project->getMetaIndex();              // 'index' or 'noindex'
```

### Frontend Builder

The package includes a WYSIWYG builder for authenticated users with the `employee` role.

**Access URL**: `/projects/edit/{uuid}`

```php
// Get builder URL
$builderUrl = route('frontend.project.edit', $project->uuid);

// In controller
use Bongo\Image\Services\BuilderService;

public function update(Request $request, $uuid)
{
    $project = Project::where('uuid', $uuid)->firstOrFail();
    $project->content = $request->get('html');

    // Process images: download remote, save local, update URLs
    $builderService = new BuilderService($project, '/projects/');
    $project->content = $builderService->process();
    $project->save();

    event(new ProjectUpdated($project));

    return redirect()
        ->route('frontend.project.edit', $project->uuid)
        ->success(trans('project::backend.update_success'));
}
```

### Events

The package dispatches events for all CRUD operations:

```php
use Bongo\Project\Events\ProjectCreated;
use Bongo\Project\Events\ProjectUpdated;
use Bongo\Project\Events\ProjectDeleted;
use Bongo\Project\Events\ProjectCategoryCreated;
use Bongo\Project\Events\ProjectCategoryUpdated;
use Bongo\Project\Events\ProjectCategoryDeleted;

// Events automatically fire listeners for:
// - Setting project date and user on creation
// - Clearing menu cache on updates/deletes
// - Updating sitemap on updates/deletes
```

### API Usage

The package includes API endpoints with resource transformations:

```php
// Get all active projects (requires auth:sanctum)
GET /api/projects

// Get all active categories
GET /api/project-categories

// Returns ProjectResource/ProjectCategoryResource collections
```

## Routes

### Backend Routes (Admin)

Prefix: `/admin/projects` and `/admin/project-categories`
Middleware: `auth`, `employee`

```
GET     /admin/projects                    backend.project.index
GET     /admin/projects/create             backend.project.create
POST    /admin/projects/store              backend.project.store
GET     /admin/projects/datatable          backend.project.datatable
GET     /admin/projects/{id}               backend.project.show
GET     /admin/projects/{id}/edit          backend.project.edit
POST    /admin/projects/{id}/update        backend.project.update
DELETE  /admin/projects/{id}/delete        backend.project.destroy
GET     /admin/projects/{id}/duplicate     backend.project.duplicate
POST    /admin/projects/{id}/image         backend.project.image

// Similar routes for project-categories
```

### Frontend Routes (Public)

Prefix: `/projects` and `/project-categories`
Middleware: `hasRedirects`, `hasShortCodes`, `minifyHtml` (public routes)

```
GET     /projects                          frontend.project.index
GET     /projects/{slug}                   frontend.project.show
GET     /projects/edit/{uuid}              frontend.project.edit (auth + employee required)
POST    /projects/update/{uuid}            frontend.project.update (auth + employee required)

GET     /project-categories                frontend.project_category.index
GET     /project-categories/{slug}         frontend.project_category.show
```

### API Routes

Prefix: `/api/projects` and `/api/project-categories`
Middleware: `auth:sanctum`

```
GET     /api/projects                      api.project.index
GET     /api/project-categories            api.project_category.index
```

## Views

The package provides Blade views for both backend and frontend:

### Backend Views
- `project::backend.index` - Datatable listing
- `project::backend.create` - Create form
- `project::backend.edit` - Edit form
- `project::backend.show` - Detail view
- `project::backend.category.*` - Category management views

### Frontend Views
- `project::frontend.show` - Project detail page
- `project::frontend.builder` - WYSIWYG builder interface
- `project::frontend.category.index` - Category listing
- `project::frontend.category.show` - Category detail with projects
- `project::frontend.partials.*` - Reusable components

You can override views by creating files in your application's `resources/views/vendor/project/` directory.

## Model Traits

### Project Model Uses:
- `HasContent` - Content/summary management
- `HasHeaderClass` - Header customisation (transparent/sticky)
- `HasImages` - Image relationships and helpers
- `HasKey` - Unique key generation
- `HasRelated` - Related projects logic
- `HasSeo` - SEO metadata management
- `HasStatus` - Status query scopes
- `HasUUID` - UUID generation
- `SoftDeletes` - Soft delete support

### Available Scopes:
```php
Project::active()      // where('status', 'active')
Project::inactive()    // where('status', 'inactive')
Project::pending()     // where('status', 'pending')
```

### Available Constants:
```php
Project::PENDING      // 'pending'
Project::ACTIVE       // 'active'
Project::INACTIVE     // 'inactive'

Project::INDEX        // 'index'
Project::NO_INDEX     // 'noindex'

Project::ENABLED      // 1
Project::DISABLED     // 0
```

## Testing

Run the test suite:

```bash
composer test
# or
vendor/bin/phpunit
```

Run code style checks:

```bash
composer pint
# or
vendor/bin/pint
```

Run static analysis:

```bash
composer analyse
# or
vendor/bin/phpstan analyse
```

## Documentation

- **ARCHITECTURE.md** - Detailed architecture documentation with diagrams, database schema, and lifecycle flows
- **CLAUDE.md** - Quick reference guide for Claude Code with commands and common tasks
- **.cursorrules** - Comprehensive coding conventions and patterns for Cursor AI
- **.github/copilot-instructions.md** - GitHub Copilot instructions with code templates

## Changelog

Please see [CHANGELOG.md](CHANGELOG.md) for information on what has changed recently.

## Contributing

This is a proprietary package for internal use. Please contact the development team for contribution guidelines.

## Security

If you discover any security-related issues, please email [stuart.elliott@bespokeuk.com](mailto:stuart.elliott@bespokeuk.com) instead of using the issue tracker.

## Credits

- [Stuart Elliott](https://github.com/stuarttodd-dev)
- [Bespoke UK](https://bespokeuk.com)
- [All Contributors](../../contributors)

## License

This package is proprietary software. Unauthorised copying, distribution, or use is strictly prohibited.

Copyright (c) 2021-2025 Bespoke UK Ltd.
