<?php

namespace Bongo\Project\Http\Controllers\Frontend;

use Bongo\Framework\Helpers\File;
use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Image\Models\Image;
use Bongo\Image\Services\BuilderService;
use Bongo\Project\Events\ProjectUpdated;
use Bongo\Project\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;

class ProjectController extends AbstractController
{
    public function index()
    {
        return redirect()->route('frontend.project_category.index');
    }

    public function show($slug)
    {
        $project = Project::where('slug', '=', $slug)->first();
        if (! $project || (! $project->isActive() && ! auth()->user())) {
            abort('404', 'Project could not be found');
        }

        // Set the selected category for related
        if (! Session::has('selected_project_category') && $primaryCategory = $project->getPrimaryCategory()) {
            Session::put('selected_project_category', $primaryCategory);
        }

        return view('project::frontend.show', compact('project'));
    }

    public function edit($uuid)
    {
        if (! $project = Project::where('uuid', '=', $uuid)->first()) {
            abort('404');
        }

        return view('project::frontend.builder', compact('project'));
    }

    public function image(Request $request, $uuid)
    {
        if (! $project = Project::where('uuid', '=', $uuid)->first()) {
            abort('404');
        }

        $file = $request->file('fileImage');
        $ext = $file->getClientOriginalExtension();
        $name = rtrim($file->getClientOriginalName(), $ext);
        $fileName = File::generateName($name, $ext);

        $filePath = config('image.public_path').'projects/'.$project->id.'/';

        if ($file->storePubliclyAs($filePath, $fileName)) {

            // Save the image to the database
            $image = new Image();
            $image->name = $fileName;
            $image->title = $project->name;
            $image->path = $filePath;
            $image->type = Image::WYSIWYG;
            $image->ext = $ext;
            $image->created_by = user() ? user()->id : null;
            $image->updated_by = user() ? user()->id : null;
            $project->images()->save($image);

            // Return the link to the file
            $fileLink = Storage::url($filePath.$fileName);

            return "<html><body onload=\"parent.applyLargerImage('".$fileLink."')\"></body></html>";
        }

        return "<html><body onload=\"alert('Sorry, your file was not uploaded.')\"></body></html>";
    }

    public function update(Request $request, $uuid)
    {
        if (! $project = Project::where('uuid', '=', $uuid)->first()) {
            abort('404');
        }

        // Get the html
        $project->content = $request->get('html');

        // Process any images then save
        $builderService = new BuilderService($project, '/projects/');
        $project->content = $builderService->process();
        $project->save();

        event(new ProjectUpdated($project));

        return redirect()
            ->route('frontend.project.edit', $project->uuid)
            ->success(trans('project::backend.update_success'));
    }
}
