# RealGreen Package - Claude Code Guide

## Overview

RealGreen API integration package for automatically exporting Bongo estimates to the RealGreen CRM system. This package listens for estimate updates and synchronises data to RealGreen's lead form API in real-time.

**Package**: `bongo/realgreen`
**Namespace**: `Bongo\RealGreen`
**Type**: Custom integration (Laravel package)
**PHP**: 8.2+ | **Laravel**: 10+

## Quick Links

- **Detailed Architecture**: See `ARCHITECTURE.md` for class diagrams, data flow, and extension points
- **Cursor AI Instructions**: See `.cursorrules` for development guidelines
- **GitHub Copilot Guide**: See `.github/copilot-instructions.md` for code templates

## Available Commands

Commands from `composer.json`:

```bash
# Run tests
composer test
vendor/bin/phpunit

# Run tests with coverage
composer test-coverage

# Code analysis with PHPStan
composer analyse
vendor/bin/phpstan analyse

# Format code with Laravel Pint
composer format
vendor/bin/pint

# Check code style without fixing
vendor/bin/pint --test

# Build and serve (testbench)
composer build
composer start

# Clear testbench cache
composer clear
```

### Artisan Commands

```bash
# Export non-exported estimates to RealGreen
php artisan realgreen:export_estimates

# Re-export all estimates (including previously exported)
php artisan realgreen:export_estimates --all
```

## Architecture Quick Reference

### Service Provider
Extends `Bongo\Framework\Providers\AbstractServiceProvider` with automatic bootstrapping:
- **Module**: `realgreen`
- **Config**: Auto-loads from `src/Config/realgreen.php`
- **Commands**: Registers `ExportEstimatesToRealGreen`
- **Listeners**: Registers `ExportEstimateToRealGreen` for `EstimateUpdated` event

### Data Flow

```
Estimate Updated → EstimateUpdated Event → ExportEstimateToRealGreen Listener
                                                      ↓
                                              LeadFormData (DTO)
                                                      ↓
                                              LeadForm Service
                                                      ↓
                                          RealGreen API (POST /LeadForm/Submit)
                                                      ↓
                                    Update estimate (exported_at, export_error)
```

### Export Conditions

Automatic export occurs when:
1. `app()->isProduction()` returns true
2. `setting('system::credentials.real_green_api_key')` is set
3. Estimate status is not `DRAFT`

## Key Files

| File | Purpose | Key Methods/Properties |
|------|---------|----------------------|
| `src/RealGreenServiceProvider.php` | Package registration | `$module = 'realgreen'`<br>`$commands`, `$listeners` |
| `src/Commands/ExportEstimatesToRealGreen.php` | CLI batch export | `handle(): void` |
| `src/Listeners/ExportEstimateToRealGreen.php` | Event-driven export | `handle($event): void` |
| `src/Services/LeadForm.php` | API client for lead forms | `export(Estimate): ?array` |
| `src/Services/SourceCode.php` | Marketing source codes | `all(): Collection`<br>`byAbbreviation(string): ?array` |
| `src/Data/LeadFormData.php` | DTO for API payload | `fromModel(Estimate): self`<br>`toArray(): array` |
| `src/Config/realgreen.php` | Configuration | `api_url`, `source_code` |
| `src/Seeders/PackageSeeder.php` | Package database record | Seeds `real_green` package |

## Configuration

### Config Keys (`config('realgreen.*')`)

```php
'api_url'     => 'https://saapi.realgreen.com'  // RealGreen API base URL
'source_code' => 28                              // Marketing source (28 = GOO/Google)
```

### Environment Variables

```bash
REALGREEN_API_URL=https://saapi.realgreen.com  # Optional, has default
```

### System Settings (Required)

```php
setting('system::credentials.real_green_api_key')  // RealGreen API authentication key
```

### Fixed Values

```php
'employeeID'    => 'JCHAPMAN'  // Fixed employee for all submissions
'actionReasonID' => 18          // Fixed action reason (18 = Note in RealGreen)
'countryCode'   => 'GB'         // Fixed country code
```

## Field Mapping Summary

| Bongo Field | RealGreen Field | Transformation |
|------------|----------------|---------------|
| `number` | `name` | Add prefix + optional name |
| `postcode` | `zipcode`, `zip` | Remove spaces |
| `line_1` + `line_2` | `streetNumberAndName` | Concatenate |
| `line_1` | `houseNumber` | Direct |
| `line_2` | `streetName` | Direct |
| `line_3` | `addressLine2` | Direct |
| `city` | `city` | Direct |
| `county` | `state` | Direct |
| `email` | `emailAddress` | Direct |
| `phone` | `cellPhoneNumber` | Direct |
| `phone` (non-07) | `homePhoneNumber` | Conditional |
| `first_name` | `firstName` | Direct |
| `last_name` | `lastName` | Direct |
| `latitude` | `latitude` | Direct |
| `longitude` | `longitude` | Direct |
| Areas, totals | `callLogNotes` | Formatted text |

**Fallback Strategy**:
- Required fields → `'-- --'` when empty
- Optional fields → `null` when empty

## Code Style Summary

### Strict Types
All files use `declare(strict_types=1);`

### Return Types
All methods have explicit return type declarations:
```php
public function export(Estimate $estimate): ?array
public function handle(): void
protected function setField(Estimate $estimate): void
```

### Exception Handling
Document thrown exceptions:
```php
/** @throws RequestException|BindingResolutionException */
```

### Laravel Pint Rules
```json
{
    "preset": "laravel",
    "rules": {
        "new_with_parentheses": false,
        "nullable_type_declaration_for_default_null_value": false,
        "declare_strict_types": true,
        "declare_parentheses": true
    }
}
```

## Common Operations

### Adding a New Field to Export

1. Add property to `LeadFormData`:
```php
public ?string $newField;
```

2. Create setter:
```php
protected function setNewField(Estimate $estimate): void
{
    $this->newField = $estimate->new_field ?? null;
}
```

3. Call in `fromModel()`:
```php
$this->setNewField($estimate);
```

4. Include in `toArray()`:
```php
'newField' => $this->newField,
```

### Adding a New Command

1. Create command class in `src/Commands/`
2. Register in `RealGreenServiceProvider`:
```php
protected array $commands = [
    ExportEstimatesToRealGreen::class,
    NewCommand::class, // Add here
];
```

### Adding a New Service

1. Create service class in `src/Services/`:
```php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Services;

use Illuminate\Support\Facades\Http;

class NewService
{
    public function call(): ?array
    {
        return Http::withoutVerifying()
            ->baseUrl(config('realgreen.api_url'))
            ->withHeaders(['apiKey' => setting('system::credentials.real_green_api_key')])
            ->get('/endpoint')
            ->throw()
            ->json();
    }
}
```

### Testing Export

```bash
# Test in development
php artisan realgreen:export_estimates

# Check for errors in estimate records
SELECT id, number, exported_at, export_error FROM estimates WHERE export_error IS NOT NULL;
```

### Debugging

Enable logging in `LeadForm::export()`:
```php
\Log::debug('RealGreen Export', $leadFormData);
```

Check Laravel logs:
```bash
tail -f storage/logs/laravel.log
```

## Dependencies

### Core Dependencies
- `bongo/framework` (^3.0): AbstractServiceProvider, helper functions
- `bongo/estimate`: Estimate model, EstimateUpdated event
- `bongo/package`: Package model, SeedsPackage trait
- `illuminate/contracts` (^10.0): Laravel framework

### Helper Functions
```php
setting($key)              // Get system setting
log_exception($exception)  // Log exception to Laravel log
console_print($message)    // Output to console
config($key)               // Get configuration value
```

## API Reference

### RealGreen API Endpoints

**Base URL**: `https://saapi.realgreen.com`
**Authentication**: Header `apiKey`

| Method | Endpoint | Purpose | Service Method |
|--------|----------|---------|---------------|
| POST | `/LeadForm/Submit` | Submit lead form | `LeadForm::export()` |
| GET | `/SourceCode` | Get source codes | `SourceCode::all()` |

### Request Headers
```php
[
    'apiKey' => setting('system::credentials.real_green_api_key')
]
```

### Example Payload
See `src/Schema/lead_form.json` for complete example:
```json
{
    "name": "KLC-8137 - Chloe West",
    "zipcode": "LE675FX",
    "streetNumberAndName": "28 Elsdon Close",
    "emailAddress": "email@example.com",
    "cellPhoneNumber": "+447943707718",
    "sourceCode": 28,
    "firstName": "Chloe",
    "lastName": "West",
    "countryOrRegionCode": "GB",
    "callLogNotes": "Total Area(m2): 29.55...",
    "employeeID": "JCHAPMAN",
    "actionReasonID": 18
}
```

## Error Handling

### Silent Failure Pattern
The listener doesn't throw exceptions to prevent blocking estimate updates:

```php
try {
    // Export logic
    $estimate->exported_at = now();
    $estimate->export_error = null;
    $estimate->save();
} catch (RequestException|BindingResolutionException $e) {
    $estimate->export_error = $e->getMessage();
    $estimate->save();

    log_exception($e);
    console_print($e->getMessage());
    // Don't rethrow
}
```

### Checking for Errors

```php
// In code
$estimate = Estimate::find($id);
if ($estimate->export_error) {
    // Handle error, possibly retry
}

// In CLI
$failedEstimates = Estimate::whereNotNull('export_error')->get();
```

### Retry Logic
Currently no automatic retry. Can be added by:
1. Creating a scheduled command to retry failed exports
2. Filtering estimates with `export_error IS NOT NULL`
3. Clearing `export_error` before retry

## Security Notes

### API Key Storage
- Never commit API keys
- Store in system settings table
- Access via `setting()` helper

### SSL Verification
Current implementation disables SSL verification (`withoutVerifying()`):
- Review for production use
- Consider enabling for security
- May be required for staging environments

### Data Sanitisation
Basic sanitisation provided:
- Postcode space removal
- Phone format checking (07 prefix)
- Fallback values for required fields

Consider adding:
- Email validation
- Phone number normalisation
- HTML sanitisation for notes

## Troubleshooting

### Exports Not Working

Check list:
1. ✓ Production mode: `app()->isProduction()`
2. ✓ API key set: `setting('system::credentials.real_green_api_key')`
3. ✓ Estimate not DRAFT: `$estimate->status !== Estimate::DRAFT`
4. ✓ Check `export_error` field in database
5. ✓ Check Laravel logs: `storage/logs/laravel.log`

### Testing in Development

Temporarily bypass production check:
```php
// In ExportEstimateToRealGreen.php
if (/* app()->isProduction() */ true  // Force enable for testing
    && !empty(setting('system::credentials.real_green_api_key'))
    && $event->estimate
    && $event->estimate->status !== Estimate::DRAFT
) {
    // Export logic
}
```

### API Connection Issues

Test API connectivity:
```php
use Illuminate\Support\Facades\Http;

$response = Http::withoutVerifying()
    ->baseUrl(config('realgreen.api_url'))
    ->withHeaders(['apiKey' => setting('system::credentials.real_green_api_key')])
    ->get('/SourceCode');

dd($response->status(), $response->json());
```

## Related Packages

- `bongo/framework`: Core framework and service provider base
- `bongo/estimate`: Estimate model and events
- `bongo/package`: Package management system

## Version History

Current version: 3.0 (matches framework dependency)

For detailed changelog, see `CHANGELOG.md` (if available) or git history:
```bash
git log --oneline
```
