<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Commands;

use Bongo\Estimate\Models\Estimate;
use Bongo\RealGreen\Services\LeadForm;
use Illuminate\Console\Command;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\Client\RequestException;

class ExportEstimatesToRealGreen extends Command
{
    /** @var string */
    protected $signature = 'realgreen:export_estimates {--all}';

    /** @var string */
    protected $description = 'Export estimates to RealGreen';

    /** @throws BindingResolutionException */
    public function handle(): void
    {
        if (empty(setting('system::credentials.real_green_api_key'))) {
            $this->error('-- Real Green API key is not set, exiting...');

            return;
        }

        $query = Estimate::query()
            ->whereNot('status', Estimate::DRAFT)
            ->orderBy('created_at');

        // If this is not a full export, then just get the records that have not been exported
        if (! $this->option('all')) {
            $query->whereNull('exported_at');
        }

        // Get the estimates or bail if there are none
        $estimates = $query->get();
        if (! $estimates || count($estimates) < 1) {
            $this->warn('-- No estimates to export, exiting...');

            return;
        }

        // Loop through each estimate
        foreach ($estimates as $estimate) {
            try {
                $this->info("-- Exporting estimate id:{$estimate->id} number:{$estimate->number}");

                // And export to real green
                (new LeadForm())->export($estimate);
                $estimate->exported_at = now();
                $estimate->export_error = null;
                $estimate->save();

            } catch (RequestException $e) {
                $estimate->export_error = $e->getMessage();
                $estimate->save();

                log_exception($e);
                console_print($e->getMessage());

                continue;
            }
        }
    }
}
