<?php

namespace Bongo\RealGreen\Data;

use Bongo\Estimate\Models\Estimate;
use Bongo\Framework\Helpers\Str as StrHelper;
use Illuminate\Support\Str;

class LeadFormData
{
    public ?string $name;
    public ?string $zipcode;
    public ?string $streetNumberAndName;
    public ?string $emailAddress;
    public ?string $homePhoneNumber;
    public ?string $cellPhoneNumber;
    public ?int $sourceCode;
    public ?string $firstName;
    public ?string $lastName;

    public ?string $houseNumber;
    public ?string $streetName;
    public ?string $streetSuffix;
    public ?string $addressLine2;
    public ?string $city;
    public ?string $state;
    public ?string $zip;
    public ?string $countryCode;
    public ?float $latitude;
    public ?float $longitude;

    public ?string $countryOrRegionCode;
    public ?string $callLogNotes;

    public function fromModel(Estimate $estimate): self
    {
        $this->setName($estimate);
        $this->setZipcode($estimate);
        $this->setStreetNumberAndName($estimate);
        $this->setEmailAddress($estimate);
        $this->setHomePhoneNumber($estimate);
        $this->setCellPhoneNumber($estimate);
        $this->setSourceCode();
        $this->setFirstName($estimate);
        $this->setLastName($estimate);

        $this->setHouseNumber($estimate);
        $this->setStreetName($estimate);
        $this->setAddressLine2($estimate);
        $this->setCity($estimate);
        $this->setState($estimate);
        $this->setZip($estimate);
        $this->setCountryCode();
        $this->setLatitude($estimate);
        $this->setLongitude($estimate);

        $this->setCountryOrRegionCode();
        $this->setCallLogNotes($estimate);

        return $this;
    }

    public function toArray(): array
    {
        return [
            "name" => $this->name,
            "zipcode" => $this->zipcode,
            "streetNumberAndName" => $this->streetNumberAndName,
            "emailAddress" => $this->emailAddress,
            "cellPhoneNumber" => $this->cellPhoneNumber,
            "sourceCode" => $this->sourceCode,
            "firstName" => $this->firstName,
            "lastName" => $this->lastName,
            "countryOrRegionCode" => $this->countryOrRegionCode,
            "callLogNotes" => $this->callLogNotes,
            'employeeID' => 'JCHAPMAN',
            'actionReasonID' => 18, // 18 is the ID for 'Note' in RealGreen
        ];
    }

    protected function setName(Estimate $estimate): void
    {
        $this->name = config('estimate.number_prefix').$estimate->number;

        if (! empty($estimate->first_name) && ! empty($estimate->last_name)) {
            $this->name .= " - {$estimate->first_name} {$estimate->last_name}";
        }
    }

    protected function setZipcode(Estimate $estimate): void
    {
        $this->zipcode = ! empty($estimate->postcode)
            ? StrHelper::noSpace($estimate->postcode)
            : '-- --';
    }

    protected function setStreetNumberAndName(Estimate $estimate): void
    {
        $streetNumberAndName = '';
        if (! empty($estimate->line_1)) {
            $streetNumberAndName .= "{$estimate->line_1} ";
        }
        if (! empty($estimate->line_2)) {
            $streetNumberAndName .= "{$estimate->line_2}";
        }

        $this->streetNumberAndName = ! empty($streetNumberAndName)
            ? $streetNumberAndName
            : '-- --';
    }

    protected function setEmailAddress(Estimate $estimate): void
    {
        $this->emailAddress = ! empty($estimate->email)
            ? $estimate->email
            : null;
    }

    protected function setHomePhoneNumber(Estimate $estimate): void
    {
        $this->homePhoneNumber = ! empty($estimate->phone) && ! Str::startsWith($estimate->phone, '07')
            ? $estimate->phone
            : null;
    }

    protected function setCellPhoneNumber(Estimate $estimate): void
    {
        $this->cellPhoneNumber = ! empty($estimate->phone)
            ? $estimate->phone
            : null;
    }

    protected function setSourceCode(): void
    {
        $this->sourceCode = config('realgreen.source_code');
    }

    protected function setFirstName(Estimate $estimate): void
    {
        $this->firstName = ! empty($estimate->first_name)
            ? $estimate->first_name
            : null;
    }

    protected function setLastName(Estimate $estimate): void
    {
        $this->lastName = ! empty($estimate->last_name)
            ? $estimate->last_name
            : null;
    }

    protected function setHouseNumber(Estimate $estimate): void
    {
        $this->houseNumber = ! empty($estimate->line_1)
            ? $estimate->line_1
            : '-- --';
    }

    protected function setStreetName(Estimate $estimate): void
    {
        $this->streetName = ! empty($estimate->line_2)
            ? $estimate->line_2
            : '-- --';
    }

    protected function setAddressLine2(Estimate $estimate): void
    {
        $this->addressLine2 = ! empty($estimate->line_3)
            ? $estimate->line_3
            : '-- --';
    }

    protected function setCity(Estimate $estimate): void
    {
        $this->city = ! empty($estimate->city)
            ? $estimate->city
            : '-- --';
    }

    protected function setState(Estimate $estimate): void
    {
        $this->state = ! empty($estimate->county)
            ? $estimate->county
            : '-- --';
    }

    protected function setZip(Estimate $estimate): void
    {
        $this->zip = ! empty($estimate->postcode)
            ? StrHelper::noSpace($estimate->postcode)
            : '-- --';
    }

    protected function setCountryCode(): void
    {
        $this->countryCode = 'GB';
    }

    protected function setLatitude(Estimate $estimate): void
    {
        $this->latitude = ! empty($estimate->latitude)
            ? $estimate->latitude
            : null;
    }

    protected function setLongitude(Estimate $estimate): void
    {
        $this->longitude = ! empty($estimate->longitude)
            ? $estimate->longitude
            : null;
    }

    protected function setCountryOrRegionCode(): void
    {
        $this->countryOrRegionCode = 'GB';
    }

    protected function setCallLogNotes(Estimate $estimate): void
    {
        $this->callLogNotes = '';

        if (count($estimate->getAreas()) > 1) {
            $this->callLogNotes .= "Areas(m2):\n";
            foreach ($estimate->getAreas() as $area) {
                $numberOfM2 = $area->no_of_m2 ? number_format($area->no_of_m2, 2) : 0;
                $this->callLogNotes .= "{$area->label} - {$numberOfM2}\n";
            }
            $this->callLogNotes .= "\n";
        }

        if (! empty($estimate->address)) {
            $totalAreaM2 = $estimate->total_area_m2 ? number_format($estimate->total_area_m2, 2) : 0;
            $this->callLogNotes .= "Total Area(m2): {$totalAreaM2}\n";
        }
        if (! empty($estimate->subtotal)) {
            $subtotal = $estimate->subtotal ? number_format($estimate->subtotal, 2) : 0;
            $this->callLogNotes .= "Subtotal: {$subtotal}\n";
        }
        if (! empty($estimate->tax)) {
            $tax = $estimate->tax ? number_format($estimate->tax, 2) : 0;
            $this->callLogNotes .= "VAT: {$tax}\n";
        }
        if (! empty($estimate->total)) {
            $total = $estimate->total ? number_format($estimate->total, 2) : 0;
            $this->callLogNotes .= "Total: {$total}\n";
        }
        $this->callLogNotes .= "\n";
        $this->callLogNotes .= "Price per treatment, of which there are ";
        $this->callLogNotes .= config('estimate.chargeable_treatments_per_year');
        $this->callLogNotes .= " chargeable treatments per year.\n";

        $this->callLogNotes .= "\n";
        $this->callLogNotes .= "Via monthly instalments = £";
        $this->callLogNotes .= ! empty($estimate->per_month) ? number_format($estimate->per_month, 2) : '0.00';
        $this->callLogNotes .= " per month\n";
    }
}
