# GitHub Copilot Instructions for RealGreen Package

## Project Overview
This Laravel package integrates with RealGreen CRM API to automatically export estimate data. It listens for estimate updates and synchronises them to RealGreen's lead form system.

**Namespace**: `Bongo\RealGreen`
**Type**: Custom integration package
**Framework**: Laravel 10+, PHP 8.2+

## Key Classes and Relationships

```
RealGreenServiceProvider
├── Commands
│   └── ExportEstimatesToRealGreen (CLI batch export)
├── Listeners
│   └── ExportEstimateToRealGreen (Event-driven export)
└── Services
    ├── LeadForm (HTTP communication)
    └── SourceCode (Marketing codes)

Data Flow:
EstimateUpdated → ExportEstimateToRealGreen → LeadForm → RealGreen API
                          ↓
                   LeadFormData (DTO)
```

### Class Responsibilities

**RealGreenServiceProvider** (src/RealGreenServiceProvider.php)
- Extends: `Bongo\Framework\Providers\AbstractServiceProvider`
- Module: `'realgreen'`
- Registers: Commands and event listeners
- Auto-loads: Config from `src/Config/realgreen.php`

**ExportEstimatesToRealGreen** (src/Commands/ExportEstimatesToRealGreen.php)
- Command signature: `realgreen:export_estimates {--all}`
- Purpose: Batch export estimates to RealGreen
- Updates: `exported_at` timestamp and `export_error` field

**ExportEstimateToRealGreen** (src/Listeners/ExportEstimateToRealGreen.php)
- Listens to: `Bongo\Estimate\Events\EstimateUpdated`
- Conditions: Production mode, API key set, non-DRAFT status
- Error handling: Silent failure with error logging

**LeadFormData** (src/Data/LeadFormData.php)
- Type: Data Transfer Object
- Purpose: Transform Bongo estimate to RealGreen format
- Methods: `fromModel()`, `toArray()`

**LeadForm** (src/Services/LeadForm.php)
- Purpose: HTTP client for lead form submission
- Endpoint: `POST /LeadForm/Submit`
- Auth: Header-based API key

**SourceCode** (src/Services/SourceCode.php)
- Purpose: Retrieve marketing source codes
- Endpoint: `GET /SourceCode`
- Methods: `all()`, `byAbbreviation()`

## Code Style Templates

### Service Method Template
```php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Services;

use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Facades\Http;

class ExampleService
{
    /** @throws RequestException */
    public function callApi(array $data): ?array
    {
        return Http::withoutVerifying()
            ->baseUrl(config('realgreen.api_url'))
            ->withHeaders(['apiKey' => setting('system::credentials.real_green_api_key')])
            ->post('/endpoint', $data)
            ->throw()
            ->json();
    }
}
```

### Event Listener Template
```php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Listeners;

use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\Client\RequestException;

class ExampleListener
{
    /** @throws BindingResolutionException */
    public function handle($event): void
    {
        if (app()->isProduction()
            && !empty(setting('system::credentials.real_green_api_key'))
            && $event->data
        ) {
            try {
                // Perform action
            } catch (RequestException|BindingResolutionException $e) {
                log_exception($e);
                console_print($e->getMessage());
            }
        }
    }
}
```

### Command Template
```php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Commands;

use Illuminate\Console\Command;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\Client\RequestException;

class ExampleCommand extends Command
{
    /** @var string */
    protected $signature = 'realgreen:example {--option}';

    /** @var string */
    protected $description = 'Example command description';

    /** @throws BindingResolutionException */
    public function handle(): void
    {
        // Check prerequisites
        if (empty(setting('system::credentials.real_green_api_key'))) {
            $this->error('-- API key not set, exiting...');
            return;
        }

        // Process items
        $items = $this->getItems();
        foreach ($items as $item) {
            try {
                $this->info("-- Processing item: {$item->id}");
                // Process item
            } catch (RequestException $e) {
                log_exception($e);
                console_print($e->getMessage());
                continue;
            }
        }
    }
}
```

### Data Transfer Object Template
```php
<?php

namespace Bongo\RealGreen\Data;

use Bongo\Estimate\Models\Estimate;

class ExampleData
{
    public ?string $field1;
    public ?int $field2;

    public function fromModel(Estimate $estimate): self
    {
        $this->setField1($estimate);
        $this->setField2($estimate);

        return $this;
    }

    public function toArray(): array
    {
        return [
            'field1' => $this->field1,
            'field2' => $this->field2,
        ];
    }

    protected function setField1(Estimate $estimate): void
    {
        $this->field1 = !empty($estimate->field1)
            ? $estimate->field1
            : null;
    }

    protected function setField2(Estimate $estimate): void
    {
        $this->field2 = $estimate->field2 ?? null;
    }
}
```

## Common Patterns

### Configuration Access
```php
// API URL
config('realgreen.api_url')

// Source code
config('realgreen.source_code')

// API key (via system settings)
setting('system::credentials.real_green_api_key')
```

### HTTP Request Pattern
```php
use Illuminate\Support\Facades\Http;

$response = Http::withoutVerifying()
    ->baseUrl(config('realgreen.api_url'))
    ->withHeaders(['apiKey' => setting('system::credentials.real_green_api_key')])
    ->post('/endpoint', $data)
    ->throw()  // Throws RequestException on HTTP errors
    ->json();
```

### Error Handling Pattern
```php
try {
    // API call or processing
    $result = $service->export($data);
    $model->exported_at = now();
    $model->export_error = null;
    $model->save();
} catch (RequestException|BindingResolutionException $e) {
    $model->export_error = $e->getMessage();
    $model->save();

    log_exception($e);
    console_print($e->getMessage());
}
```

### Export Condition Check
```php
if (app()->isProduction()
    && !empty(setting('system::credentials.real_green_api_key'))
    && $estimate->status !== Estimate::DRAFT
) {
    // Perform export
}
```

### Fallback Value Pattern
```php
// For required fields, use placeholder
$this->zipcode = !empty($estimate->postcode)
    ? StrHelper::noSpace($estimate->postcode)
    : '-- --';

// For optional fields, use null
$this->emailAddress = !empty($estimate->email)
    ? $estimate->email
    : null;
```

## Important Notes

### Strict Types
All files use `declare(strict_types=1);` at the top

### Return Type Declarations
Always include return types on methods:
```php
public function export(Estimate $estimate): ?array
public function handle(): void
protected function setField(Estimate $estimate): void
```

### Exception Documentation
Document thrown exceptions in PHPDoc:
```php
/** @throws RequestException|BindingResolutionException */
```

### Property Visibility
- Command properties: `protected`
- DTO properties: `public`
- Service methods: `public`
- DTO setter methods: `protected`

### Naming Conventions
- Commands: `{Verb}{Noun}{ToRealGreen}` (e.g., `ExportEstimatesToRealGreen`)
- Listeners: `{Verb}{Noun}{ToRealGreen}` (e.g., `ExportEstimateToRealGreen`)
- Services: `{Noun}` (e.g., `LeadForm`, `SourceCode`)
- Data objects: `{Noun}Data` (e.g., `LeadFormData`)

### Helper Functions
Available helper functions:
- `setting($key)`: Get system setting
- `log_exception($exception)`: Log exception
- `console_print($message)`: Print to console
- `config($key)`: Get configuration value

### Service Provider Extension
The package extends `Bongo\Framework\Providers\AbstractServiceProvider` which automatically:
- Loads config from `src/Config/{module}.php`
- Registers commands from `$commands` array
- Registers listeners from `$listeners` array
- Requires `protected string $module` property

## Testing Hints

### Testing Commands
```php
use Bongo\RealGreen\Tests\TestCase;

class ExportEstimatesToRealGreenTest extends TestCase
{
    public function test_command_requires_api_key(): void
    {
        $this->artisan('realgreen:export_estimates')
            ->expectsOutput('-- Real Green API key is not set, exiting...')
            ->assertExitCode(0);
    }
}
```

### Mocking HTTP Requests
```php
use Illuminate\Support\Facades\Http;

Http::fake([
    '*/LeadForm/Submit' => Http::response(['success' => true], 200),
]);
```

### Testing Event Listeners
```php
use Bongo\Estimate\Events\EstimateUpdated;

Event::fake([EstimateUpdated::class]);

// Trigger event
EstimateUpdated::dispatch($estimate);

Event::assertListening(
    EstimateUpdated::class,
    ExportEstimateToRealGreen::class
);
```

## Field Mapping Reference

| Bongo Estimate Field | RealGreen Field | Transformation |
|---------------------|----------------|----------------|
| `number` (prefixed) | `name` | Add estimate number prefix |
| `first_name`, `last_name` | `name` | Append to name if present |
| `postcode` | `zipcode`, `zip` | Remove spaces |
| `line_1`, `line_2` | `streetNumberAndName` | Concatenate |
| `line_1` | `houseNumber` | Direct map |
| `line_2` | `streetName` | Direct map |
| `line_3` | `addressLine2` | Direct map |
| `city` | `city` | Direct map |
| `county` | `state` | Direct map |
| `email` | `emailAddress` | Direct map |
| `phone` | `cellPhoneNumber` | Direct map |
| `phone` (non-07) | `homePhoneNumber` | Conditional |
| `first_name` | `firstName` | Direct map |
| `last_name` | `lastName` | Direct map |
| `latitude` | `latitude` | Direct map |
| `longitude` | `longitude` | Direct map |
| Config | `sourceCode` | From config |
| Fixed | `countryCode`, `countryOrRegionCode` | Always 'GB' |
| Calculated | `callLogNotes` | Format areas, totals, payments |
| Fixed | `employeeID` | Always 'JCHAPMAN' |
| Fixed | `actionReasonID` | Always 18 (Note) |

## Configuration Values

```php
// Default API URL
'api_url' => 'https://saapi.realgreen.com'

// Default source code (28 = GOO/Google)
'source_code' => 28

// Fixed employee ID
'employeeID' => 'JCHAPMAN'

// Fixed action reason (Note type)
'actionReasonID' => 18
```
