# RealGreen Package Architecture

## Table of Contents
- [Overview](#overview)
- [Directory Structure](#directory-structure)
- [Class Diagrams](#class-diagrams)
- [Data Flow](#data-flow)
- [Configuration](#configuration)
- [Extension Points](#extension-points)
- [Adding New Features](#adding-new-features)

## Overview

The RealGreen package provides seamless integration between Bongo's estimate system and RealGreen's CRM platform. It automatically exports estimate data to RealGreen's API when estimates are created or updated.

**Key Design Principles**:
- Event-driven architecture for real-time synchronisation
- Data Transfer Object pattern for API payload construction
- Service layer for HTTP communication
- Silent failure to prevent blocking main application flow
- Idempotent operations (safe to retry)

## Directory Structure

```
src/
├── Commands/
│   └── ExportEstimatesToRealGreen.php
│       Purpose: CLI command for batch exports
│       Signature: realgreen:export_estimates {--all}
│       Updates: exported_at, export_error fields
│
├── Config/
│   └── realgreen.php
│       Purpose: Package configuration
│       Keys: api_url, source_code
│       Loaded by: AbstractServiceProvider
│
├── Data/
│   └── LeadFormData.php
│       Purpose: DTO for RealGreen lead form structure
│       Pattern: Builder pattern with setter methods
│       Methods: fromModel(), toArray()
│
├── Listeners/
│   └── ExportEstimateToRealGreen.php
│       Purpose: Event listener for real-time exports
│       Trigger: EstimateUpdated event
│       Conditions: Production, API key set, non-DRAFT
│
├── Schema/
│   └── lead_form.json
│       Purpose: Example payload for documentation
│       Used by: Developers for reference
│
├── Seeders/
│   └── PackageSeeder.php
│       Purpose: Register package in system
│       Creates: Package record with key 'real_green'
│
├── Services/
│   ├── LeadForm.php
│   │   Purpose: HTTP client for lead form submission
│   │   Endpoint: POST /LeadForm/Submit
│   │   Method: export(Estimate): ?array
│   │
│   └── SourceCode.php
│       Purpose: Retrieve marketing source codes
│       Endpoint: GET /SourceCode
│       Methods: all(), byAbbreviation()
│
└── RealGreenServiceProvider.php
    Purpose: Package bootstrapping
    Extends: Bongo\Framework\Providers\AbstractServiceProvider
    Registers: Commands, listeners
    Module: 'realgreen'
```

## Class Diagrams

### Service Provider Registration
```
┌─────────────────────────────────────────┐
│  AbstractServiceProvider                │
│  (from bongo/framework)                 │
│                                         │
│  + boot(): void                         │
│  # registerConfig(): void               │
│  # registerCommands(): void             │
│  # registerListeners(): void            │
└─────────────────┬───────────────────────┘
                  │ extends
                  │
┌─────────────────▼───────────────────────┐
│  RealGreenServiceProvider               │
│                                         │
│  # $module = 'realgreen'                │
│  # $commands = [                        │
│      ExportEstimatesToRealGreen::class  │
│  ]                                      │
│  # $listeners = [                       │
│      EstimateUpdated::class => [        │
│          ExportEstimateToRealGreen      │
│      ]                                  │
│  ]                                      │
└─────────────────────────────────────────┘
```

### Export Flow Classes
```
┌────────────────────────┐
│  ExportEstimateToRealGreen  │
│  (Listener)                 │
│                             │
│  + handle($event): void     │
└────────┬────────────────────┘
         │ uses
         │
┌────────▼────────────────┐       ┌──────────────────────┐
│  LeadForm               │       │  LeadFormData        │
│  (Service)              │◄──────│  (DTO)               │
│                         │ uses  │                      │
│  + export(Estimate)     │       │  + fromModel()       │
│    : ?array             │       │  + toArray(): array  │
└────────┬────────────────┘       └──────────────────────┘
         │ calls
         │
┌────────▼─────────────────────────────────┐
│  RealGreen API                           │
│  https://saapi.realgreen.com             │
│                                          │
│  POST /LeadForm/Submit                   │
│  Auth: apiKey header                     │
└──────────────────────────────────────────┘
```

### Data Transfer Object Structure
```
┌────────────────────────────────────────────┐
│  LeadFormData                              │
│                                            │
│  Public Properties:                        │
│  + name: ?string                           │
│  + zipcode: ?string                        │
│  + streetNumberAndName: ?string            │
│  + emailAddress: ?string                   │
│  + homePhoneNumber: ?string                │
│  + cellPhoneNumber: ?string                │
│  + sourceCode: ?int                        │
│  + firstName: ?string                      │
│  + lastName: ?string                       │
│  + houseNumber: ?string                    │
│  + streetName: ?string                     │
│  + streetSuffix: ?string                   │
│  + addressLine2: ?string                   │
│  + city: ?string                           │
│  + state: ?string                          │
│  + zip: ?string                            │
│  + countryCode: ?string                    │
│  + latitude: ?float                        │
│  + longitude: ?float                       │
│  + countryOrRegionCode: ?string            │
│  + callLogNotes: ?string                   │
│                                            │
│  Public Methods:                           │
│  + fromModel(Estimate): self               │
│  + toArray(): array                        │
│                                            │
│  Protected Setters:                        │
│  # setName(Estimate): void                 │
│  # setZipcode(Estimate): void              │
│  # setStreetNumberAndName(Estimate): void  │
│  # setEmailAddress(Estimate): void         │
│  # setHomePhoneNumber(Estimate): void      │
│  # setCellPhoneNumber(Estimate): void      │
│  # setSourceCode(): void                   │
│  # setFirstName(Estimate): void            │
│  # setLastName(Estimate): void             │
│  # setHouseNumber(Estimate): void          │
│  # setStreetName(Estimate): void           │
│  # setAddressLine2(Estimate): void         │
│  # setCity(Estimate): void                 │
│  # setState(Estimate): void                │
│  # setZip(Estimate): void                  │
│  # setCountryCode(): void                  │
│  # setLatitude(Estimate): void             │
│  # setLongitude(Estimate): void            │
│  # setCountryOrRegionCode(): void          │
│  # setCallLogNotes(Estimate): void         │
└────────────────────────────────────────────┘
```

## Data Flow

### Real-Time Export Flow (Event-Driven)
```
┌──────────────┐
│   Estimate   │
│   Updated    │
└──────┬───────┘
       │
       │ Fires
       │
┌──────▼──────────────┐
│  EstimateUpdated    │
│  Event              │
└──────┬──────────────┘
       │
       │ Triggers
       │
┌──────▼──────────────────────────────────────┐
│  ExportEstimateToRealGreen Listener         │
│                                             │
│  Checks:                                    │
│  ✓ app()->isProduction()                    │
│  ✓ API key configured                       │
│  ✓ Estimate not DRAFT                       │
└──────┬──────────────────────────────────────┘
       │
       │ Creates
       │
┌──────▼──────────────┐
│  LeadFormData       │
│  DTO                │
│                     │
│  fromModel()        │
│  toArray()          │
└──────┬──────────────┘
       │
       │ Passes to
       │
┌──────▼──────────────┐
│  LeadForm Service   │
│                     │
│  export()           │
└──────┬──────────────┘
       │
       │ HTTP POST
       │
┌──────▼──────────────────────┐
│  RealGreen API              │
│  /LeadForm/Submit           │
└──────┬──────────────────────┘
       │
       │ Success/Error
       │
┌──────▼──────────────────────┐
│  Update Estimate            │
│                             │
│  Success:                   │
│    exported_at = now()      │
│    export_error = null      │
│                             │
│  Error:                     │
│    export_error = message   │
│    log_exception()          │
└─────────────────────────────┘
```

### Batch Export Flow (CLI Command)
```
┌──────────────────────────────────┐
│  php artisan                     │
│  realgreen:export_estimates      │
│  [--all]                         │
└──────┬───────────────────────────┘
       │
       │ Executes
       │
┌──────▼────────────────────────────┐
│  ExportEstimatesToRealGreen      │
│  Command                          │
│                                   │
│  1. Check API key configured      │
│  2. Query estimates               │
│     - NOT DRAFT status            │
│     - WHERE exported_at IS NULL   │
│       (unless --all flag)         │
└──────┬────────────────────────────┘
       │
       │ For each estimate
       │
       ├──────────────────────┐
       │                      │
┌──────▼──────────┐  ┌────────▼──────────┐
│  LeadFormData   │  │  LeadFormData     │
│  DTO            │  │  DTO              │
└──────┬──────────┘  └────────┬──────────┘
       │                      │
┌──────▼──────────┐  ┌────────▼──────────┐
│  LeadForm       │  │  LeadForm         │
│  Service        │  │  Service          │
└──────┬──────────┘  └────────┬──────────┘
       │                      │
       │ HTTP POST            │ HTTP POST
       │                      │
┌──────▼──────────────────────▼──────────┐
│  RealGreen API                         │
│  /LeadForm/Submit                      │
└──────┬─────────────────────────────────┘
       │
       │ Parallel Updates
       │
┌──────▼─────────────────────────────────┐
│  Update Estimates in Database          │
│                                        │
│  On Success:                           │
│    estimate.exported_at = now()        │
│    estimate.export_error = null        │
│                                        │
│  On Error:                             │
│    estimate.export_error = message     │
│    continue (don't abort batch)        │
└────────────────────────────────────────┘
```

### Data Transformation Flow
```
┌─────────────────────────────────────────────┐
│  Bongo Estimate Model                       │
│                                             │
│  Fields:                                    │
│    number, first_name, last_name,           │
│    line_1, line_2, line_3,                  │
│    city, county, postcode,                  │
│    email, phone,                            │
│    latitude, longitude,                     │
│    areas[], subtotal, tax, total            │
└──────┬──────────────────────────────────────┘
       │
       │ Transform via LeadFormData.fromModel()
       │
       │ Mapping Rules:
       │ ┌─────────────────────────────────────┐
       │ │ name = prefix + number + full name  │
       │ │ zipcode = postcode (no spaces)      │
       │ │ streetNumberAndName = line_1+line_2 │
       │ │ houseNumber = line_1                │
       │ │ streetName = line_2                 │
       │ │ addressLine2 = line_3               │
       │ │ city = city                         │
       │ │ state = county                      │
       │ │ zip = postcode (no spaces)          │
       │ │ countryCode = 'GB'                  │
       │ │ cellPhoneNumber = phone             │
       │ │ homePhoneNumber = phone (non-07)    │
       │ │ emailAddress = email                │
       │ │ firstName = first_name              │
       │ │ lastName = last_name                │
       │ │ sourceCode = config value (28)      │
       │ │ latitude = latitude                 │
       │ │ longitude = longitude               │
       │ │ countryOrRegionCode = 'GB'          │
       │ │ callLogNotes = formatted summary    │
       │ │ employeeID = 'JCHAPMAN'             │
       │ │ actionReasonID = 18                 │
       │ └─────────────────────────────────────┘
       │
       │ Fallback Logic:
       │ ┌─────────────────────────────────────┐
       │ │ Required fields → '-- --'           │
       │ │ Optional fields → null              │
       │ └─────────────────────────────────────┘
       │
┌──────▼──────────────────────────────────────┐
│  RealGreen Lead Form Payload                │
│                                             │
│  {                                          │
│    "name": "KLC-8137 - Chloe West",         │
│    "zipcode": "LE675FX",                    │
│    "streetNumberAndName": "28 Elsdon Close",│
│    "emailAddress": "email@example.com",     │
│    "cellPhoneNumber": "+447943707718",      │
│    "sourceCode": 28,                        │
│    "firstName": "Chloe",                    │
│    "lastName": "West",                      │
│    "countryOrRegionCode": "GB",             │
│    "callLogNotes": "Total Area(m2): 29.55   │
│                     Subtotal: 26.67         │
│                     VAT: 5.33               │
│                     Total: 32.00...",       │
│    "employeeID": "JCHAPMAN",                │
│    "actionReasonID": 18                     │
│  }                                          │
└─────────────────────────────────────────────┘
```

## Configuration

### Config File Structure (src/Config/realgreen.php)
```php
<?php

return [
    // API Configuration
    // Environment: REALGREEN_API_URL
    // Default: https://saapi.realgreen.com
    'api_url' => env('REALGREEN_API_URL', 'https://saapi.realgreen.com'),

    // Marketing Source Codes
    // Source: Services > Source Code API
    // Current: 28 (GOO = Google)
    'source_code' => 28,
];
```

### Environment Variables
```bash
# Optional: Override API URL (defaults to production)
REALGREEN_API_URL=https://saapi.realgreen.com

# Note: API key is stored in system settings, not .env
# Access via: setting('system::credentials.real_green_api_key')
```

### System Settings Required
- `system::credentials.real_green_api_key`: RealGreen API authentication key

### Hard-Coded Values
```php
// Fixed employee ID for all submissions
'employeeID' => 'JCHAPMAN'

// Fixed action reason (18 = Note type in RealGreen)
'actionReasonID' => 18

// Fixed country codes
'countryCode' => 'GB'
'countryOrRegionCode' => 'GB'
```

## Extension Points

### Adding New API Endpoints

**Step 1**: Create service class
```php
// src/Services/NewEndpoint.php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Services;

use Illuminate\Http\Client\RequestException;
use Illuminate\Support\Facades\Http;

class NewEndpoint
{
    /** @throws RequestException */
    public function call(array $data): ?array
    {
        return Http::withoutVerifying()
            ->baseUrl(config('realgreen.api_url'))
            ->withHeaders(['apiKey' => setting('system::credentials.real_green_api_key')])
            ->post('/NewEndpoint/Action', $data)
            ->throw()
            ->json();
    }
}
```

### Adding New Commands

**Step 1**: Create command class
```php
// src/Commands/NewCommand.php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Commands;

use Illuminate\Console\Command;

class NewCommand extends Command
{
    protected $signature = 'realgreen:new_command {--option}';
    protected $description = 'Description of new command';

    public function handle(): void
    {
        // Implementation
    }
}
```

**Step 2**: Register in service provider
```php
// src/RealGreenServiceProvider.php
protected array $commands = [
    ExportEstimatesToRealGreen::class,
    NewCommand::class, // Add here
];
```

### Adding New Event Listeners

**Step 1**: Create listener class
```php
// src/Listeners/NewListener.php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Listeners;

class NewListener
{
    public function handle($event): void
    {
        // Implementation
    }
}
```

**Step 2**: Register in service provider
```php
// src/RealGreenServiceProvider.php
protected array $listeners = [
    EstimateUpdated::class => [
        ExportEstimateToRealGreen::class,
    ],
    NewEvent::class => [ // Add here
        NewListener::class,
    ],
];
```

### Customising Data Transformation

**Modify LeadFormData class**:

1. Add new property:
```php
public ?string $newField;
```

2. Add setter method:
```php
protected function setNewField(Estimate $estimate): void
{
    $this->newField = $estimate->new_field ?? null;
}
```

3. Call in `fromModel()`:
```php
public function fromModel(Estimate $estimate): self
{
    // ... existing setters
    $this->setNewField($estimate);

    return $this;
}
```

4. Include in `toArray()`:
```php
public function toArray(): array
{
    return [
        // ... existing fields
        'newField' => $this->newField,
    ];
}
```

## Adding New Features

### Example: Add Support for Custom Fields

**Step 1**: Update configuration
```php
// src/Config/realgreen.php
return [
    'api_url' => env('REALGREEN_API_URL', 'https://saapi.realgreen.com'),
    'source_code' => 28,
    'custom_field_mapping' => [ // New
        'internal_field' => 'external_field',
    ],
];
```

**Step 2**: Update LeadFormData
```php
public function fromModel(Estimate $estimate): self
{
    // ... existing setters

    // Add custom field mapping
    $customMapping = config('realgreen.custom_field_mapping', []);
    foreach ($customMapping as $internal => $external) {
        if (isset($estimate->$internal)) {
            $this->$external = $estimate->$internal;
        }
    }

    return $this;
}
```

### Example: Add Retry Logic

**Step 1**: Create retry service wrapper
```php
// src/Services/RetryableLeadForm.php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Services;

use Bongo\Estimate\Models\Estimate;
use Illuminate\Support\Facades\Log;

class RetryableLeadForm
{
    protected LeadForm $leadForm;
    protected int $maxAttempts = 3;

    public function __construct(LeadForm $leadForm)
    {
        $this->leadForm = $leadForm;
    }

    public function export(Estimate $estimate): ?array
    {
        $attempts = 0;
        $lastException = null;

        while ($attempts < $this->maxAttempts) {
            try {
                return $this->leadForm->export($estimate);
            } catch (\Exception $e) {
                $lastException = $e;
                $attempts++;
                Log::warning("Export attempt {$attempts} failed", [
                    'estimate_id' => $estimate->id,
                    'error' => $e->getMessage(),
                ]);

                if ($attempts < $this->maxAttempts) {
                    sleep(2 ** $attempts); // Exponential backoff
                }
            }
        }

        throw $lastException;
    }
}
```

**Step 2**: Update listener to use retry wrapper
```php
// src/Listeners/ExportEstimateToRealGreen.php
try {
    $leadForm = new RetryableLeadForm(new LeadForm());
    $leadForm->export($event->estimate);
    // ... success handling
}
```

### Example: Add Export Filtering

**Step 1**: Create filter trait
```php
// src/Traits/FiltersEstimatesForExport.php
<?php

declare(strict_types=1);

namespace Bongo\RealGreen\Traits;

use Bongo\Estimate\Models\Estimate;
use Illuminate\Database\Eloquent\Builder;

trait FiltersEstimatesForExport
{
    protected function getExportableEstimates(bool $includeExported = false): Builder
    {
        $query = Estimate::query()
            ->whereNot('status', Estimate::DRAFT)
            ->orderBy('created_at');

        if (!$includeExported) {
            $query->whereNull('exported_at');
        }

        // Add custom filters
        if ($this->hasMinimumTotalFilter()) {
            $query->where('total', '>=', config('realgreen.minimum_export_total', 0));
        }

        return $query;
    }

    protected function hasMinimumTotalFilter(): bool
    {
        return config('realgreen.minimum_export_total') !== null;
    }
}
```

**Step 2**: Use in command
```php
// src/Commands/ExportEstimatesToRealGreen.php
use Bongo\RealGreen\Traits\FiltersEstimatesForExport;

class ExportEstimatesToRealGreen extends Command
{
    use FiltersEstimatesForExport;

    public function handle(): void
    {
        $estimates = $this->getExportableEstimates($this->option('all'))->get();
        // ... rest of implementation
    }
}
```

## Trait and Interface Reference

### Used Traits

| Trait | Source | Used In | Purpose |
|-------|--------|---------|---------|
| `SeedsPackage` | `Bongo\Package\Traits\SeedsPackage` | `PackageSeeder` | Provides `package()` helper method for seeding |

### Used Interfaces

| Interface | Implemented By | Purpose |
|-----------|---------------|---------|
| None | - | This package doesn't implement custom interfaces |

### Available for Extension

| Class | Type | Can Be Extended |
|-------|------|----------------|
| `RealGreenServiceProvider` | Service Provider | No (final via framework) |
| `ExportEstimatesToRealGreen` | Command | Yes |
| `ExportEstimateToRealGreen` | Listener | Yes |
| `LeadFormData` | DTO | Yes |
| `LeadForm` | Service | Yes |
| `SourceCode` | Service | Yes |

## Testing Strategy

### Unit Tests
Test individual components in isolation:
- LeadFormData transformation logic
- SourceCode API parsing
- Command validation logic

### Integration Tests
Test interactions between components:
- Event listener triggering export
- Command querying and updating estimates
- Service making HTTP requests (with HTTP fake)

### Example Test Structure
```
tests/
├── Unit/
│   ├── Data/
│   │   └── LeadFormDataTest.php
│   └── Services/
│       ├── LeadFormTest.php
│       └── SourceCodeTest.php
├── Feature/
│   ├── Commands/
│   │   └── ExportEstimatesToRealGreenTest.php
│   └── Listeners/
│       └── ExportEstimateToRealGreenTest.php
└── TestCase.php
```

## Security Considerations

### API Key Storage
- Never commit API keys to version control
- Store in system settings: `setting('system::credentials.real_green_api_key')`
- Access only when needed

### SSL Verification
Current implementation uses `withoutVerifying()` for SSL certificate verification. Consider:
- Enabling verification in production
- Using proper CA certificates
- Only disabling for staging/development environments

### Data Sanitisation
LeadFormData performs basic sanitisation:
- Removes spaces from postcodes
- Validates phone number format (07 prefix check)
- Provides fallback values for required fields

Consider adding:
- Email validation
- Phone number formatting
- HTML sanitisation for notes fields

### Error Information Disclosure
Current error handling stores full exception messages in database:
```php
$estimate->export_error = $e->getMessage();
```

Consider:
- Sanitising error messages before storage
- Logging detailed errors separately
- Showing generic errors to users
