# RealGreen API Integration

[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-blue.svg)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-%5E10.0-red.svg)](https://laravel.com)

Laravel package for integrating Bongo's estimate system with RealGreen's CRM platform. Automatically exports estimate data to RealGreen's lead form API when estimates are created or updated.

## Features

- **Event-Driven Export**: Automatically exports estimates when updated
- **Batch Export Command**: CLI command for bulk exports
- **Silent Failure**: Errors don't block main application flow
- **Idempotent Operations**: Safe to retry failed exports
- **Data Transformation**: Maps Bongo estimate structure to RealGreen format
- **Error Tracking**: Records export errors in database for monitoring

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0
- `bongo/estimate` package
- `bongo/package` package

## Installation

### Via Composer

```bash
composer require bongo/realgreen
```

### Laravel Auto-Discovery

Laravel 10+ automatically discovers and registers the service provider. No manual registration needed.

### For Laravel 5 (Legacy)

Add the service provider to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\RealGreen\RealGreenServiceProvider::class,
],
```

### Configuration

Publish the configuration file (optional):

```bash
php artisan vendor:publish --provider="Bongo\RealGreen\RealGreenServiceProvider"
```

### Environment Variables

Add to your `.env` file:

```bash
# RealGreen API URL (optional, defaults to production)
REALGREEN_API_URL=https://saapi.realgreen.com
```

### System Settings

Configure the RealGreen API key in your system settings:

```php
setting()->set('system::credentials.real_green_api_key', 'your-api-key-here');
```

Or via the admin interface under System > Credentials.

### Database Migration

Add the following columns to your `estimates` table:

```php
Schema::table('estimates', function (Blueprint $table) {
    $table->timestamp('exported_at')->nullable()->after('updated_at');
    $table->text('export_error')->nullable()->after('exported_at');
});
```

### Run Package Seeder

Register the package in your system:

```bash
php artisan db:seed --class="Bongo\RealGreen\Seeders\PackageSeeder"
```

## Usage

### Automatic Export

Estimates are automatically exported to RealGreen when:

1. The application is in production mode
2. The RealGreen API key is configured
3. An estimate is updated
4. The estimate status is not `DRAFT`

No manual intervention required. The package listens for `EstimateUpdated` events.

### Manual Batch Export

Export all non-exported estimates:

```bash
php artisan realgreen:export_estimates
```

Re-export all estimates (including previously exported):

```bash
php artisan realgreen:export_estimates --all
```

### Programmatic Export

```php
use Bongo\RealGreen\Services\LeadForm;
use Bongo\Estimate\Models\Estimate;

$estimate = Estimate::find($id);
$leadForm = new LeadForm();

try {
    $response = $leadForm->export($estimate);
    // Handle success
} catch (\Illuminate\Http\Client\RequestException $e) {
    // Handle error
}
```

### Retrieving Marketing Source Codes

```php
use Bongo\RealGreen\Services\SourceCode;

$sourceCodeService = new SourceCode();

// Get all source codes
$allCodes = $sourceCodeService->all();

// Find by abbreviation (e.g., 'GOO' for Google)
$googleCode = $sourceCodeService->byAbbreviation('GOO');
```

### Checking Export Status

```php
use Bongo\Estimate\Models\Estimate;

// Check if estimate has been exported
$estimate = Estimate::find($id);

if ($estimate->exported_at) {
    echo "Exported at: {$estimate->exported_at}";
}

// Check for export errors
if ($estimate->export_error) {
    echo "Export failed: {$estimate->export_error}";
}

// Get all failed exports
$failedExports = Estimate::whereNotNull('export_error')->get();
```

## Configuration

### Config File

The package configuration is located at `config/realgreen.php`:

```php
return [
    // RealGreen API base URL
    'api_url' => env('REALGREEN_API_URL', 'https://saapi.realgreen.com'),

    // Marketing source code (28 = GOO/Google)
    // Retrieved from RealGreen Services > Source Code API
    'source_code' => 28,
];
```

### Customising Field Mapping

To customise how estimate data maps to RealGreen fields, extend the `LeadFormData` class:

```php
namespace App\Services;

use Bongo\RealGreen\Data\LeadFormData as BaseLeadFormData;

class CustomLeadFormData extends BaseLeadFormData
{
    protected function setSourceCode(): void
    {
        // Override to use dynamic source code
        $this->sourceCode = $this->estimate->source_code ?? config('realgreen.source_code');
    }
}
```

Then bind it in your service provider:

```php
$this->app->bind(
    \Bongo\RealGreen\Data\LeadFormData::class,
    \App\Services\CustomLeadFormData::class
);
```

## Architecture

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider` which provides automatic bootstrapping:

- **Config**: Auto-loaded from `src/Config/realgreen.php`
- **Commands**: Registers `ExportEstimatesToRealGreen` console command
- **Listeners**: Registers `ExportEstimateToRealGreen` for `EstimateUpdated` event

### Event-Driven Architecture

```
Estimate Updated → EstimateUpdated Event → ExportEstimateToRealGreen Listener
                                                      ↓
                                              LeadFormData (DTO)
                                                      ↓
                                              LeadForm Service
                                                      ↓
                                          RealGreen API (/LeadForm/Submit)
                                                      ↓
                                    Update estimate (exported_at, export_error)
```

### Key Components

| Component | Purpose |
|-----------|---------|
| **RealGreenServiceProvider** | Package bootstrapping and registration |
| **ExportEstimatesToRealGreen** | CLI command for batch exports |
| **ExportEstimateToRealGreen** | Event listener for real-time exports |
| **LeadForm** | HTTP client for API communication |
| **LeadFormData** | Data Transfer Object for payload transformation |
| **SourceCode** | Service for retrieving marketing source codes |

### Data Transformation

The `LeadFormData` DTO transforms Bongo estimate fields to RealGreen's required format:

| Bongo Field | RealGreen Field | Transformation |
|------------|----------------|---------------|
| `number` | `name` | Prefixed with estimate number prefix |
| `postcode` | `zipcode`, `zip` | Spaces removed |
| `line_1` + `line_2` | `streetNumberAndName` | Concatenated |
| `email` | `emailAddress` | Direct mapping |
| `phone` | `cellPhoneNumber` | Direct mapping |
| `county` | `state` | Direct mapping |
| Areas + totals | `callLogNotes` | Formatted as text |

**Fallback Values**:
- Required fields: `'-- --'` when empty
- Optional fields: `null` when empty

See `src/Schema/lead_form.json` for example payload structure.

## Testing

### Run Tests

```bash
composer test
# or
vendor/bin/phpunit
```

### Run Tests with Coverage

```bash
composer test-coverage
```

### Code Analysis

```bash
composer analyse
# or
vendor/bin/phpstan analyse
```

### Code Formatting

Check code style:
```bash
vendor/bin/pint --test
```

Fix code style:
```bash
composer format
# or
vendor/bin/pint
```

## Error Handling

### Silent Failure

The event listener uses silent failure to prevent blocking estimate updates. Errors are:

1. Stored in `estimate.export_error` field
2. Logged via `log_exception()` helper
3. Printed to console via `console_print()` helper

### Retry Strategy

Failed exports are not automatically retried. To retry:

1. Query estimates with errors:
```php
$failed = Estimate::whereNotNull('export_error')->get();
```

2. Clear error and re-export:
```php
foreach ($failed as $estimate) {
    $estimate->export_error = null;
    $estimate->exported_at = null;
    $estimate->save();
}

// Then run export command
php artisan realgreen:export_estimates
```

### Common Issues

**Exports not happening in development:**
- Check: `app()->isProduction()` must return `true`
- Solution: Set `APP_ENV=production` in `.env` or modify listener condition

**API key not found:**
- Check: `setting('system::credentials.real_green_api_key')` is set
- Solution: Configure via admin interface or system settings

**SSL verification errors:**
- Check: Using `withoutVerifying()` in HTTP client
- Solution: Enable SSL verification for production or install proper CA certificates

## Security

### API Key Storage

- Never commit API keys to version control
- Store in system settings table
- Access via `setting()` helper function

### SSL Verification

The package currently disables SSL certificate verification using `withoutVerifying()`. Consider:

- Enabling verification for production environments
- Using proper CA certificate bundles
- Only disabling for staging/development if required

### Data Sanitisation

The package performs basic sanitisation:
- Removes spaces from postcodes
- Validates phone number format (07 prefix check)
- Provides fallback values for required fields

Consider additional validation for:
- Email addresses
- Phone number formatting
- HTML content in notes fields

## Troubleshooting

### Debug Exports

Add logging to `LeadForm::export()`:

```php
\Log::debug('RealGreen Export Attempt', [
    'estimate_id' => $estimate->id,
    'payload' => $leadFormData,
]);
```

Check Laravel logs:
```bash
tail -f storage/logs/laravel.log
```

### Test API Connection

```php
use Illuminate\Support\Facades\Http;

$response = Http::withoutVerifying()
    ->baseUrl(config('realgreen.api_url'))
    ->withHeaders(['apiKey' => setting('system::credentials.real_green_api_key')])
    ->get('/SourceCode');

dd($response->status(), $response->json());
```

### Query Export Status

```sql
-- Find estimates that failed to export
SELECT id, number, created_at, export_error
FROM estimates
WHERE export_error IS NOT NULL;

-- Find estimates that haven't been exported
SELECT id, number, created_at
FROM estimates
WHERE status != 'draft'
AND exported_at IS NULL;
```

## Documentation

- **Architecture Details**: See `ARCHITECTURE.md` for class diagrams, data flow, and extension points
- **Development Guide**: See `.cursorrules` for Cursor AI development guidelines
- **Code Templates**: See `.github/copilot-instructions.md` for GitHub Copilot templates
- **Quick Reference**: See `CLAUDE.md` for Claude Code quick reference

## API Reference

### RealGreen API Endpoints

**Base URL**: `https://saapi.realgreen.com`

**Authentication**: Header-based API key

```
Authorization: apiKey YOUR_API_KEY
```

**Endpoints**:

| Method | Endpoint | Purpose |
|--------|----------|---------|
| POST | `/LeadForm/Submit` | Submit lead form data |
| GET | `/SourceCode` | Retrieve marketing source codes |

## Contributing

This package is part of the Bongo monorepo. Contributions should follow:

1. PHP 8.2+ type declarations
2. Strict types enabled (`declare(strict_types=1);`)
3. Laravel Pint code style (Laravel preset)
4. PHPStan level 8 compliance
5. Comprehensive tests for new features

### Code Style

The package uses Laravel Pint with custom rules:

```json
{
    "preset": "laravel",
    "rules": {
        "new_with_parentheses": false,
        "nullable_type_declaration_for_default_null_value": false,
        "declare_strict_types": true,
        "declare_parentheses": true
    }
}
```

Run Pint before committing:
```bash
composer format
```

## Changelog

See git history for changes:
```bash
git log --oneline
```

## License

MIT License - see `LICENSE` file for details.

## Support

This package is actively being developed. Please submit feedback and issues to the maintainer:

- **Author**: Stuart Elliott
- **Email**: stuart.elliott@bespokeuk.com
- **Homepage**: https://bespokeuk.com
- **Repository**: https://bitbucket.org/designtec/realgreen

## Related Packages

- `bongo/framework`: Core framework and service provider base
- `bongo/estimate`: Estimate model and events
- `bongo/package`: Package management system

## Credits

Built with [Laravel](https://laravel.com) framework.
