# GitHub Copilot Instructions - Bongo Redirect Package

## Project Overview

This is a Laravel package that provides URL redirect management with an admin interface, automatic URL normalization (www removal, trailing slash handling), and legacy CMS URL migration capabilities. Part of the Bongo monorepo ecosystem, it extends the `bongo/framework` package.

**Namespace**: `Bongo\Redirect`
**PHP Version**: 8.2+
**Laravel Version**: 10+

## Key Classes and Relationships

### Core Model
```php
Bongo\Redirect\Models\Redirect extends Bongo\Framework\Models\AbstractModel
```
- **Fillable**: `from`, `to`
- **Mutators**: Automatically adds leading slashes, lowercases `to` attribute
- **Table**: `redirects` (id, from, to, timestamps)
- **Location**: `src/Models/Redirect.php`

### Controllers

#### RedirectController
```php
Bongo\Redirect\Http\Controllers\Backend\RedirectController extends AbstractController
```
**Methods**:
- `index(): View` - List redirects
- `create(): View` - Show create form
- `store(StoreRedirectRequest $request): RedirectResponse` - Create redirect
- `show(Redirect $redirect): View` - Show single redirect
- `edit(Redirect $redirect): View` - Show edit form
- `update(UpdateRedirectRequest $request, Redirect $redirect): RedirectResponse` - Update redirect
- `destroy(Redirect $redirect)` - Delete redirect

**Location**: `src/Http/Controllers/Backend/RedirectController.php`

#### RedirectDatatableController
```php
Bongo\Redirect\Http\Controllers\Backend\RedirectDatatableController extends AbstractDatatableController
```
**Methods**:
- `getBaseQuery(): Builder` - Returns base query for datatable
- `applyOrderBy()` - Orders by `from`, then `to` ASC

**Location**: `src/Http/Controllers/Backend/RedirectDatatableController.php`

### Middleware

#### HasRedirects
Checks for database redirects and handles legacy URL patterns:
- Database redirects (from `redirects` table)
- Legacy page URLs: `?p=page.name` → named route
- Legacy blog URLs: `/blogs/archive/YYYY/MM/DD/slug.aspx` → named route
- Legacy review URLs: `?review` → review index

**Location**: `src/Http/Middleware/HasRedirects.php`

#### RemoveWWW
Removes www prefix from URLs with 301 redirect.

**Location**: `src/Http/Middleware/RemoveWWW.php`

#### RemoveSlash
Removes trailing slashes from URLs (except homepage) with 301 redirect.

**Location**: `src/Http/Middleware/RemoveSlash.php`

### Events and Listeners

#### Events
```php
Bongo\Redirect\Events\RedirectCreated
Bongo\Redirect\Events\RedirectUpdated
```
Both contain public `Redirect $redirect` property.

#### Listener
```php
Bongo\Redirect\Listeners\ClearRedirectCache implements ShouldQueue
```
- Clears `'redirects'` cache key
- Queued with 3 retry attempts
- Listens to both `RedirectCreated` and `RedirectUpdated`

### Service Provider
```php
Bongo\Redirect\RedirectServiceProvider extends Bongo\Framework\Providers\AbstractServiceProvider
```
- Registers middleware: `removeWWW`, `removeSlash`, `hasRedirects`
- Auto-registers backend routes, views, migrations, translations
- Pushes `RemoveWWW` and `RemoveSlash` to 'web' middleware group

**Location**: `src/RedirectServiceProvider.php`

## Code Style Templates

### Controller Method Template
```php
public function methodName(FormRequest $request, Model $model): RedirectResponse|View
{
    // Logic here

    return redirect()
        ->route('backend.redirect.show', $model->id)
        ->success(trans('redirect::backend.action_success'));
}
```

### Event Template
```php
namespace Bongo\Redirect\Events;

use Bongo\Redirect\Models\Redirect;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class EventName
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public Redirect $redirect;

    public function __construct(Redirect $redirect)
    {
        $this->redirect = $redirect;
    }
}
```

### Middleware Template
```php
namespace Bongo\Redirect\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class MiddlewareName
{
    public function handle(Request $request, Closure $next)
    {
        // Check condition
        if ($this->shouldRedirect($request)) {
            return redirect()->to('/new-path', 301);
        }

        return $next($request);
    }

    protected function shouldRedirect(Request $request): bool
    {
        // Logic here
    }
}
```

### Form Request Template
```php
namespace Bongo\Redirect\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class ActionRedirectRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'from' => 'required',
            'to' => 'required',
        ];
    }
}
```

### Listener Template (Queued)
```php
namespace Bongo\Redirect\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;

class ListenerName implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        // Handle event
    }
}
```

## Common Patterns

### Creating Redirects with Events
```php
// In controller
$redirect = $this->redirect->create($request->all());
event(new RedirectCreated($redirect));

return redirect()
    ->route('backend.redirect.show', $redirect->id)
    ->success(trans('redirect::backend.store_success'));
```

### Model Attribute Mutators
```php
// Automatically normalize URLs
public function setFromAttribute($value)
{
    if (! empty($value)) {
        $this->attributes['from'] = '/'.ltrim($value, '/');
    }
}
```

### Middleware Registration
```php
// In service provider
protected array $middlewares = [
    'middlewareAlias' => MiddlewareClass::class,
];

// Apply globally to middleware group
public function boot(): void
{
    parent::boot();

    $this->app['router']->pushMiddlewareToGroup('web', MiddlewareClass::class);
}
```

### Event Listener Registration
```php
// In service provider
protected array $listeners = [
    EventClass::class => [
        ListenerClass::class,
    ],
];
```

### Route Model Binding
```php
// In controller - automatically resolved
public function show(Redirect $redirect): View
{
    return view('redirect::backend.show', compact('redirect'));
}

// In routes - automatic binding by parameter name
Route::get('{redirect}/', [RedirectController::class, 'show'])
    ->name('show');
```

### Translation Keys
```php
// Always namespace with module
trans('redirect::backend.store_success')
trans('redirect::backend.delete_failed')
```

### View Names
```php
// Always namespace with module
view('redirect::backend.index')
view('redirect::backend.create', compact('redirect'))
```

### Datatable Controller Pattern
```php
protected function getBaseQuery(): Builder
{
    return $this->model->newQuery();
}

protected function applyOrderBy()
{
    $this->query
        ->orderBy('column1', 'asc')
        ->orderBy('column2', 'asc');
}
```

### Checking for Package Availability
```php
// Used in middleware for optional integrations
if ($this->isABlogRedirect($request) && package()->isEnabled('post')) {
    // Handle blog redirect
}
```

### Query String Preservation
```php
// Preserve query params in redirects
if ($params = $request->getQueryString()) {
    return redirect()->to("{$redirect->to}?{$params}", 301);
}
```

### Cache Management
```php
// Clear cache after operations
Cache::forget('redirects');
```

## Framework Utilities

### URL Helper (from bongo/framework)
```php
use Bongo\Framework\Helpers\URL;

URL::isHomePage($request);      // Check if homepage
URL::isAFile($request);         // Check if requesting a file
URL::hasQueryString($request);  // Check if has query string
URL::endsWithSlash($request);   // Check if ends with /
URL::hasWWW($request);          // Check if has www prefix
URL::withoutWWW($request);      // Get host without www
```

## Naming Conventions

- **Routes**: `backend.redirect.{action}` (index, create, store, show, edit, update, destroy, datatable)
- **Views**: `redirect::backend.{action}`
- **Translations**: `redirect::backend.{key}`
- **Middleware Aliases**: camelCase (removeWWW, removeSlash, hasRedirects)
- **Events**: `{Model}{Action}` (RedirectCreated, RedirectUpdated)
- **Listeners**: `{Action}{Noun}` (ClearRedirectCache)
- **Controllers**: `{Model}Controller`, `{Model}DatatableController`

## Configuration

Config file: `src/Config/redirect.php`
```php
return [
    'prefix' => 'redirects', // Admin route prefix
];
```

Access: `config('redirect.prefix')`

## Testing

Test base class: `Bongo\Redirect\Tests\TestCase`
```php
namespace Bongo\Redirect\Tests;

class MyTest extends TestCase
{
    public function test_example()
    {
        // Test code
    }
}
```

## Important Notes

1. **Always fire events** after create/update operations to clear cache
2. **Use route model binding** instead of manual queries
3. **Namespace everything** with `redirect::`
4. **Return type declarations** required on all public methods
5. **Middleware order matters**: RemoveWWW → RemoveSlash → HasRedirects
6. **Database redirects** take precedence over legacy URL handlers
7. **Query strings** are preserved when performing redirects
8. **Cache key** is always `'redirects'` (singular)
9. **Admin routes** require `developer` middleware
10. **All redirects** use 301 (permanent) status code
