# Architecture Documentation - Bongo Redirect Package

## Overview

The Bongo Redirect package provides a comprehensive URL redirect management system for Laravel applications. It combines database-driven redirects with legacy URL migration handlers and automatic URL normalization middleware.

**Key Capabilities**:
- Admin CRUD interface for managing redirects
- Database-driven redirect lookups
- Legacy Bongo CMS URL pattern handling (pages, blogs, reviews)
- Automatic URL normalization (www removal, trailing slash removal)
- Event-driven cache invalidation
- 301 permanent redirects for SEO compliance

## Directory Structure

```
redirect/
├── src/
│   ├── Config/
│   │   └── redirect.php                    # Configuration file
│   │
│   ├── Events/
│   │   ├── RedirectCreated.php             # Event fired on creation
│   │   └── RedirectUpdated.php             # Event fired on update
│   │
│   ├── Http/
│   │   ├── Controllers/
│   │   │   └── Backend/
│   │   │       ├── RedirectController.php         # CRUD operations
│   │   │       └── RedirectDatatableController.php # DataTables API
│   │   │
│   │   ├── Middleware/
│   │   │   ├── HasRedirects.php            # Main redirect logic
│   │   │   ├── RemoveSlash.php             # Trailing slash removal
│   │   │   └── RemoveWWW.php               # WWW prefix removal
│   │   │
│   │   └── Requests/
│   │       ├── StoreRedirectRequest.php    # Create validation
│   │       └── UpdateRedirectRequest.php   # Update validation
│   │
│   ├── Listeners/
│   │   └── ClearRedirectCache.php          # Cache invalidation listener
│   │
│   ├── Migrations/
│   │   └── 2021_01_01_000001_create_redirects_table.php
│   │
│   ├── Models/
│   │   └── Redirect.php                    # Eloquent model
│   │
│   ├── Routes/
│   │   └── backend.php                     # Admin routes
│   │
│   ├── Translations/
│   │   └── en/
│   │       └── backend.php                 # English translations
│   │
│   ├── Views/
│   │   └── backend/
│   │       ├── index.blade.php             # List view
│   │       ├── create.blade.php            # Create form
│   │       ├── edit.blade.php              # Edit form
│   │       ├── show.blade.php              # Detail view
│   │       └── partials/
│   │           └── form/
│   │               └── details.blade.php
│   │
│   └── RedirectServiceProvider.php         # Service provider
│
├── tests/
│   └── TestCase.php                        # Base test case
│
├── composer.json
├── phpunit.xml
└── README.md
```

## Core Architecture

### Class Hierarchy

```
AbstractServiceProvider (framework)
    └── RedirectServiceProvider
            ├── registers → RemoveWWW middleware
            ├── registers → RemoveSlash middleware
            ├── registers → HasRedirects middleware
            └── registers → ClearRedirectCache listener

AbstractModel (framework)
    └── Redirect
            ├── $fillable: [from, to]
            ├── setFromAttribute()  # Adds leading slash
            └── setToAttribute()    # Adds leading slash, lowercases

AbstractController (framework)
    └── RedirectController
            ├── index()
            ├── create()
            ├── store()
            ├── show()
            ├── edit()
            ├── update()
            └── destroy()

AbstractDatatableController (framework)
    └── RedirectDatatableController
            ├── getBaseQuery()
            └── applyOrderBy()
```

### Database Schema

```
redirects
├── id (increments, primary key)
├── from (string, indexed)       # Source URL path
├── to (string, indexed)         # Target URL path
├── created_at (timestamp)
└── updated_at (timestamp)
```

**Indexes**: Both `from` and `to` columns are indexed for fast lookups.

## Request Lifecycle Diagrams

### 1. Incoming Request Flow (Middleware Chain)

```
HTTP Request
    │
    ├─[web middleware group]
    │   │
    │   ├─[RemoveWWW]──────────────┐
    │   │   Has www prefix?        │
    │   │   └─ YES → 301 redirect  │
    │   │                           │
    │   ├─[RemoveSlash]─────────────┤
    │   │   Has trailing slash?     │
    │   │   └─ YES → 301 redirect   │
    │   │                           │
    │   └─[HasRedirects]────────────┤ (if manually applied)
    │       │                       │
    │       ├─ Check database       │
    │       │   └─ FOUND → 301      │
    │       │                       │
    │       ├─ Check legacy pages   │
    │       │   └─ MATCH → 301      │
    │       │                       │
    │       ├─ Check legacy blogs   │
    │       │   └─ MATCH → 301      │
    │       │                       │
    │       └─ Check legacy reviews │
    │           └─ MATCH → 301      │
    │                               │
    └───────────────────────────────┘
              │
         [next middleware]
              │
         [Application]
```

### 2. Admin CRUD Flow

```
User Action → Controller → Model → Event → Listener
                                      │
                                      └─→ Cache::forget('redirects')

Example: Creating a Redirect
────────────────────────────

1. User submits form
       ↓
2. StoreRedirectRequest validates
       ↓
3. RedirectController::store()
       ↓
4. Redirect::create(['from' => '...', 'to' => '...'])
       ├─ setFromAttribute() adds leading /
       └─ setToAttribute() adds leading /, lowercases
       ↓
5. event(new RedirectCreated($redirect))
       ↓
6. ClearRedirectCache::handle() [queued]
       └─ Cache::forget('redirects')
       ↓
7. Redirect to show page with success message
```

### 3. HasRedirects Middleware Decision Tree

```
Request arrives
    │
    ├─ Check redirects table (DB lookup by 'from' path)
    │   └─ FOUND?
    │       ├─ YES → Has query string?
    │       │        ├─ YES → redirect("{to}?{params}", 301)
    │       │        └─ NO  → redirect("{to}", 301)
    │       │
    │       └─ NO → Continue to legacy handlers
    │
    ├─ Check if legacy page (?p=page.name)
    │   └─ MATCH? → redirect('frontend.page.show', slug, 301)
    │
    ├─ Check if legacy blog (?blogs, ?blog, /blogs/archive/*, /blog/archive/*)
    │   ├─ Is 'post' package enabled?
    │   │   └─ YES → Parse YYYY/MM/DD/slug
    │   │            └─ redirect('frontend.post.show', slug, 301)
    │   └─ NO → Skip
    │
    ├─ Check if legacy review (?review)
    │   ├─ Is 'review' package enabled?
    │   │   └─ YES → redirect('frontend.review.index', 301)
    │   └─ NO → Skip
    │
    └─ No redirect found → Pass through to application
```

## Component Reference

### Models

#### Redirect
**Location**: `src/Models/Redirect.php`

| Property/Method | Type | Description |
|----------------|------|-------------|
| `$fillable` | `array` | `['from', 'to']` |
| `setFromAttribute($value)` | Mutator | Adds leading slash to 'from' |
| `setToAttribute($value)` | Mutator | Adds leading slash and lowercases 'to' |

**Extends**: `Bongo\Framework\Models\AbstractModel`

**Usage Example**:
```php
$redirect = Redirect::create([
    'from' => 'old-page',      // Stored as: /old-page
    'to' => 'New-Page',        // Stored as: /new-page
]);
```

### Controllers

#### RedirectController
**Location**: `src/Http/Controllers/Backend/RedirectController.php`

| Method | Return Type | Description |
|--------|-------------|-------------|
| `index()` | `View` | List all redirects (datatable) |
| `create()` | `View` | Show create form |
| `store(StoreRedirectRequest)` | `RedirectResponse` | Store new redirect, fire event |
| `show(Redirect)` | `View` | Show redirect details |
| `edit(Redirect)` | `View` | Show edit form |
| `update(UpdateRedirectRequest, Redirect)` | `RedirectResponse` | Update redirect, fire event |
| `destroy(Redirect)` | `RedirectResponse` | Delete redirect, clear cache |

**Route Naming**: All routes prefixed with `backend.redirect.*`

#### RedirectDatatableController
**Location**: `src/Http/Controllers/Backend/RedirectDatatableController.php`

| Method | Description |
|--------|-------------|
| `getBaseQuery()` | Returns base Eloquent query |
| `applyOrderBy()` | Orders by 'from', then 'to' ASC |

**Extends**: `Bongo\Framework\Http\Controllers\AbstractDatatableController`

**Usage**: Called via AJAX from datatable view (`backend.redirect.datatable` route)

### Middleware

#### RemoveWWW
**Location**: `src/Http/Middleware/RemoveWWW.php`

**Purpose**: Redirects www.domain.com → domain.com (301)

**Algorithm**:
1. Check if request has www prefix using `URL::hasWWW($request)`
2. If yes, modify host header to remove www
3. Return 301 redirect to new URL

**Applied**: Automatically to 'web' middleware group

#### RemoveSlash
**Location**: `src/Http/Middleware/RemoveSlash.php`

**Purpose**: Redirects /path/ → /path (301)

**Algorithm**:
1. Only process GET requests
2. Skip if homepage, file request, or has query string
3. If ends with slash, redirect without it

**Applied**: Automatically to 'web' middleware group

#### HasRedirects
**Location**: `src/Http/Middleware/HasRedirects.php`

**Purpose**: Main redirect logic - database + legacy URLs

**Algorithm**:
1. Check database for matching redirect (by 'from' path)
2. If found, redirect to 'to' path (preserve query string)
3. Check for legacy page pattern (?p=page.name)
4. Check for legacy blog patterns (multiple formats)
5. Check for legacy review pattern (?review)
6. If no match, pass through

**Applied**: Manually via middleware alias `hasRedirects`

**Helper Methods**:

| Method | Return Type | Description |
|--------|-------------|-------------|
| `existsInRedirectsTable($request)` | `?Redirect` | Query DB for matching redirect |
| `isAPageRedirect($request)` | `bool` | Check for ?p param |
| `isABlogRedirect($request)` | `bool` | Check for blog URL patterns |
| `isAReviewRedirect($request)` | `bool` | Check for ?review param |
| `removeBlogsPrefix($request)` | `string` | Strip blog archive prefix |
| `removeFileExtension($slug)` | `string` | Strip .aspx extension |

### Events

#### RedirectCreated
**Location**: `src/Events/RedirectCreated.php`

| Property | Type | Description |
|----------|------|-------------|
| `$redirect` | `Redirect` | The created redirect instance |

**Traits**: `Dispatchable`, `InteractsWithSockets`, `SerializesModels`

**Fired**: After redirect created in `RedirectController::store()`

#### RedirectUpdated
**Location**: `src/Events/RedirectUpdated.php`

| Property | Type | Description |
|----------|------|-------------|
| `$redirect` | `Redirect` | The updated redirect instance |

**Traits**: `Dispatchable`, `InteractsWithSockets`, `SerializesModels`

**Fired**: After redirect updated in `RedirectController::update()`

### Listeners

#### ClearRedirectCache
**Location**: `src/Listeners/ClearRedirectCache.php`

**Implements**: `ShouldQueue`

| Property/Method | Type | Description |
|----------------|------|-------------|
| `$tries` | `int` | 3 retry attempts |
| `handle($event)` | `void` | Clears 'redirects' cache key |

**Listens To**: `RedirectCreated`, `RedirectUpdated`

**Behavior**: Queued execution, clears cache key after redirect changes

### Form Requests

#### StoreRedirectRequest
**Location**: `src/Http/Requests/StoreRedirectRequest.php`

```php
public function rules(): array
{
    return [
        'from' => 'required',
        'to' => 'required',
    ];
}
```

#### UpdateRedirectRequest
**Location**: `src/Http/Requests/UpdateRedirectRequest.php`

Same validation rules as `StoreRedirectRequest`.

### Service Provider

#### RedirectServiceProvider
**Location**: `src/RedirectServiceProvider.php`

**Extends**: `Bongo\Framework\Providers\AbstractServiceProvider`

| Property | Type | Description |
|----------|------|-------------|
| `$module` | `string` | `'redirect'` - matches config/view namespace |
| `$listeners` | `array` | Maps events to listeners |
| `$middlewares` | `array` | Registers middleware aliases |

**Registered Listeners**:
```php
[
    RedirectCreated::class => [ClearRedirectCache::class],
    RedirectUpdated::class => [ClearRedirectCache::class],
]
```

**Registered Middleware**:
```php
[
    'removeWWW' => RemoveWWW::class,
    'removeSlash' => RemoveSlash::class,
    'hasRedirects' => HasRedirects::class,
]
```

**Boot Method**:
- Calls `parent::boot()` for automatic bootstrapping
- Pushes `RemoveWWW` to 'web' middleware group
- Pushes `RemoveSlash` to 'web' middleware group

**Auto-Bootstrapped** (by AbstractServiceProvider):
- Config from `src/Config/redirect.php`
- Routes from `src/Routes/backend.php`
- Views from `src/Views/` (namespace: `redirect::`)
- Migrations from `src/Migrations/`
- Translations from `src/Translations/` (namespace: `redirect::`)

## Configuration

**File**: `src/Config/redirect.php`

```php
return [
    'prefix' => 'redirects',  // Admin route prefix
];
```

**Access**: `config('redirect.prefix')`

**Used In**: `src/Routes/backend.php` for route group prefix

## Routes

**File**: `src/Routes/backend.php`

All routes are:
- Prefixed with `config('redirect.prefix')` (default: `redirects`)
- Named with `backend.redirect.*` prefix
- Protected by `developer` middleware
- Protected by `auth` + `employee` middleware (via AbstractServiceProvider)

| Method | URI | Name | Controller Method |
|--------|-----|------|------------------|
| GET | `/` | `backend.redirect.index` | `index()` |
| GET | `/create` | `backend.redirect.create` | `create()` |
| POST | `/store` | `backend.redirect.store` | `store()` |
| GET | `/datatable` | `backend.redirect.datatable` | DataTable `index()` |
| GET | `/{redirect}` | `backend.redirect.show` | `show()` |
| GET | `/{redirect}/edit` | `backend.redirect.edit` | `edit()` |
| POST | `/{redirect}/update` | `backend.redirect.update` | `update()` |
| DELETE | `/{redirect}/delete` | `backend.redirect.destroy` | `destroy()` |

## Views

**Namespace**: `redirect::`
**Location**: `src/Views/backend/`

| View | Purpose |
|------|---------|
| `index.blade.php` | Datatable listing page |
| `create.blade.php` | Create form |
| `edit.blade.php` | Edit form |
| `show.blade.php` | Detail view |
| `partials/form/details.blade.php` | Form fields partial |

## Translations

**Namespace**: `redirect::`
**Location**: `src/Translations/en/backend.php`

| Key | Value |
|-----|-------|
| `index` | Redirects |
| `create` | New |
| `store_success` | Redirect successfully created! |
| `update_success` | Redirect successfully updated! |
| `delete_success` | Redirect successfully deleted! |
| `delete_failed` | Unable to delete redirect! |

## Extension Points

### 1. Adding Custom Validation Rules

Edit `src/Http/Requests/StoreRedirectRequest.php`:

```php
public function rules(): array
{
    return [
        'from' => 'required|unique:redirects,from',
        'to' => 'required|url|different:from',
    ];
}
```

### 2. Adding Custom Legacy URL Handlers

Edit `src/Http/Middleware/HasRedirects.php` and add new methods:

```php
// Add detection method
protected function isACustomRedirect($request): bool
{
    return $request->filled('custom_param');
}

// Add redirect logic in handle() method before return $next($request)
if ($this->isACustomRedirect($request)) {
    $slug = $request->get('custom_param');
    return redirect()->route('frontend.custom.show', $slug, 301);
}
```

### 3. Adding Model Scopes

Edit `src/Models/Redirect.php`:

```php
public function scopeActive($query)
{
    return $query->where('active', true);
}

public function scopeForDomain($query, string $domain)
{
    return $query->where('domain', $domain);
}
```

### 4. Adding Events

Create new event class in `src/Events/`:

```php
namespace Bongo\Redirect\Events;

use Bongo\Redirect\Models\Redirect;
use Illuminate\Foundation\Events\Dispatchable;

class RedirectDeleted
{
    use Dispatchable;

    public Redirect $redirect;

    public function __construct(Redirect $redirect)
    {
        $this->redirect = $redirect;
    }
}
```

Register in `RedirectServiceProvider`:

```php
protected array $listeners = [
    RedirectDeleted::class => [ClearRedirectCache::class],
];
```

Fire in `RedirectController::destroy()`:

```php
if ($redirect->delete()) {
    event(new RedirectDeleted($redirect));
    // ...
}
```

### 5. Custom Middleware Behavior

Apply `HasRedirects` middleware to specific route groups:

```php
// In your application routes
Route::middleware(['web', 'hasRedirects'])->group(function () {
    // Your routes
});
```

### 6. Extending the Redirect Model

Add relationships, attributes, or methods:

```php
// In src/Models/Redirect.php

public function user()
{
    return $this->belongsTo(User::class);
}

public function getFullFromUrlAttribute(): string
{
    return url($this->from);
}

protected $appends = ['full_from_url'];
```

### 7. Customizing Cache Strategy

Edit `src/Listeners/ClearRedirectCache.php`:

```php
public function handle($event): void
{
    Cache::tags(['redirects'])->flush();

    // Or clear specific keys
    Cache::forget("redirect.{$event->redirect->id}");
}
```

### 8. Adding Datatable Filters

Edit `src/Http/Controllers/Backend/RedirectDatatableController.php`:

```php
protected function applyFilters()
{
    if ($this->request->filled('search')) {
        $search = $this->request->get('search');
        $this->query->where(function ($q) use ($search) {
            $q->where('from', 'like', "%{$search}%")
              ->orWhere('to', 'like', "%{$search}%");
        });
    }
}
```

## Testing Strategy

**Base Class**: `Bongo\Redirect\Tests\TestCase`
**Location**: `tests/TestCase.php`

### Test Examples

```php
namespace Bongo\Redirect\Tests;

use Bongo\Redirect\Models\Redirect;

class RedirectTest extends TestCase
{
    /** @test */
    public function it_adds_leading_slashes_to_from_attribute()
    {
        $redirect = Redirect::create([
            'from' => 'old-url',
            'to' => '/new-url',
        ]);

        $this->assertEquals('/old-url', $redirect->from);
    }

    /** @test */
    public function it_lowercases_to_attribute()
    {
        $redirect = Redirect::create([
            'from' => '/old',
            'to' => 'NEW-URL',
        ]);

        $this->assertEquals('/new-url', $redirect->to);
    }
}
```

## Dependencies and Integration

### Framework Dependencies
- `Bongo\Framework\Providers\AbstractServiceProvider`
- `Bongo\Framework\Models\AbstractModel`
- `Bongo\Framework\Http\Controllers\AbstractController`
- `Bongo\Framework\Http\Controllers\AbstractDatatableController`
- `Bongo\Framework\Helpers\URL`

### Optional Package Integrations
- **post package**: Required for legacy blog redirects
- **review package**: Required for legacy review redirects
- **page package**: Likely required for legacy page redirects

### Checking Package Availability
```php
if (package()->isEnabled('post')) {
    // Handle post-related redirect
}
```

## Performance Considerations

1. **Database Indexes**: Both `from` and `to` columns are indexed for fast lookups
2. **Caching**: Use `'redirects'` cache key for frequently accessed redirects
3. **Middleware Order**: URL normalization happens before redirect checking
4. **Query String Handling**: Preserved automatically to maintain functionality
5. **Queued Listeners**: Cache clearing is queued to avoid blocking requests

## Security Considerations

1. **301 Redirects**: Prevents redirect loops (browser caches permanent redirects)
2. **Input Sanitization**: URLs automatically normalized with leading slashes
3. **Developer Middleware**: Admin routes restricted to developers only
4. **Form Validation**: Required fields prevent empty redirects
5. **Route Model Binding**: Prevents manual ID manipulation vulnerabilities
