# Bongo Redirect Package - Claude Code Guidance

## Overview

This Laravel package provides URL redirect management with an admin interface, automatic URL normalization (www/trailing slash removal), and legacy CMS URL migration handlers. Part of the Bongo monorepo, it extends `bongo/framework`.

**Quick Links**:
- **Architecture Details**: See [ARCHITECTURE.md](ARCHITECTURE.md)
- **Cursor AI Rules**: See [.cursorrules](.cursorrules)
- **GitHub Copilot**: See [.github/copilot-instructions.md](.github/copilot-instructions.md)

**Package Info**:
- **Name**: `bongo/redirect`
- **Namespace**: `Bongo\Redirect`
- **PHP**: 8.2+ | **Laravel**: 10+
- **Repository**: https://bitbucket.org/designtec/redirect

## Available Commands

```bash
# Install dependencies
composer install

# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Update dependencies
rm -f composer.lock && composer update -W
```

## Quick Architecture Reference

### Service Provider (`src/RedirectServiceProvider.php:13-39`)
Extends `Bongo\Framework\Providers\AbstractServiceProvider`:
- **Module**: `redirect`
- **Middleware**: `removeWWW`, `removeSlash`, `hasRedirects`
- **Event Listeners**: Clears cache on `RedirectCreated` and `RedirectUpdated`
- **Auto-Bootstraps**: Routes, views, migrations, translations, config

### Middleware Chain (on 'web' group)
1. **RemoveWWW** → Redirects www.domain.com to domain.com (301)
2. **RemoveSlash** → Redirects /path/ to /path (301)
3. **HasRedirects** → Checks database + legacy URL patterns (manual application)

### Core Model (`src/Models/Redirect.php`)
```php
Bongo\Redirect\Models\Redirect extends AbstractModel
```
- **Fillable**: `from`, `to`
- **Mutators**: Auto-adds leading slashes, lowercases `to` attribute
- **Table**: `redirects` (id, from, to, timestamps)

### Controllers
- **RedirectController**: CRUD operations (index, create, store, show, edit, update, destroy)
- **RedirectDatatableController**: DataTables API (extends AbstractDatatableController)

### Events & Listeners
- **RedirectCreated** / **RedirectUpdated** → **ClearRedirectCache** (queued, 3 retries)

## Key Files

| File | Purpose |
|------|---------|
| `src/RedirectServiceProvider.php` | Service provider with middleware & listener registration |
| `src/Models/Redirect.php` | Eloquent model with attribute mutators |
| `src/Http/Middleware/HasRedirects.php` | Main redirect logic (database + legacy URLs) |
| `src/Http/Middleware/RemoveWWW.php` | WWW prefix removal (301) |
| `src/Http/Middleware/RemoveSlash.php` | Trailing slash removal (301) |
| `src/Http/Controllers/Backend/RedirectController.php` | CRUD controller |
| `src/Http/Controllers/Backend/RedirectDatatableController.php` | DataTables API |
| `src/Listeners/ClearRedirectCache.php` | Cache invalidation (queued) |
| `src/Routes/backend.php` | Admin routes (`/admin/redirects`) |
| `src/Config/redirect.php` | Configuration (prefix) |
| `src/Migrations/2021_01_01_000001_create_redirects_table.php` | Database schema |

## Request Flow

```
HTTP Request
    ↓
[RemoveWWW] → Has www? → 301 redirect
    ↓
[RemoveSlash] → Ends with /? → 301 redirect
    ↓
[HasRedirects] (if applied manually)
    ├─ Check database → Found? → 301 redirect (preserve query string)
    ├─ Check legacy pages (?p=page.name) → 301 to frontend.page.show
    ├─ Check legacy blogs (various patterns) → 301 to frontend.post.show
    └─ Check legacy reviews (?review) → 301 to frontend.review.index
    ↓
Application
```

## Admin Interface

**Routes**: All under `/admin/redirects` (configurable via `config('redirect.prefix')`)
**Middleware**: `auth`, `employee`, `developer`

| Route | Method | Name | Action |
|-------|--------|------|--------|
| `/` | GET | `backend.redirect.index` | List redirects |
| `/create` | GET | `backend.redirect.create` | Create form |
| `/store` | POST | `backend.redirect.store` | Store redirect |
| `/datatable` | GET | `backend.redirect.datatable` | DataTables API |
| `/{redirect}` | GET | `backend.redirect.show` | Show redirect |
| `/{redirect}/edit` | GET | `backend.redirect.edit` | Edit form |
| `/{redirect}/update` | POST | `backend.redirect.update` | Update redirect |
| `/{redirect}/delete` | DELETE | `backend.redirect.destroy` | Delete redirect |

## Code Style Summary

### PHP Standards
- **Strict Types**: Use `declare(strict_types=1);`
- **Return Types**: Required on all public methods
- **Type Hints**: Use PHP 8.2+ typed properties and parameters
- **Arrays**: Short syntax `[]` not `array()`

### Laravel Conventions
- **Route Model Binding**: Use implicit binding (e.g., `Redirect $redirect` parameter)
- **Form Requests**: Validation in dedicated request classes
- **Flash Messages**: `->success()` and `->error()` macros on redirects
- **Namespacing**: Views/translations use `redirect::` prefix

### Framework Integration
- **Base Classes**: Extend `Abstract*` classes from `bongo/framework`
- **Events**: Use `Dispatchable`, `InteractsWithSockets`, `SerializesModels` traits
- **Listeners**: Implement `ShouldQueue` for async processing
- **Middleware**: Return 301 status for permanent redirects

## Common Patterns

### Creating a Redirect (with Event)
```php
$redirect = $this->redirect->create($request->all());
event(new RedirectCreated($redirect));

return redirect()
    ->route('backend.redirect.show', $redirect->id)
    ->success(trans('redirect::backend.store_success'));
```

### URL Normalization (Model Mutators)
```php
// Automatically adds leading slash
public function setFromAttribute($value)
{
    if (! empty($value)) {
        $this->attributes['from'] = '/'.ltrim($value, '/');
    }
}
```

### Checking Package Availability
```php
// For optional integrations
if ($this->isABlogRedirect($request) && package()->isEnabled('post')) {
    // Handle blog redirect
}
```

### Preserving Query Strings
```php
if ($params = $request->getQueryString()) {
    return redirect()->to("{$redirect->to}?{$params}", 301);
}
```

## Framework Utilities (from bongo/framework)

```php
use Bongo\Framework\Helpers\URL;

URL::isHomePage($request);      // Check if homepage
URL::isAFile($request);         // Check if file request
URL::hasQueryString($request);  // Check for query string
URL::endsWithSlash($request);   // Check for trailing slash
URL::hasWWW($request);          // Check for www prefix
URL::withoutWWW($request);      // Get host without www
```

## Configuration

**File**: `src/Config/redirect.php`

```php
return [
    'prefix' => 'redirects',  // Admin route prefix
];
```

**Access**: `config('redirect.prefix')`

## Legacy URL Patterns

### Page Redirects
```
?p=page.name → frontend.page.show (slug: page-name)
```

### Blog Redirects (requires 'post' package)
```
?blogs=2023/01/15/post-name → frontend.post.show
/blogs/archive/2023/01/15/post-name.aspx → frontend.post.show
/blog/archive/2023/01/15/post-name → frontend.post.show
```

### Review Redirects (requires 'review' package)
```
?review → frontend.review.index
```

## Testing

**Base Class**: `Bongo\Redirect\Tests\TestCase`
**Config**: `phpunit.xml` (uses Orchestra Testbench)
**Database**: `testing` connection (in-memory SQLite)

### Example Test
```php
namespace Bongo\Redirect\Tests;

use Bongo\Redirect\Models\Redirect;

class RedirectTest extends TestCase
{
    /** @test */
    public function it_adds_leading_slashes()
    {
        $redirect = Redirect::create([
            'from' => 'old-url',
            'to' => 'new-url',
        ]);

        $this->assertEquals('/old-url', $redirect->from);
        $this->assertEquals('/new-url', $redirect->to);
    }
}
```

## Database Schema

```sql
CREATE TABLE redirects (
    id INT AUTO_INCREMENT PRIMARY KEY,
    from VARCHAR(255) INDEXED,
    to VARCHAR(255) INDEXED,
    created_at TIMESTAMP,
    updated_at TIMESTAMP
);
```

## Cache Strategy

- **Cache Key**: `'redirects'` (singular)
- **Cleared By**: `ClearRedirectCache` listener (queued, 3 retry attempts)
- **Triggered On**: `RedirectCreated`, `RedirectUpdated`, and manual delete
- **Implementation**: `Cache::forget('redirects')`

## Important Notes

1. **Middleware Order**: RemoveWWW → RemoveSlash → HasRedirects (matters!)
2. **Database First**: Database redirects take precedence over legacy handlers
3. **301 Redirects**: All redirects use 301 (permanent) for SEO
4. **Query Preservation**: Query strings automatically preserved
5. **Event-Driven**: Always fire events after create/update for cache consistency
6. **Developer Access**: Admin routes require `developer` middleware
7. **Route Naming**: All routes prefixed with `backend.redirect.*`
8. **View Namespace**: All views use `redirect::` prefix
9. **Translation Keys**: All translations use `redirect::backend.*` format
10. **Leading Slashes**: Automatically added by model mutators

## Extension Points

See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed extension examples:
- Adding custom validation rules
- Adding custom legacy URL handlers
- Adding model scopes and relationships
- Adding new events and listeners
- Customizing middleware behavior
- Extending datatable filters
- Custom cache strategies

## Dependencies

- `bongo/framework` ^3.0 (provides base classes and utilities)
- `illuminate/contracts` ^10.0
- PHP 8.2+, Laravel 10+

## Development Workflow

1. **Read the code** - Use Read tool to understand existing implementation
2. **Follow patterns** - Match existing code style and structure
3. **Test changes** - Run `vendor/bin/phpunit`
4. **Fix style** - Run `vendor/bin/pint`
5. **Fire events** - Always trigger events after create/update operations
6. **Clear cache** - Ensure cache invalidation works correctly

## Notes for AI Assistants

- Always use route model binding instead of `findOrFail()`
- The `$module` property must match config filename and view namespace
- Middleware is registered but manually applied to 'web' group in `boot()`
- Flash messages use custom macros: `->success()` and `->error()`
- Events fire after database operations to ensure cache consistency
- Query strings are preserved when performing redirects
- All redirects use 301 status for permanent redirection
- The datatable controller handles pagination/filtering automatically
- Package availability checks use `package()->isEnabled('package-name')`
