# Bongo Redirect Package

Provides URL redirect management and legacy URL migration capabilities for Laravel applications. Includes an admin CRUD interface, automatic URL normalization middleware, and built-in handlers for legacy Bongo CMS URLs.

## Features

- **Database-Driven Redirects**: Manage redirects through an admin interface
- **Legacy URL Migration**: Built-in handlers for old Bongo CMS page, blog, and review URLs
- **URL Normalization**: Automatic www removal and trailing slash handling (301 redirects)
- **Event-Driven Cache**: Automatic cache invalidation on redirect changes
- **SEO Friendly**: All redirects use 301 (permanent) status codes
- **Query String Preservation**: Maintains query parameters during redirects

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0

## Installation

### Composer

Install via Composer from the private repository:

```bash
composer require bongo/redirect
```

### Configuration

The package uses auto-discovery in Laravel 5.5+. For older versions, manually register the service provider:

```php
// config/app.php
'providers' => [
    // ...
    Bongo\Redirect\RedirectServiceProvider::class,
],
```

### Publish Configuration (Optional)

```bash
php artisan vendor:publish --provider="Bongo\Redirect\RedirectServiceProvider"
```

### Run Migrations

```bash
php artisan migrate
```

This creates the `redirects` table with the following schema:
- `id` - Primary key
- `from` - Source URL path (indexed)
- `to` - Target URL path (indexed)
- `created_at` / `updated_at` - Timestamps

## Usage

### Admin Interface

Navigate to `/admin/redirects` (requires authentication and developer permissions) to:
- View all redirects in a datatable
- Create new redirects
- Edit existing redirects
- Delete redirects

### Creating Redirects Programmatically

```php
use Bongo\Redirect\Models\Redirect;
use Bongo\Redirect\Events\RedirectCreated;

$redirect = Redirect::create([
    'from' => '/old-page',
    'to' => '/new-page',
]);

event(new RedirectCreated($redirect));
```

**Note**: Leading slashes are automatically added by model mutators, and the `to` URL is lowercased.

### Middleware

The package provides three middleware components:

#### 1. RemoveWWW
Redirects `www.domain.com` → `domain.com` (301)

**Applied**: Automatically to 'web' middleware group

#### 2. RemoveSlash
Redirects `/path/` → `/path` (301)

**Applied**: Automatically to 'web' middleware group

**Conditions**:
- GET requests only
- Not the homepage
- Not a file request
- No query string present
- Ends with trailing slash

#### 3. HasRedirects
Main redirect logic - checks database and legacy URL patterns

**Applied**: Manually via middleware alias `hasRedirects`

```php
// In your routes file
Route::middleware(['web', 'hasRedirects'])->group(function () {
    // Your routes
});
```

**Redirect Priority**:
1. Database redirects (from `redirects` table)
2. Legacy page URLs (`?p=page.name`)
3. Legacy blog URLs (multiple patterns)
4. Legacy review URLs (`?review`)

### Legacy URL Patterns

#### Page Redirects
```
?p=page.name → frontend.page.show (slug: page-name)
```

#### Blog Redirects (requires 'post' package)
```
?blogs=2023/01/15/post-name → frontend.post.show
?blog=2023/01/15/post-name → frontend.post.show
/blogs/archive/2023/01/15/post-name.aspx → frontend.post.show
/blog/archive/2023/01/15/post-name.aspx → frontend.post.show
```

#### Review Redirects (requires 'review' package)
```
?review → frontend.review.index
```

### Configuration

The package has minimal configuration:

```php
// config/redirect.php
return [
    'prefix' => 'redirects',  // Admin route prefix
];
```

Change this to customize the admin URL:

```php
'prefix' => 'custom-redirects',  // /admin/custom-redirects
```

### Events

The package fires two events:

- **RedirectCreated**: Fired when a redirect is created
- **RedirectUpdated**: Fired when a redirect is updated

Both events trigger the `ClearRedirectCache` listener (queued) which clears the `'redirects'` cache key.

### Routes

All admin routes are named with the `backend.redirect.*` prefix:

| Route Name | URL | Description |
|------------|-----|-------------|
| `backend.redirect.index` | `/admin/redirects` | List redirects |
| `backend.redirect.create` | `/admin/redirects/create` | Create form |
| `backend.redirect.store` | `/admin/redirects/store` | Store redirect |
| `backend.redirect.datatable` | `/admin/redirects/datatable` | DataTables API |
| `backend.redirect.show` | `/admin/redirects/{redirect}` | Show redirect |
| `backend.redirect.edit` | `/admin/redirects/{redirect}/edit` | Edit form |
| `backend.redirect.update` | `/admin/redirects/{redirect}/update` | Update redirect |
| `backend.redirect.destroy` | `/admin/redirects/{redirect}/delete` | Delete redirect |

**Middleware**: `auth`, `employee`, `developer`

## Testing

Run the package tests:

```bash
vendor/bin/phpunit
```

The package uses Orchestra Testbench for isolated testing.

## Code Style

The package follows Laravel conventions and uses Laravel Pint for code style enforcement:

```bash
# Check code style
vendor/bin/pint --test

# Fix code style issues
vendor/bin/pint
```

## Documentation

- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for Claude Code
- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation
- **[.cursorrules](.cursorrules)** - Cursor AI instructions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions

## Architecture Overview

### Service Provider
Extends `Bongo\Framework\Providers\AbstractServiceProvider` which provides automatic bootstrapping:
- Config from `src/Config/redirect.php`
- Routes from `src/Routes/backend.php`
- Views from `src/Views/` (namespace: `redirect::`)
- Migrations from `src/Migrations/`
- Translations from `src/Translations/` (namespace: `redirect::`)

### Middleware Chain
```
Request → RemoveWWW → RemoveSlash → HasRedirects → Application
```

### Request Flow
```
HasRedirects Middleware
    ├─ Check database (redirects table)
    │   └─ Found? → 301 redirect (preserve query string)
    │
    ├─ Check legacy pages (?p=page.name)
    │   └─ Match? → 301 to frontend.page.show
    │
    ├─ Check legacy blogs (multiple patterns)
    │   └─ Match? → 301 to frontend.post.show
    │
    └─ Check legacy reviews (?review)
        └─ Match? → 301 to frontend.review.index
```

### Cache Strategy
- Cache key: `'redirects'`
- Cleared automatically via queued listener on create/update/delete
- 3 retry attempts for failed cache clears

## Dependencies

This package depends on:
- **bongo/framework** - Provides base classes and utilities
  - `AbstractServiceProvider` - Auto-bootstrapping service provider
  - `AbstractModel` - Base Eloquent model
  - `AbstractController` - Base controller
  - `AbstractDatatableController` - DataTables controller
  - `URL` helper class - URL manipulation utilities

## Contributing

This package is part of the Bongo monorepo. Each package is a separate Git repository.

## License

The MIT License (MIT). Please see [LICENSE](LICENSE) for more information.

## Credits

- **Stuart Elliott** - [Bespoke UK](https://bespokeuk.com)

## Support

For issues and questions:
- Repository: https://bitbucket.org/designtec/redirect
- Email: stuart.elliott@bespokeuk.com
