<?php

namespace Bongo\Redirect\Http\Middleware;

use Bongo\Redirect\Models\Redirect;
use Closure;
use Illuminate\Support\Str;

class HasRedirects
{
    public function handle($request, Closure $next)
    {
        // Have we already handled this redirect
        if ($redirect = $this->existsInRedirectsTable($request)) {

            // Does this redirect have any params
            if ($params = $request->getQueryString()) {
                return redirect()->to("{$redirect->to}?{$params}", 301);
            }

            return redirect()->to($redirect->to, 301);
        }

        // Check if this is an old bongo page url
        if ($this->isAPageRedirect($request)) {
            $slug = str_replace('.', '-', $request->get('p'));

            return redirect()->route('frontend.page.show', Str::slug($slug), 301);
        }

        // Check if this is an old bongo blog url
        if ($this->isABlogRedirect($request) && package()->isEnabled('post')) {
            $slug = $this->removeBlogsPrefix($request);
            $slug = $this->removeFileExtension($slug);

            // Check if the url format is /YYYY/MM/DD/{name}
            $yearMonthDaySlugRegex = '/^([0-9]{4})\/([0-9]{2})\/([0-9]{2})(.*)?$/';
            if (preg_match_all($yearMonthDaySlugRegex, $slug, $matches)) {

                // If the 4th match is not empty, we have a slug
                if (isset($matches[4])) {
                    $slug = is_array($matches[4]) ? $matches[4][0] : $matches[4];
                    $slug = str_replace('/', '', $slug);
                    $slug = str_replace('.', '-', $slug);

                    return redirect()->route('frontend.post.show', Str::slug($slug), 301);
                }
            }
        }

        // Check if this is an old bongo review url
        if ($this->isAReviewRedirect($request) && package()->isEnabled('review')) {
            return redirect()->route('frontend.review.index', [], 301);
        }

        return $next($request);
    }

    protected function existsInRedirectsTable($request): ?Redirect
    {
        return Redirect::query()
            ->where('from', $request->getPathInfo())
            ->first();
    }

    protected function isAPageRedirect($request): bool
    {
        return $request->filled('p');
    }

    protected function isABlogRedirect($request): bool
    {
        return $request->filled('blogs')
            || $request->filled('blog')
            || Str::startsWith($request->getPathInfo(), '/blogs/archive/')
            || Str::startsWith($request->getPathInfo(), '/blog/archive/');
    }

    protected function isAReviewRedirect($request): bool
    {
        return array_key_exists('review', $request->query() ?? []);
    }

    protected function removeBlogsPrefix($request): string
    {
        if ($request->filled('blogs')) {
            return ltrim($request->get('blogs'), 'blogs/archive/');
        }
        if ($request->filled('blog')) {
            return ltrim($request->get('blog'), 'blogs/archive/');
        }
        if (Str::startsWith($request->getPathInfo(), '/blogs/archive/')) {
            return ltrim($request->getPathInfo(), '/blogs/archive/');
        }

        return ltrim($request->getPathInfo(), '/blog/archive/');
    }

    protected function removeFileExtension(string $slug): string
    {
        if (Str::endsWith($slug, '.aspx')) {
            return rtrim($slug, '.aspx');
        }

        return rtrim($slug, '-aspx');
    }
}
