<?php

namespace Bongo\Referrer\Services;

use Bongo\Framework\Helpers\URL;
use Illuminate\Contracts\Session\Session;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class Referrer
{
    protected string $sessionKey;

    protected Session $session;

    public function __construct(Session $session)
    {
        $this->sessionKey = config('referrer.session_key', 'bongo_referrer');
        $this->session = $session;
    }

    public function get(): string
    {
        return $this->session->get($this->sessionKey, '');
    }

    public function forget(): void
    {
        $this->session->forget($this->sessionKey);
    }

    public function put(string $referer): void
    {
        $this->session->put($this->sessionKey, $referer);
    }

    public function putFromRequest(Request $request): void
    {
        $referer = $request->header('referer', '');
        if (empty($referer)) {
            return;
        }

        // Extract the host from the referrer
        $refererHost = URL::getHost($referer);
        if (empty($refererHost)) {
            return;
        }

        // Make sure the referrer host is not this app
        if ($refererHost === $request->getHost()) {
            return;
        }

        // Next see if the url contains query string params
        if (! empty($request->getQueryString())) {
            $this->put($refererHost.'?'.$request->getQueryString());

            return;
        }

        // Otherwise just store the referer
        $this->put($refererHost);
    }

    public function isGoogle(): bool
    {
        return Str::contains($this->get(), 'google');
    }

    public function isGoogleAds(): bool
    {
        return Str::contains($this->get(), ['gclid']);
    }

    public function isFacebookAds(): bool
    {
        return Str::contains($this->get(), 'facebook');
    }

    public function isLinkedinAds(): bool
    {
        return Str::contains($this->get(), 'linkedin');
    }
}
