# GitHub Copilot Instructions - Bongo Referrer

## Project Overview

This is a Laravel package that captures and tracks HTTP referrer information at the application level. It stores the first external referrer in the session and provides detection methods for common advertising platforms (Google Ads, Facebook Ads, LinkedIn Ads).

## Key Classes and Relationships

### 1. Service Provider
**Class**: `Bongo\Referrer\ReferrerServiceProvider`
- Extends `Bongo\Framework\Providers\AbstractServiceProvider`
- Registers `Referrer` service as singleton
- Loads helper functions after app boots

### 2. Core Service
**Class**: `Bongo\Referrer\Services\Referrer`

**Dependencies**:
- `Illuminate\Contracts\Session\Session` (injected)
- `Bongo\Framework\Helpers\URL::getHost()` (static call)

**Properties**:
```php
protected string $sessionKey;  // From config('referrer.session_key')
protected Session $session;    // Injected Laravel session
```

**Public Methods**:
```php
__construct(Session $session): void
get(): string                              // Retrieve referrer from session
put(string $referer): void                // Store referrer in session
forget(): void                             // Clear referrer from session
putFromRequest(Request $request): void    // Capture referrer from HTTP request
isGoogle(): bool                          // Check if referrer contains 'google'
isGoogleAds(): bool                       // Check for 'gclid' parameter
isFacebookAds(): bool                     // Check if referrer contains 'facebook'
isLinkedinAds(): bool                     // Check if referrer contains 'linkedin'
```

**Logic Flow for `putFromRequest()`**:
1. Extract `referer` header from request
2. Parse hostname from referrer URL
3. Filter out same-domain traffic (internal navigation)
4. Append query string if present (for UTM/ad tracking)
5. Store hostname in session

### 3. Middleware
**Class**: `Bongo\Referrer\Http\Middleware\CaptureReferrer`

**Dependencies**:
- `Bongo\Referrer\Services\Referrer` (injected)

**Method**:
```php
public function handle($request, Closure $next)
{
    $this->referrer->putFromRequest($request);
    return $next($request);
}
```

### 4. Helper Functions
**File**: `src/helpers.php`

```php
referrer(): Referrer         // Get Referrer service instance
get_referrer(): string       // Get referrer string directly
```

## Architecture Diagram

```
┌─────────────────────────────────────┐
│  HTTP Request                       │
│  Header: Referer: google.com        │
└──────────────┬──────────────────────┘
               │
               ▼
┌─────────────────────────────────────┐
│  CaptureReferrer Middleware         │
│  - Intercepts request               │
│  - Calls putFromRequest()           │
└──────────────┬──────────────────────┘
               │
               ▼
┌─────────────────────────────────────┐
│  Referrer Service                   │
│  - Extract hostname                 │
│  - Filter same-domain traffic       │
│  - Preserve query strings           │
│  - Store in session                 │
└──────────────┬──────────────────────┘
               │
               ▼
┌─────────────────────────────────────┐
│  Session Storage                    │
│  Key: bongo_referrer                │
│  Value: "google.com?gclid=..."      │
└─────────────────────────────────────┘
```

## Code Style Templates

### Adding Platform Detection Method
```php
public function isTwitterAds(): bool
{
    return Str::contains($this->get(), 'twitter');
}

// For multiple keywords:
public function isBingAds(): bool
{
    return Str::contains($this->get(), ['bing', 'msclkid']);
}
```

### Service Method Template
```php
public function methodName(): returnType
{
    // Implementation
}
```

### Middleware Template
```php
public function handle($request, Closure $next)
{
    // Pre-processing
    $this->referrer->someAction($request);

    return $next($request);
}
```

## Common Patterns

### 1. Type Declarations
Always use strict types and full type hints:
```php
<?php

declare(strict_types = 1);

namespace Bongo\Referrer\Services;

class Example
{
    protected string $property;

    public function method(string $param): bool
    {
        // ...
    }
}
```

### 2. Session Interaction
```php
// Get from session
$value = $this->session->get($this->sessionKey, 'default');

// Store in session
$this->session->put($this->sessionKey, $value);

// Remove from session
$this->session->forget($this->sessionKey);
```

### 3. String Detection Pattern
```php
// Single keyword
if (Str::contains($string, 'keyword')) {
    // ...
}

// Multiple keywords (any match)
if (Str::contains($string, ['keyword1', 'keyword2'])) {
    // ...
}
```

### 4. URL Parsing
```php
use Bongo\Framework\Helpers\URL;

$host = URL::getHost($url);  // Extract hostname from URL
```

### 5. Config Access
```php
$value = config('referrer.session_key', 'default_value');
```

### 6. Helper Function Pattern
```php
if (! function_exists('function_name')) {
    function function_name(): ReturnType
    {
        return app(ServiceClass::class);
    }
}
```

## Configuration

**File**: `src/Config/referrer.php`
```php
return [
    'session_key' => 'bongo_referrer',  // Session storage key
];
```

## Testing Patterns

### Base Test Case
```php
namespace Bongo\Referrer\Tests;

class ExampleTest extends TestCase
{
    /** @test */
    public function it_does_something()
    {
        // Arrange

        // Act

        // Assert
    }
}
```

### Service Provider Registration
Already handled in `TestCase::getPackageProviders()`:
```php
protected function getPackageProviders(Application $app): array
{
    return [
        ReferrerServiceProvider::class,
    ];
}
```

## Dependencies

### Framework
- `bongo/framework` - Provides `AbstractServiceProvider` and `URL` helper

### Laravel
- `illuminate/contracts` - Session contract, Request
- `illuminate/support` - Str helper, Service Provider

## Naming Conventions

- **Classes**: PascalCase, singular nouns (`Referrer`)
- **Methods**: camelCase, descriptive verbs (`putFromRequest`)
- **Detection methods**: `is*()` prefix (`isGoogle`, `isGoogleAds`)
- **Properties**: camelCase with type hints (`protected string $sessionKey`)
- **Config files**: Lowercase, match module name (`referrer.php`)
- **Helper functions**: snake_case (`get_referrer`)

## File Locations

- **Service classes**: `src/Services/`
- **Middleware**: `src/Http/Middleware/`
- **Config**: `src/Config/{module}.php`
- **Helpers**: `src/helpers.php`
- **Service Provider**: `src/{Package}ServiceProvider.php`
- **Tests**: `tests/`

## Important Notes

1. **Strict types required**: Always include `declare(strict_types = 1);`
2. **Session dependency**: Service requires active session
3. **First referrer only**: Captures initial external referrer, ignores subsequent
4. **Query string preservation**: Maintains UTM and ad tracking parameters
5. **Same-domain filtering**: Automatically excludes internal navigation
6. **Singleton service**: Registered as singleton in container
7. **Host-only storage**: Stores hostname + query string, not full URL path
