# Architecture Documentation - Bongo Referrer

## Overview

The Bongo Referrer package provides session-based HTTP referrer tracking for Laravel applications. It captures the first external referrer (traffic source) and provides detection methods for common advertising platforms.

**Purpose**: Track where visitors come from (Google, Facebook, LinkedIn, etc.) and preserve ad tracking parameters (UTM, gclid, etc.) for conversion attribution.

## Directory Structure

```
referrer/
├── src/
│   ├── Config/
│   │   └── referrer.php                  # Configuration file (session key)
│   ├── Http/
│   │   └── Middleware/
│   │       └── CaptureReferrer.php       # Request middleware to capture referrer
│   ├── Services/
│   │   └── Referrer.php                  # Core service (session management + detection)
│   ├── ReferrerServiceProvider.php       # Laravel service provider (registers services)
│   └── helpers.php                       # Global helper functions
├── tests/
│   └── TestCase.php                      # Base test case for package tests
├── composer.json                          # Package dependencies and metadata
├── phpunit.xml                            # PHPUnit configuration
└── README.md                              # Package documentation
```

## Core Components

### 1. Service Provider (`ReferrerServiceProvider`)

**File**: `src/ReferrerServiceProvider.php`

**Responsibility**: Bootstrap the package and register services.

**Inheritance**:
```
Illuminate\Support\ServiceProvider
    ↑
Bongo\Framework\Providers\AbstractServiceProvider
    ↑
Bongo\Referrer\ReferrerServiceProvider
```

**Key Properties**:
```php
protected string $module = 'referrer';  // Used for config/view namespacing
```

**Registration Logic** (in `boot()` method):
```php
public function boot(): void
{
    parent::boot();  // Calls AbstractServiceProvider::boot()
                     // - Registers config from src/Config/referrer.php
                     // - Registers routes (if any)
                     // - Registers views (if any)

    // Singleton binding
    $this->app->singleton(Referrer::class);
    $this->app->alias(Referrer::class, 'referrer');

    // Load helpers after app boots
    $this->app->booted(function () {
        include __DIR__.'/helpers.php';
    });
}
```

**What AbstractServiceProvider provides**:
- Auto-loads config from `src/Config/referrer.php`
- Auto-registers routes from `src/Routes/*.php` (none in this package)
- Auto-loads views from `src/Views/referrer/` (none in this package)
- Auto-loads migrations from `src/Migrations/` (none in this package)
- Auto-loads translations from `src/Translations/` (none in this package)

### 2. Core Service (`Referrer`)

**File**: `src/Services/Referrer.php`

**Responsibility**: Manage referrer data in session and provide detection methods.

**Dependencies**:
```php
use Bongo\Framework\Helpers\URL;           // For URL parsing
use Illuminate\Contracts\Session\Session;  // For session storage
use Illuminate\Http\Request;               // For request handling
use Illuminate\Support\Str;                // For string operations
```

**Properties**:
```php
protected string $sessionKey;   // Session key from config (default: 'bongo_referrer')
protected Session $session;     // Laravel session instance
```

**Constructor**:
```php
public function __construct(Session $session)
{
    $this->sessionKey = config('referrer.session_key', 'bongo_referrer');
    $this->session = $session;
}
```

**Public API**:

| Method | Return Type | Description |
|--------|-------------|-------------|
| `get()` | `string` | Retrieve current referrer from session |
| `put(string $referer)` | `void` | Manually store referrer in session |
| `forget()` | `void` | Clear referrer from session |
| `putFromRequest(Request $request)` | `void` | Extract and store referrer from HTTP request |
| `isGoogle()` | `bool` | Check if referrer contains 'google' |
| `isGoogleAds()` | `bool` | Check if referrer contains 'gclid' (Google Ads click ID) |
| `isFacebookAds()` | `bool` | Check if referrer contains 'facebook' |
| `isLinkedinAds()` | `bool` | Check if referrer contains 'linkedin' |

### 3. Middleware (`CaptureReferrer`)

**File**: `src/Http/Middleware/CaptureReferrer.php`

**Responsibility**: Intercept incoming requests and capture external referrers.

**Dependencies**:
```php
use Bongo\Referrer\Services\Referrer;  // The core service
use Closure;                            // For middleware pattern
```

**Implementation**:
```php
public function __construct(Referrer $referrer)
{
    $this->referrer = $referrer;
}

public function handle($request, Closure $next)
{
    $this->referrer->putFromRequest($request);
    return $next($request);
}
```

**Middleware Flow**:
```
Request → CaptureReferrer → Application → Response
            ↓
       Referrer::putFromRequest()
            ↓
       Session Storage
```

### 4. Helper Functions

**File**: `src/helpers.php`

**Functions**:
```php
referrer(): Referrer
// Returns singleton instance of Referrer service
// Usage: referrer()->get()

get_referrer(): string
// Shortcut to get referrer string directly
// Usage: $source = get_referrer();
```

**Pattern**: Global functions with existence checks:
```php
if (! function_exists('referrer')) {
    function referrer(): Referrer
    {
        return app(Referrer::class);
    }
}
```

## Data Flow Diagrams

### Referrer Capture Flow

```
┌─────────────────────────────────────────────────────────────┐
│ 1. HTTP Request arrives                                     │
│    Header: Referer: https://google.com/search?q=laravel     │
└────────────────────────┬────────────────────────────────────┘
                         │
                         ▼
┌─────────────────────────────────────────────────────────────┐
│ 2. CaptureReferrer Middleware                               │
│    handle($request, $next)                                  │
│    - Calls: $this->referrer->putFromRequest($request)       │
└────────────────────────┬────────────────────────────────────┘
                         │
                         ▼
┌─────────────────────────────────────────────────────────────┐
│ 3. Referrer::putFromRequest()                               │
│    ┌────────────────────────────────────────────────────┐   │
│    │ a) Extract referer header                          │   │
│    │    $referer = $request->header('referer', '')      │   │
│    │    → "https://google.com/search?q=laravel"         │   │
│    └────────────────────────────────────────────────────┘   │
│    ┌────────────────────────────────────────────────────┐   │
│    │ b) Parse hostname                                  │   │
│    │    $refererHost = URL::getHost($referer)           │   │
│    │    → "google.com"                                  │   │
│    └────────────────────────────────────────────────────┘   │
│    ┌────────────────────────────────────────────────────┐   │
│    │ c) Filter same-domain traffic                      │   │
│    │    if ($refererHost === $request->getHost())       │   │
│    │        return; // Ignore internal navigation       │   │
│    └────────────────────────────────────────────────────┘   │
│    ┌────────────────────────────────────────────────────┐   │
│    │ d) Preserve query string (ad tracking params)      │   │
│    │    if (! empty($request->getQueryString()))        │   │
│    │        $this->put($refererHost.'?'.$queryString)   │   │
│    │    → "google.com?utm_source=adwords&gclid=..."     │   │
│    └────────────────────────────────────────────────────┘   │
└────────────────────────┬────────────────────────────────────┘
                         │
                         ▼
┌─────────────────────────────────────────────────────────────┐
│ 4. Session Storage                                          │
│    $session->put('bongo_referrer', 'google.com?gclid=...')  │
└─────────────────────────────────────────────────────────────┘
```

### Referrer Retrieval Flow

```
┌──────────────────────────────────────┐
│ Controller/View needs referrer data  │
└──────────────┬───────────────────────┘
               │
               ├─────────────────┬──────────────────┐
               ▼                 ▼                  ▼
┌──────────────────┐  ┌──────────────────┐  ┌──────────────┐
│ referrer()->get()│  │ get_referrer()   │  │ app(Referrer │
│                  │  │                  │  │ ::class)     │
└────────┬─────────┘  └────────┬─────────┘  └──────┬───────┘
         │                     │                   │
         └─────────────────────┴───────────────────┘
                               │
                               ▼
               ┌───────────────────────────────┐
               │ Referrer::get()               │
               │ return $session->get(         │
               │     $sessionKey, ''           │
               │ )                             │
               └───────────────┬───────────────┘
                               │
                               ▼
               ┌───────────────────────────────┐
               │ Session Storage                │
               │ Returns: "google.com?gclid..." │
               └───────────────────────────────┘
```

### Platform Detection Flow

```
┌────────────────────────────────────┐
│ referrer()->isGoogleAds()          │
└──────────────┬─────────────────────┘
               │
               ▼
┌────────────────────────────────────┐
│ Referrer::isGoogleAds()            │
│ {                                  │
│   return Str::contains(            │
│       $this->get(),                │
│       ['gclid']                    │
│   );                               │
│ }                                  │
└──────────────┬─────────────────────┘
               │
               ├──────────────┬──────────────┐
               ▼              ▼              ▼
┌──────────────────┐  ┌────────────┐  ┌────────────┐
│ get() returns    │  │ Contains   │  │ Return     │
│ "google.com?     │→ │ "gclid"?   │→ │ true/false │
│ gclid=abc123"    │  │            │  │            │
└──────────────────┘  └────────────┘  └────────────┘
```

## Class Relationships

### Dependency Injection Graph

```
ReferrerServiceProvider
    │
    ├─→ registers: Referrer (singleton)
    │       │
    │       └─→ requires: Session (Laravel)
    │
    └─→ loads: helpers.php
            │
            └─→ exposes: referrer(), get_referrer()

CaptureReferrer (Middleware)
    │
    └─→ requires: Referrer (injected)
```

### Inheritance Hierarchy

```
Illuminate\Support\ServiceProvider
    ↑
Bongo\Framework\Providers\AbstractServiceProvider
    ↑
Bongo\Referrer\ReferrerServiceProvider
```

```
Orchestra\Testbench\TestCase
    ↑
Bongo\Referrer\Tests\TestCase
```

## Configuration Reference

### Config File Structure

**File**: `src/Config/referrer.php`

```php
return [
    'session_key' => 'bongo_referrer',  // Key used to store referrer in session
];
```

**Access Pattern**:
```php
$sessionKey = config('referrer.session_key', 'bongo_referrer');
```

**Publishing Config** (in consuming application):
```bash
php artisan vendor:publish --provider="Bongo\Referrer\ReferrerServiceProvider" --tag=config
```

## Extension Points

### 1. Adding New Platform Detection

**Location**: `src/Services/Referrer.php`

**Pattern**:
```php
public function isPlatformName(): bool
{
    return Str::contains($this->get(), 'keyword');
}
```

**Example - Add Twitter Ads Detection**:
```php
public function isTwitterAds(): bool
{
    return Str::contains($this->get(), ['twitter', 'twclid']);
}
```

**Multi-keyword Detection**:
```php
public function isBingAds(): bool
{
    return Str::contains($this->get(), ['bing', 'msclkid']);
}
```

### 2. Custom Storage Backend

**Current**: Session-based storage via `Illuminate\Contracts\Session\Session`

**To Use Different Storage** (e.g., Redis, Database):

Option A - Override methods:
```php
class CustomReferrer extends Referrer
{
    public function get(): string
    {
        // Custom retrieval logic
        return Redis::get('referrer:'.$userId) ?? '';
    }

    public function put(string $referer): void
    {
        // Custom storage logic
        Redis::set('referrer:'.$userId, $referer);
    }

    public function forget(): void
    {
        // Custom deletion logic
        Redis::del('referrer:'.$userId);
    }
}
```

Option B - Create new service implementing same interface:
```php
class DatabaseReferrer
{
    public function get(): string { /* ... */ }
    public function put(string $referer): void { /* ... */ }
    public function forget(): void { /* ... */ }
    public function putFromRequest(Request $request): void { /* ... */ }
    public function isGoogle(): bool { /* ... */ }
    // ... other methods
}

// In service provider:
$this->app->singleton(Referrer::class, DatabaseReferrer::class);
```

### 3. Customising Referrer Capture Logic

**Location**: `src/Services/Referrer.php` → `putFromRequest()` method

**Current Logic**:
1. Extract `referer` header from request (line 39)
2. Parse hostname using `URL::getHost()` (line 45)
3. Filter out same-domain traffic (lines 51-53)
4. Append query string if present (lines 56-60)
5. Store hostname only (line 63)

**Customisation Points**:

**A. Store full URL instead of hostname**:
```php
public function putFromRequest(Request $request): void
{
    $referer = $request->header('referer', '');
    if (empty($referer)) {
        return;
    }

    $refererHost = URL::getHost($referer);
    if (empty($refererHost) || $refererHost === $request->getHost()) {
        return;
    }

    // Store full URL instead of just hostname
    $this->put($referer);
}
```

**B. Allowlist specific domains**:
```php
protected array $allowedDomains = ['google.com', 'facebook.com', 'linkedin.com'];

public function putFromRequest(Request $request): void
{
    $referer = $request->header('referer', '');
    if (empty($referer)) {
        return;
    }

    $refererHost = URL::getHost($referer);

    // Only store if from allowed domains
    if (! Str::contains($refererHost, $this->allowedDomains)) {
        return;
    }

    $this->put($refererHost);
}
```

**C. Override existing referrer**:
```php
public function putFromRequest(Request $request): void
{
    // ... existing logic ...

    // Current implementation only stores if empty
    // To always update, change:
    $this->put($refererHost);

    // Instead of checking if already exists
}
```

### 4. Adding Middleware to Routes

**In consuming application**:

**Option A - Global Middleware**:
```php
// app/Http/Kernel.php
protected $middleware = [
    // ...
    \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
];
```

**Option B - Middleware Group**:
```php
// app/Http/Kernel.php
protected $middlewareGroups = [
    'web' => [
        // ...
        \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
    ],
];
```

**Option C - Route-specific**:
```php
// routes/web.php
Route::middleware([\Bongo\Referrer\Http\Middleware\CaptureReferrer::class])
    ->group(function () {
        Route::get('/landing-page', [LandingController::class, 'index']);
    });
```

**Option D - Named Middleware**:
```php
// app/Http/Kernel.php
protected $middlewareAliases = [
    'capture.referrer' => \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
];

// routes/web.php
Route::middleware('capture.referrer')->get('/landing', /* ... */);
```

## How to Add New Features

### Example: Add Referrer Expiry

**Requirement**: Clear referrer after 30 days.

**Step 1** - Add timestamp to stored data:
```php
// src/Services/Referrer.php

public function put(string $referer): void
{
    $data = [
        'referrer' => $referer,
        'captured_at' => now()->timestamp,
    ];
    $this->session->put($this->sessionKey, $data);
}
```

**Step 2** - Check expiry on retrieval:
```php
public function get(): string
{
    $data = $this->session->get($this->sessionKey);

    if (! is_array($data)) {
        return '';
    }

    // Check if expired (30 days = 2592000 seconds)
    if (now()->timestamp - $data['captured_at'] > 2592000) {
        $this->forget();
        return '';
    }

    return $data['referrer'] ?? '';
}
```

**Step 3** - Add configuration:
```php
// src/Config/referrer.php
return [
    'session_key' => 'bongo_referrer',
    'expiry_days' => 30,  // New config
];

// Update get() method:
$expirySeconds = config('referrer.expiry_days', 30) * 86400;
if (now()->timestamp - $data['captured_at'] > $expirySeconds) {
    // ...
}
```

### Example: Add Event Dispatching

**Requirement**: Dispatch event when referrer is captured.

**Step 1** - Create event class:
```php
// src/Events/ReferrerCaptured.php
namespace Bongo\Referrer\Events;

class ReferrerCaptured
{
    public string $referrer;
    public string $host;

    public function __construct(string $referrer, string $host)
    {
        $this->referrer = $referrer;
        $this->host = $host;
    }
}
```

**Step 2** - Dispatch event:
```php
// src/Services/Referrer.php
use Bongo\Referrer\Events\ReferrerCaptured;

public function putFromRequest(Request $request): void
{
    // ... existing logic ...

    $this->put($refererHost);

    // Dispatch event
    event(new ReferrerCaptured($refererHost, $request->getHost()));
}
```

**Step 3** - Register event in service provider:
```php
// src/ReferrerServiceProvider.php
protected array $listen = [
    ReferrerCaptured::class => [
        // Listener classes
    ],
];
```

## Testing Architecture

### Test Structure

```
tests/
└── TestCase.php          # Base test case (extends Orchestra\Testbench\TestCase)
```

**Base Class**: `Bongo\Referrer\Tests\TestCase`

**Responsibilities**:
- Register `ReferrerServiceProvider`
- Set up test environment
- Provide base for package tests

**Implementation**:
```php
protected function getPackageProviders(Application $app): array
{
    return [
        ReferrerServiceProvider::class,  // Auto-loads package
    ];
}

protected function getEnvironmentSetUp(Application $app)
{
    // Configure test environment
}
```

### Example Test Structure

```php
namespace Bongo\Referrer\Tests;

class ReferrerTest extends TestCase
{
    /** @test */
    public function it_captures_external_referrer()
    {
        // Arrange
        $request = Request::create('/test', 'GET', [], [], [], [
            'HTTP_REFERER' => 'https://google.com',
        ]);

        // Act
        app(Referrer::class)->putFromRequest($request);

        // Assert
        $this->assertEquals('google.com', referrer()->get());
    }

    /** @test */
    public function it_detects_google_ads()
    {
        // Arrange
        referrer()->put('google.com?gclid=abc123');

        // Act & Assert
        $this->assertTrue(referrer()->isGoogleAds());
    }
}
```

## Performance Considerations

1. **Session Storage**: Fast for small data, single read/write per request
2. **Singleton Service**: One instance per request lifecycle (efficient)
3. **Middleware Position**: Place early in middleware stack to capture all traffic
4. **No Database Queries**: Session-only, no additional DB load
5. **String Operations**: Uses optimised Laravel `Str::contains()`

## Security Considerations

1. **No Full URL Storage**: Only hostname stored (privacy-conscious)
2. **Same-domain Filtering**: Prevents session poisoning via internal links
3. **No User Input**: Referrer comes from HTTP header (server-controlled)
4. **Session-based**: Inherits Laravel's session security (encryption, CSRF)
5. **Query String Sanitisation**: Consider sanitising query params if storing user-generated data

## Dependencies

### Required Packages
- `bongo/framework` ^3.0 - Provides `AbstractServiceProvider` and `URL` helper
- `illuminate/contracts` ^10.0 - Laravel contracts (Session, Request)
- PHP 8.2+

### Development Packages
- `phpunit/phpunit` ^10.0 - Testing framework
- `orchestra/testbench` ^8.0 - Laravel package testing
- `laravel/pint` ^1.0 - Code style fixer
- `nunomaduro/larastan` ^2.0 - Static analysis

## Integration with Bongo Framework

### AbstractServiceProvider Features Used

**File**: `bongo/framework/src/Providers/AbstractServiceProvider.php`

**Auto-loaded by parent `boot()` method**:
- Config: `src/Config/referrer.php` → `config('referrer.*')`
- Routes: `src/Routes/*.php` (none in this package)
- Views: `src/Views/referrer/` (none in this package)
- Migrations: `src/Migrations/` (none in this package)
- Translations: `src/Translations/` (none in this package)

**Module Property**:
```php
protected string $module = 'referrer';  // Must match config filename
```

### Bongo\Framework\Helpers\URL Usage

**Method**: `URL::getHost(string $url): string`

**Purpose**: Extract hostname from URL string

**Usage in this package**:
```php
use Bongo\Framework\Helpers\URL;

$refererHost = URL::getHost('https://google.com/search?q=test');
// Returns: 'google.com'
```

## Common Pitfalls

1. **Session Not Started**: Ensure session middleware is active
2. **Middleware Order**: Place `CaptureReferrer` after session middleware
3. **Same-domain Traffic**: Referrer won't capture internal navigation (by design)
4. **Empty Referrer**: Direct traffic (bookmarks, typed URLs) have no referrer
5. **HTTPS → HTTP**: Some browsers don't send referrer for protocol downgrade
6. **Helper Loading**: Helpers loaded after app boots, not available in service providers
