# CLAUDE.md - Bongo Referrer Package

This file provides guidance to Claude Code when working with the Bongo Referrer package.

## Overview

A Laravel package that captures and tracks HTTP referrer information at the application level. It stores the first external referrer in the session and provides platform detection methods for common advertising sources.

**Key Features**:
- Session-based referrer storage
- Automatic external referrer capture via middleware
- Platform detection (Google, Google Ads, Facebook Ads, LinkedIn Ads)
- Query string preservation for ad tracking (UTM, gclid, etc.)
- Helper functions for easy access

## Documentation Structure

- **ARCHITECTURE.md** - Detailed architecture, data flows, class diagrams, and extension points
- **.cursorrules** - Cursor AI instructions with project structure and coding conventions
- **.github/copilot-instructions.md** - GitHub Copilot patterns and code templates
- **README.md** - User-facing documentation and installation guide

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0

## Commands

From `composer.json`:

```bash
# Run tests
composer test
vendor/bin/phpunit

# Fix code style
vendor/bin/pint

# Check code style (without fixing)
vendor/bin/pint --test

# Static analysis (if configured)
vendor/bin/phpstan analyse
```

## Quick Architecture Reference

### Service Provider Pattern

**Class**: `Bongo\Referrer\ReferrerServiceProvider`
- Extends `Bongo\Framework\Providers\AbstractServiceProvider`
- Module name: `referrer`
- Registers `Referrer` service as singleton with alias `'referrer'`
- Loads helper functions after app boots

**Auto-loaded by AbstractServiceProvider**:
- Config from `src/Config/referrer.php`
- Routes from `src/Routes/*.php` (none in this package)
- Views from `src/Views/referrer/` (none in this package)

### Core Components

```
ReferrerServiceProvider (src/ReferrerServiceProvider.php)
    ↓ registers
Referrer Service (src/Services/Referrer.php)
    ↓ used by
CaptureReferrer Middleware (src/Http/Middleware/CaptureReferrer.php)
    ↓ captures
HTTP Referrer Header → Session Storage
```

### Data Flow

```
1. Request arrives with Referer header
2. CaptureReferrer middleware intercepts
3. Referrer::putFromRequest() processes:
   - Extract hostname from referrer URL
   - Filter out same-domain traffic
   - Preserve query string (ad tracking params)
   - Store in session
4. Later: Controller/view retrieves via referrer()->get()
```

## Key Files

| File | Purpose | Key Methods/Properties |
|------|---------|------------------------|
| `src/ReferrerServiceProvider.php` | Bootstrap package | `boot()`, `$module = 'referrer'` |
| `src/Services/Referrer.php` | Core service | `get()`, `put()`, `putFromRequest()`, `isGoogle()`, `isGoogleAds()`, `isFacebookAds()`, `isLinkedinAds()` |
| `src/Http/Middleware/CaptureReferrer.php` | Request middleware | `handle($request, $next)` |
| `src/Config/referrer.php` | Configuration | `session_key` |
| `src/helpers.php` | Global helpers | `referrer()`, `get_referrer()` |
| `tests/TestCase.php` | Base test case | `getPackageProviders()` |

## Configuration

**File**: `src/Config/referrer.php`

```php
return [
    'session_key' => 'bongo_referrer',  // Session key for storing referrer
];
```

**Access**: `config('referrer.session_key')`

## Public API

### Referrer Service

**Class**: `Bongo\Referrer\Services\Referrer`

```php
// Retrieve referrer
public function get(): string

// Store referrer manually
public function put(string $referer): void

// Clear referrer
public function forget(): void

// Capture from request (used by middleware)
public function putFromRequest(Request $request): void

// Platform detection
public function isGoogle(): bool
public function isGoogleAds(): bool
public function isFacebookAds(): bool
public function isLinkedinAds(): bool
```

### Helper Functions

```php
referrer(): Referrer        // Get Referrer service instance
get_referrer(): string      // Get referrer string directly
```

### Usage Examples

```php
// Get referrer
$source = referrer()->get();
$source = get_referrer();

// Store manually
referrer()->put('google.com?utm_source=adwords');

// Clear referrer
referrer()->forget();

// Platform detection
if (referrer()->isGoogleAds()) {
    // Google Ads conversion logic
}

// Check if from Google (any)
if (referrer()->isGoogle()) {
    // Organic Google traffic
}
```

## Code Style

### Required Patterns

1. **Strict types**: Always include `declare(strict_types = 1);`
2. **Type declarations**: All parameters and return types must be typed
3. **Typed properties**: Use `protected string $property;` format
4. **Laravel Pint**: Follow Laravel preset code style

### Examples

```php
<?php

declare(strict_types = 1);

namespace Bongo\Referrer\Services;

class Example
{
    protected string $property;

    public function __construct(Session $session)
    {
        $this->property = 'value';
    }

    public function method(string $param): bool
    {
        return true;
    }
}
```

## Common Tasks

### Add New Platform Detection

**File**: `src/Services/Referrer.php`

```php
public function isTwitterAds(): bool
{
    return Str::contains($this->get(), 'twitter');
}

// For multiple keywords:
public function isBingAds(): bool
{
    return Str::contains($this->get(), ['bing', 'msclkid']);
}
```

### Modify Referrer Capture Logic

**File**: `src/Services/Referrer.php` → `putFromRequest()` method

**Current logic**:
- Line 39: Extract referer header
- Line 45: Parse hostname
- Lines 51-53: Filter same-domain traffic
- Lines 56-60: Append query string
- Line 63: Store in session

**Example - Store full URL**:
```php
public function putFromRequest(Request $request): void
{
    $referer = $request->header('referer', '');
    if (empty($referer)) {
        return;
    }

    $refererHost = URL::getHost($referer);
    if (empty($refererHost) || $refererHost === $request->getHost()) {
        return;
    }

    // Store full URL instead of hostname
    $this->put($referer);
}
```

### Add Middleware to Routes

**In consuming application**:

```php
// Option 1: Global middleware (app/Http/Kernel.php)
protected $middleware = [
    \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
];

// Option 2: Web middleware group (app/Http/Kernel.php)
protected $middlewareGroups = [
    'web' => [
        \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
    ],
];

// Option 3: Named middleware alias (app/Http/Kernel.php)
protected $middlewareAliases = [
    'capture.referrer' => \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
];

// routes/web.php
Route::middleware('capture.referrer')->get('/landing', /* ... */);
```

### Change Session Key

**In consuming application**:

1. Publish config:
```bash
php artisan vendor:publish --provider="Bongo\Referrer\ReferrerServiceProvider" --tag=config
```

2. Modify `config/referrer.php`:
```php
return [
    'session_key' => 'custom_referrer_key',
];
```

### Run Tests

```bash
# From package directory
vendor/bin/phpunit

# With coverage
vendor/bin/phpunit --coverage-html coverage
```

### Fix Code Style

```bash
# Check issues
vendor/bin/pint --test

# Fix issues
vendor/bin/pint
```

## Dependencies

### Framework Dependencies
- `bongo/framework` ^3.0
  - Provides `AbstractServiceProvider` (base class)
  - Provides `URL::getHost()` helper (URL parsing)

### Laravel Dependencies
- `illuminate/contracts` ^10.0
  - `Session` contract
  - `Request` class
- `illuminate/support`
  - `Str` helper
  - `ServiceProvider` base class

## Integration with Bongo Framework

### AbstractServiceProvider

**Provides automatic bootstrapping**:
- Config registration from `src/Config/{module}.php`
- Route registration from `src/Routes/*.php`
- View registration from `src/Views/{module}/`
- Migration loading from `src/Migrations/`
- Translation loading from `src/Translations/`

**Required property**:
```php
protected string $module = 'referrer';  // Must match config filename
```

### URL Helper

**Usage**:
```php
use Bongo\Framework\Helpers\URL;

$host = URL::getHost('https://google.com/search');
// Returns: 'google.com'
```

## Important Notes

1. **Session Dependency**: Service requires active Laravel session
2. **First Referrer Only**: Captures initial external referrer, ignores subsequent
3. **Same-domain Filtering**: Automatically excludes internal navigation
4. **Query String Preservation**: Maintains UTM parameters and ad tracking codes
5. **Host-only Storage**: Stores hostname + query string, not full URL path
6. **Middleware Placement**: Must come after session middleware in stack
7. **Helper Availability**: Global helpers loaded after app boots, not in service providers
8. **Singleton Service**: One instance per request lifecycle

## Testing Approach

**Framework**: PHPUnit 10 with Orchestra Testbench 8

**Base Test Case**: `Bongo\Referrer\Tests\TestCase`
- Automatically registers `ReferrerServiceProvider`
- Sets up test environment

**Example Test**:
```php
namespace Bongo\Referrer\Tests;

class ReferrerTest extends TestCase
{
    /** @test */
    public function it_captures_external_referrer()
    {
        $request = Request::create('/test', 'GET', [], [], [], [
            'HTTP_REFERER' => 'https://google.com',
        ]);

        app(Referrer::class)->putFromRequest($request);

        $this->assertEquals('google.com', referrer()->get());
    }
}
```

## Typical Use Cases

1. **Conversion Tracking**: Identify which ad platform generated a sale
2. **Analytics**: Track traffic sources at application level
3. **A/B Testing**: Segment users by referrer source
4. **Commission Attribution**: Credit affiliates based on referrer
5. **Marketing ROI**: Measure campaign effectiveness

## Extension Points

See **ARCHITECTURE.md** for detailed extension guides:
- Adding new platform detection methods
- Customising storage backend (Redis, Database)
- Modifying capture logic (allowlists, full URLs, expiry)
- Adding event dispatching
- Creating custom middleware
