# Bongo Referrer

A Laravel package that provides application-level HTTP referrer tracking and detection. Captures the first external referrer in session storage and provides platform detection methods for common advertising sources.

## Features

- **Automatic Referrer Capture**: Middleware-based capture of external HTTP referrers
- **Session Storage**: Lightweight session-based storage (no database required)
- **Platform Detection**: Built-in detection for Google, Google Ads, Facebook Ads, and LinkedIn Ads
- **Ad Tracking**: Preserves query string parameters (UTM codes, gclid, etc.)
- **Privacy-Conscious**: Stores hostname only, not full URL paths
- **Same-domain Filtering**: Automatically ignores internal navigation
- **Helper Functions**: Convenient global helper functions for easy access

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0

## Installation

### Composer

```bash
composer require bongo/referrer
```

### Laravel Auto-discovery

The package uses Laravel's auto-discovery feature. The service provider will be automatically registered.

For Laravel 5.x, manually add the service provider to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Referrer\ReferrerServiceProvider::class,
],
```

### Middleware Registration

Add the middleware to your application's HTTP kernel to enable automatic referrer capture.

**Option 1: Global Middleware** (captures on all requests)

In `app/Http/Kernel.php`:

```php
protected $middleware = [
    // ... other middleware
    \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
];
```

**Option 2: Web Middleware Group** (captures on web requests only)

```php
protected $middlewareGroups = [
    'web' => [
        // ... other middleware
        \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
    ],
];
```

**Option 3: Named Middleware** (apply to specific routes)

```php
protected $middlewareAliases = [
    // ... other aliases
    'capture.referrer' => \Bongo\Referrer\Http\Middleware\CaptureReferrer::class,
];
```

Then in your routes:

```php
Route::middleware('capture.referrer')->group(function () {
    Route::get('/landing-page', [LandingController::class, 'index']);
});
```

### Configuration (Optional)

Publish the configuration file:

```bash
php artisan vendor:publish --provider="Bongo\Referrer\ReferrerServiceProvider" --tag=config
```

This creates `config/referrer.php`:

```php
return [
    'session_key' => 'bongo_referrer',  // Session key for storing referrer
];
```

## Usage

### Basic Usage

```php
// Get the current referrer
$referrer = referrer()->get();
// or
$referrer = get_referrer();

// Manually store a referrer
referrer()->put('google.com?utm_source=adwords');

// Clear the stored referrer
referrer()->forget();
```

### Platform Detection

```php
// Check if referrer is from Google (any)
if (referrer()->isGoogle()) {
    echo "Visitor came from Google";
}

// Check if referrer is from Google Ads
if (referrer()->isGoogleAds()) {
    echo "Visitor came from a Google Ads campaign";
    // Track conversion, etc.
}

// Check if referrer is from Facebook Ads
if (referrer()->isFacebookAds()) {
    echo "Visitor came from Facebook";
}

// Check if referrer is from LinkedIn Ads
if (referrer()->isLinkedinAds()) {
    echo "Visitor came from LinkedIn";
}
```

### In Controllers

```php
namespace App\Http\Controllers;

use Illuminate\Http\Request;

class LandingController extends Controller
{
    public function index()
    {
        $referrer = referrer();

        // Get the referrer string
        $source = $referrer->get();

        // Platform-specific logic
        if ($referrer->isGoogleAds()) {
            // Track Google Ads conversion
            $this->trackGoogleAdsConversion();
        } elseif ($referrer->isFacebookAds()) {
            // Track Facebook Ads conversion
            $this->trackFacebookAdsConversion();
        }

        return view('landing-page', [
            'referrer' => $source,
        ]);
    }
}
```

### In Views

```blade
@if(referrer()->isGoogleAds())
    <!-- Google Ads conversion tracking pixel -->
    <script>
        gtag('event', 'conversion', {
            'send_to': 'AW-XXXXXXXXX/XXXXXX',
            'transaction_id': ''
        });
    </script>
@endif

<p>You came from: {{ get_referrer() }}</p>
```

### Manual Capture

```php
use Illuminate\Http\Request;

public function storeLead(Request $request)
{
    // The middleware automatically captures the referrer,
    // but you can also manually capture it:
    referrer()->putFromRequest($request);

    // Or set it manually:
    referrer()->put('custom-source.com');

    // Store lead with referrer
    Lead::create([
        'email' => $request->email,
        'referrer' => referrer()->get(),
    ]);
}
```

## How It Works

1. **Middleware Intercepts Request**: The `CaptureReferrer` middleware runs on incoming requests
2. **Extract Referrer Header**: Reads the `Referer` HTTP header from the request
3. **Parse Hostname**: Extracts the hostname from the referrer URL
4. **Filter Same-domain Traffic**: Ignores referrers from the same domain (internal navigation)
5. **Preserve Query Strings**: Appends query string parameters (for ad tracking like `gclid`, `utm_source`, etc.)
6. **Store in Session**: Saves the referrer hostname in the session (first external referrer wins)

**Example Flow**:

```
User clicks ad: https://google.com/search?q=laravel&gclid=abc123
    ↓
Lands on: https://yoursite.com/landing?utm_source=google&gclid=abc123
    ↓
Middleware captures: google.com?utm_source=google&gclid=abc123
    ↓
Stored in session: 'bongo_referrer' => 'google.com?utm_source=google&gclid=abc123'
    ↓
User navigates to: https://yoursite.com/contact
    ↓
Middleware sees same-domain referrer, ignores it
    ↓
Original referrer preserved: google.com?utm_source=google&gclid=abc123
```

## API Reference

### Referrer Service

**Class**: `Bongo\Referrer\Services\Referrer`

#### Methods

```php
// Retrieve the stored referrer
public function get(): string

// Manually store a referrer
public function put(string $referer): void

// Clear the stored referrer
public function forget(): void

// Capture referrer from HTTP request (used by middleware)
public function putFromRequest(Request $request): void

// Check if referrer contains 'google'
public function isGoogle(): bool

// Check if referrer contains 'gclid' (Google Ads click ID)
public function isGoogleAds(): bool

// Check if referrer contains 'facebook'
public function isFacebookAds(): bool

// Check if referrer contains 'linkedin'
public function isLinkedinAds(): bool
```

### Helper Functions

```php
// Get the Referrer service instance
referrer(): Bongo\Referrer\Services\Referrer

// Get referrer string directly (shortcut for referrer()->get())
get_referrer(): string
```

## Extending the Package

### Adding Custom Platform Detection

You can extend the `Referrer` class to add custom platform detection:

```php
namespace App\Services;

use Bongo\Referrer\Services\Referrer as BaseReferrer;
use Illuminate\Support\Str;

class CustomReferrer extends BaseReferrer
{
    public function isTwitterAds(): bool
    {
        return Str::contains($this->get(), ['twitter', 'twclid']);
    }

    public function isBingAds(): bool
    {
        return Str::contains($this->get(), ['bing', 'msclkid']);
    }

    public function isTikTokAds(): bool
    {
        return Str::contains($this->get(), 'tiktok');
    }
}
```

Then bind your custom class in a service provider:

```php
public function register()
{
    $this->app->singleton(\Bongo\Referrer\Services\Referrer::class, CustomReferrer::class);
}
```

## Configuration Options

| Option | Type | Default | Description |
|--------|------|---------|-------------|
| `session_key` | string | `'bongo_referrer'` | The session key used to store the referrer data |

## Use Cases

- **Conversion Tracking**: Identify which advertising platform generated sales/leads
- **Analytics**: Track traffic sources at the application level
- **A/B Testing**: Segment users by referrer source
- **Affiliate Attribution**: Credit affiliates based on referrer
- **Marketing ROI**: Measure campaign effectiveness across platforms
- **Commission Tracking**: Attribute commissions to the correct traffic source

## Important Notes

- **First Referrer Wins**: Only the first external referrer is captured and stored
- **Session-based**: Requires an active Laravel session to function
- **Same-domain Filtering**: Internal navigation (same-domain referrers) is automatically ignored
- **Privacy-conscious**: Only the hostname (and query string) is stored, not full URL paths
- **Middleware Order**: The `CaptureReferrer` middleware must come after session middleware in the stack

## Testing

```bash
# Run tests
vendor/bin/phpunit

# Run tests with coverage
vendor/bin/phpunit --coverage-html coverage

# Code style check
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

## Documentation

- **CLAUDE.md** - Claude Code guidance and quick reference
- **ARCHITECTURE.md** - Detailed architecture documentation with diagrams
- **.cursorrules** - Cursor AI instructions
- **.github/copilot-instructions.md** - GitHub Copilot patterns

## License

MIT

## Author

- **Stuart Elliott** - [stuart.elliott@bespokeuk.com](mailto:stuart.elliott@bespokeuk.com)
- **Bespoke UK** - [https://bespokeuk.com](https://bespokeuk.com)

## Support

For issues, questions, or contributions, please visit the [Bitbucket repository](https://bitbucket.org/designtec/referrer).
