# Bongo Review Package - Claude Code Guidance

## Overview

This Laravel package provides customer review and testimonial management functionality with both admin and public interfaces.

**Package:** `bongo/review`
**Namespace:** `Bongo\Review`
**Requirements:** PHP 8.2+, Laravel 10+

### Key Features
- Customer review submission with spam protection (reCAPTCHA + Honeypot)
- Admin review moderation and CRUD management
- Star rating system (1-5 stars)
- Email notifications (customer + admin)
- Cached statistics (count, average, latest)
- Excel export via artisan command
- UUID-based public URLs
- Event-driven cache invalidation

### Documentation
- **Architecture Details:** See `ARCHITECTURE.md` for comprehensive architecture documentation
- **Cursor AI Instructions:** See `.cursorrules` for detailed development guidance
- **GitHub Copilot:** See `.github/copilot-instructions.md` for code templates and patterns

## Commands

Commands available from `composer.json`:

```bash
# Run tests
composer test
# or
vendor/bin/phpunit

# Check code style
composer pint:test
# or
vendor/bin/pint --test

# Fix code style
composer pint
# or
vendor/bin/pint

# Static analysis
composer analyse
# or
vendor/bin/phpstan analyse
```

### Artisan Commands

```bash
# Export all reviews to CSV
php artisan review:export
# Exports to: storage/app/reviews.csv
```

## Quick Architecture Reference

### Service Provider Pattern

Extends `Bongo\Framework\Providers\AbstractServiceProvider` which auto-registers:
- **Config:** `src/Config/review.php`
- **Routes:** `src/Routes/backend.php`, `src/Routes/frontend.php`
- **Views:** `src/Views/review/` (namespace: `review::`)
- **Migrations:** `src/Migrations/`
- **Translations:** `src/Translations/` (namespace: `review::`)
- **Commands:** `ExportReviewsCommand`
- **View Composers:** `NumberOfReviewsComposer`, `LatestReviewComposer`, `AverageRatingComposer`
- **Event Listeners:** `ClearReviewCache` (on ReviewCreated/Updated)

### Model Architecture

**Review Model** (`src/Models/Review.php`):
```php
class Review extends AbstractModel
{
    use HasContent;   // Content field handling
    use HasStatus;    // Provides: active(), pending(), inactive() scopes
    use HasUUID;      // Auto-generates UUID on creation
    use SoftDeletes;  // Soft deletion support

    const PENDING = 'pending';
    const ACTIVE = 'active';
    const INACTIVE = 'inactive';

    protected $fillable = ['date', 'title', 'content', 'rating', 'status', 'name', 'email'];
    protected $casts = ['date' => Date::class];
}
```

### Route Structure

**Backend (Admin):**
- Prefix: `/admin/reviews` (configurable: `config('review.prefix')`)
- Named: `backend.review.*`
- Middleware: `auth`, `employee`
- Routes: index, create, store, show, edit, update, destroy, datatable

**Frontend (Public):**
- Prefix: `/reviews` (configurable: `config('review.prefix')`)
- Named: `frontend.review.*`
- Middleware: `web`, honeypot (on store)
- Routes: index, show (by UUID), store

### Events and Listeners

**Events:**
- `ReviewCreated` - Fired when review created (backend or frontend)
- `ReviewUpdated` - Fired when review updated
- `ReviewDeleted` - Fired when review deleted

**Listeners:**
- `ClearReviewCache` (queued) - Clears cached stats on ReviewCreated/ReviewUpdated
  - Clears: `no_of_reviews`, `latest_review`, `average_rating`

### Caching

View composers cache statistics:
- `no_of_reviews` - Count of active reviews
- `latest_review` - Most recent active review
- `average_rating` - Average rating of active reviews (ceiled)

Cache TTL: `config('settings.cache_default')`

Cache invalidation: Automatic via `ClearReviewCache` listener (queued)

### Email Workflow

Frontend submission sends two emails:
1. `ReviewMailable` → Customer email (confirmation)
2. `AdminReviewMailable` → Admin email (notification to `setting('client::company.email')`)

Templates: `src/Views/mail/` (HTML + plain text versions)

## Key Files Reference

| File | Purpose | Key Methods/Properties |
|------|---------|------------------------|
| `src/ReviewServiceProvider.php` | Service provider | `$module = 'review'`, `$commands`, `$composers`, `$listeners` |
| `src/Models/Review.php` | Review eloquent model | `active()`, `pending()`, `inactive()`, `isActive()` scopes |
| `src/Http/Controllers/Backend/ReviewController.php` | Admin CRUD | `index()`, `create()`, `store()`, `show()`, `edit()`, `update()`, `destroy()` |
| `src/Http/Controllers/Frontend/ReviewController.php` | Public submission | `index()`, `store()`, `show()` |
| `src/Http/Requests/StoreFrontendReviewRequest.php` | Frontend validation | reCAPTCHA validation via `Captcha` rule |
| `src/Listeners/ClearReviewCache.php` | Cache invalidation | Implements `ShouldQueue`, clears 3 cache keys |
| `src/Commands/ExportReviewsCommand.php` | Export command | Signature: `review:export` |
| `src/Config/review.php` | Configuration | `prefix`, `success_url`, `recaptcha` settings |

## Code Style Summary

### Naming Conventions
- Models: Singular (Review)
- Controllers: Singular + Controller (ReviewController)
- Events: Past tense (ReviewCreated)
- Listeners: Action-based (ClearReviewCache)
- Requests: Action + Request (StoreFrontendReviewRequest)
- View Composers: Purpose + Composer (AverageRatingComposer)

### Return Types
All controller methods use explicit return types:
- `View` for view responses
- `RedirectResponse` for redirects
- `void` for listeners/composers

### Controller Pattern

```php
public function store(StoreReviewRequest $request): RedirectResponse
{
    $review = $this->review->create($request->all());
    event(new ReviewCreated($review));

    return redirect()
        ->route('backend.review.show', $review->id)
        ->success(trans('review::backend.store_success'));
}
```

### Event Dispatching

Always dispatch events after model changes:
```php
event(new ReviewCreated($review));   // After create
event(new ReviewUpdated($review));   // After update
event(new ReviewDeleted($review));   // After delete
```

### Cache Usage

View composers follow this pattern:
```php
$data = Cache::remember('cache_key', config('settings.cache_default'), function () {
    return Review::active()->someQuery();
});

$view->with(compact('data'));
```

### Validation Pattern

Frontend requests include reCAPTCHA:
```php
public function rules(): array
{
    $recaptchaEnabled = setting()->reCaptchaEnabled() && config('review.recaptcha.enabled');
    $recaptchaMinScore = config('review.recaptcha.min_score', 0.5);

    return [
        'title' => 'required',
        'content' => 'required',
        'rating' => 'required',
        'name' => 'required',
        'email' => 'required|string|email:rfc,dns|max:50',
        'captcha-response' => new Captcha(
            action: 'review',
            minScore: $recaptchaMinScore,
            enabled: $recaptchaEnabled,
        ),
    ];
}
```

## Common Queries

```php
// Get paginated active reviews
Review::active()->orderBy('date', 'DESC')->paginate(8)

// Get latest active review
Review::active()->whereNotNull('date')->latest('date')->first()

// Get average rating
ceil(Review::active()->average('rating'))

// Count active reviews
Review::active()->count()

// Get review by UUID
Review::where('uuid', $uuid)->first()

// Filter by rating
Review::active()->where('rating', 5)->get()
```

## Configuration

**Config file:** `src/Config/review.php`

```php
return [
    'prefix' => 'reviews',                       // Route prefix for both frontend and backend
    'success_url' => '/thank-you-for-your-review', // Redirect after frontend submission
    'recaptcha' => [
        'enabled' => true,                       // Package-level reCAPTCHA toggle
        'min_score' => 0.5,                      // Score threshold (0.1 - 1.0)
    ],
];
```

Publish config:
```bash
php artisan vendor:publish --provider="Bongo\Review\ReviewServiceProvider"
```

## Database Schema

**Table:** `reviews`

Key columns:
- `id` (increments) - Primary key
- `uuid` (uuid, indexed) - Public identifier
- `name` (string) - Reviewer name
- `email` (string) - Reviewer email
- `title` (string) - Review title
- `content` (text, nullable) - Review content
- `date` (date, nullable) - Review date
- `rating` (unsignedTinyInteger, default: 5, indexed) - Star rating (1-5)
- `status` (enum: pending|active|inactive, default: pending)
- Audit fields: `created_by`, `updated_by`, `deleted_by`
- Timestamps: `created_at`, `updated_at`, `deleted_at`

## Security Features

1. **reCAPTCHA v3** - Prevents automated spam (configurable score threshold)
2. **Honeypot** - Additional spam protection via Spatie package
3. **Email Validation** - RFC and DNS validation
4. **Route Middleware** - Admin routes protected by `auth` + `employee` middleware
5. **Status Moderation** - Reviews default to 'pending' status
6. **UUID URLs** - Public review URLs use UUID instead of sequential ID

## Dependencies on Bongo Framework

This package requires `bongo/framework` for:

### Base Classes
- `AbstractServiceProvider` - Auto-bootstrapping service provider
- `AbstractModel` - Base model with audit fields
- `AbstractController` - Base controller

### Traits
- `HasContent` - Content field handling
- `HasStatus` - Status scopes (active, pending, inactive) and check methods
- `HasUUID` - Auto-generates UUID on model creation

### Casts
- `Date` - Custom date cast for review date field

### Helpers
- `setting()` - Access global settings (used for admin email)
- `console_print()` - Console output helper (used in export command)

## Common Development Tasks

### Adding a New Field

1. Create migration in `src/Migrations/`
2. Add to `$fillable` in `Review` model
3. Update form requests: `StoreReviewRequest`, `StoreFrontendReviewRequest`, `UpdateReviewRequest`
4. Update backend form: `src/Views/backend/partials/form/details.blade.php`
5. Update frontend form: `src/Views/frontend/partials/form.blade.php`
6. Update display views: `src/Views/backend/show.blade.php`, `src/Views/frontend/show.blade.php`
7. Update export template: `src/Views/exports/reviews.blade.php`

### Customizing Email Templates

Edit templates in `src/Views/mail/`:
- `review.blade.php` - Customer HTML email
- `review_plain.blade.php` - Customer plain text email
- `admin_review.blade.php` - Admin HTML email
- `admin_review_plain.blade.php` - Admin plain text email

Variable available: `$review` (Review model instance)

### Adding New Cached Statistic

1. Create view composer in `src/Http/ViewComposers/`
2. Register in `ReviewServiceProvider::$composers`
3. Add cache key to `ClearReviewCache::handle()`

### Adjusting reCAPTCHA Settings

Edit `src/Config/review.php`:
```php
'recaptcha' => [
    'enabled' => true,      // Enable/disable reCAPTCHA
    'min_score' => 0.5,     // Score threshold (0.1 = lenient, 1.0 = strict)
],
```

Note: Also requires global setting via `setting()->reCaptchaEnabled()`

### Auto-Approve Reviews

Modify `Frontend/ReviewController::store()`:
```php
$review = Review::create(array_merge(
    Arr::except($request->validated(), ['captcha-response']),
    ['status' => Review::ACTIVE]
));
```

## View Namespacing

Views use `review::` namespace:
- `review::backend.*` - Admin interface
- `review::frontend.*` - Public interface
- `review::mail.*` - Email templates
- `review::exports.*` - Export templates

## Translation Keys

Translations use `review::` namespace:
- `review::backend.*` - Admin translations
- `review::frontend.*` - Frontend translations

Files: `src/Translations/en/backend.php`, `src/Translations/en/frontend.php`

## Testing

**Test Setup:**
- Base test case: `tests/TestCase.php`
- Database connection: `testing` (configured in `phpunit.xml`)
- Coverage: Tests should be created in `tests/` directory

**Run Tests:**
```bash
vendor/bin/phpunit
```

## Package Structure

```
src/
├── Config/review.php          # Package configuration
├── Models/Review.php          # Review model
├── Http/
│   ├── Controllers/
│   │   ├── Backend/           # Admin controllers
│   │   └── Frontend/          # Public controllers
│   ├── Requests/              # Form validation
│   └── ViewComposers/         # View composers for statistics
├── Events/                    # ReviewCreated, ReviewUpdated, ReviewDeleted
├── Listeners/                 # ClearReviewCache
├── Mailables/                 # ReviewMailable, AdminReviewMailable
├── Commands/                  # ExportReviewsCommand
├── Exports/                   # ReviewExport (Excel)
├── Migrations/                # Database migrations
├── Views/                     # Blade templates
├── Routes/                    # backend.php, frontend.php
├── Seeders/                   # PackageSeeder
└── Translations/              # en/backend.php, en/frontend.php
```

## Extension Points

1. **Custom Status Types** - Add new status constants and scopes
2. **Additional Fields** - Extend review model with custom fields
3. **Event Listeners** - Add custom listeners to review events
4. **View Composers** - Add new cached statistics
5. **Email Templates** - Customize notification emails
6. **Controller Override** - Extend controllers for custom logic
7. **Export Format** - Customize Excel export template
8. **Validation Rules** - Add custom validation to form requests

See `ARCHITECTURE.md` for detailed implementation examples.
