<?php

namespace Bongo\Review\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Review\Events\ReviewCreated;
use Bongo\Review\Events\ReviewDeleted;
use Bongo\Review\Events\ReviewUpdated;
use Bongo\Review\Http\Requests\StoreReviewRequest;
use Bongo\Review\Http\Requests\UpdateReviewRequest;
use Bongo\Review\Models\Review;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class ReviewController extends AbstractController
{
    protected Review $review;

    public function __construct(Review $review)
    {
        $this->review = $review;
    }

    public function index(): View
    {
        return view('review::backend.index');
    }

    public function create(): View
    {
        $review = $this->review;

        return view('review::backend.create', compact('review'));
    }

    public function store(StoreReviewRequest $request): RedirectResponse
    {
        $review = $this->review->create($request->all());
        event(new ReviewCreated($review));

        return redirect()
            ->route('backend.review.show', $review->id)
            ->success(trans('review::backend.store_success'));
    }

    public function show(Review $review): View
    {
        return view('review::backend.show', compact('review'));
    }

    public function edit(Review $review): View
    {
        return view('review::backend.edit', compact('review'));
    }

    public function update(UpdateReviewRequest $request, Review $review): RedirectResponse
    {
        $review->update($request->all());
        event(new ReviewUpdated($review));

        return redirect()
            ->route('backend.review.show', $review->id)
            ->success(trans('review::backend.update_success'));
    }

    public function destroy(Review $review)
    {
        if ($review->delete()) {
            event(new ReviewDeleted($review));
            $redirect = back();

            // If the review was deleted from their detail view,
            // we'll redirect back to the review list view.
            if (route('backend.review.show', $review) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.review.index');
            }

            return $redirect->success(trans('review::backend.delete_success'));
        }

        return back()->error(trans('review::backend.delete_failed'));
    }
}
