# GitHub Copilot Instructions - Bongo Review Package

## Project Overview

This is a Laravel package (`bongo/review`) for managing customer reviews and testimonials. It provides:
- Backend admin interface for review moderation
- Frontend public interface for review submission
- Star rating system (1-5 stars)
- Email notifications
- Excel export
- Cached statistics

**Namespace:** `Bongo\Review`
**Framework:** Laravel 10+, PHP 8.2+
**Dependencies:** `bongo/framework`, `bongo/captcha`

## Key Classes and Relationships

### Core Model

```php
namespace Bongo\Review\Models;

class Review extends AbstractModel
{
    use HasContent, HasStatus, HasUUID, SoftDeletes;

    const PENDING = 'pending';
    const ACTIVE = 'active';
    const INACTIVE = 'inactive';

    // Fillable: date, title, content, rating, status, name, email
    // Casts: date => Date::class
}
```

**Traits:**
- `HasContent` (from bongo/framework) - Content field handling
- `HasStatus` (from bongo/framework) - Status scopes: `active()`, `pending()`, `inactive()`
- `HasUUID` (from bongo/framework) - Auto-generates UUID on creation
- `SoftDeletes` (Laravel) - Soft deletion support

### Controllers

**Backend Admin Controller:**
```php
namespace Bongo\Review\Http\Controllers\Backend;

class ReviewController extends AbstractController
{
    // CRUD operations
    public function index(): View
    public function create(): View
    public function store(StoreReviewRequest $request): RedirectResponse
    public function show(Review $review): View
    public function edit(Review $review): View
    public function update(UpdateReviewRequest $request, Review $review): RedirectResponse
    public function destroy(Review $review): RedirectResponse
}
```

**Frontend Public Controller:**
```php
namespace Bongo\Review\Http\Controllers\Frontend;

class ReviewController extends AbstractController
{
    // Public review listing and submission
    public function index(): View // Paginated active reviews
    public function store(StoreFrontendReviewRequest $request): RedirectResponse // Create review + send emails
    public function show(string $uuid): View // Show single review by UUID
}
```

### Events and Listeners

**Events:**
```php
namespace Bongo\Review\Events;

class ReviewCreated    { public Review $review; }
class ReviewUpdated    { public Review $review; }
class ReviewDeleted    { public Review $review; }
```

**Listeners:**
```php
namespace Bongo\Review\Listeners;

class ClearReviewCache implements ShouldQueue
{
    public function handle($event): void
    {
        Cache::forget('no_of_reviews');
        Cache::forget('latest_review');
        Cache::forget('average_rating');
    }
}
```

### View Composers

**Inject cached statistics into summary view:**
```php
namespace Bongo\Review\Http\ViewComposers;

class NumberOfReviewsComposer { /* Injects $noOfReviews */ }
class LatestReviewComposer    { /* Injects $latestReview */ }
class AverageRatingComposer   { /* Injects $averageRating, $averageRatingText */ }
```

All three composers inject data into `review::frontend.partials.summary`.

### Form Requests

**Frontend (with reCAPTCHA):**
```php
namespace Bongo\Review\Http\Requests;

class StoreFrontendReviewRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'title' => 'required',
            'content' => 'required',
            'rating' => 'required',
            'name' => 'required',
            'email' => 'required|string|email:rfc,dns|max:50',
            'captcha-response' => new Captcha(/* ... */),
        ];
    }
}
```

**Backend:**
```php
class StoreReviewRequest extends FormRequest { /* ... */ }
class UpdateReviewRequest extends FormRequest { /* ... */ }
```

### Mailables

```php
namespace Bongo\Review\Mailables;

class ReviewMailable extends Mailable      { /* Sent to customer */ }
class AdminReviewMailable extends Mailable { /* Sent to admin */ }
```

Both mailables receive `Review $review` in constructor and use Blade templates from `src/Views/mail/`.

### Export

```php
namespace Bongo\Review\Exports;

class ReviewExport implements FromView, WithStyles
{
    public function view(): View
    {
        return view('review::exports.reviews', ['rows' => $this->rows]);
    }
}
```

### Command

```php
namespace Bongo\Review\Commands;

class ExportReviewsCommand extends Command
{
    protected $signature = 'review:export';
    // Exports all reviews to storage/app/reviews.csv
}
```

## Code Style Templates

### Controller Method Template (Backend)

```php
public function update(UpdateReviewRequest $request, Review $review): RedirectResponse
{
    $review->update($request->all());
    event(new ReviewUpdated($review));

    return redirect()
        ->route('backend.review.show', $review->id)
        ->success(trans('review::backend.update_success'));
}
```

### Controller Method Template (Frontend)

```php
public function store(StoreFrontendReviewRequest $request): RedirectResponse
{
    $review = Review::create(Arr::except($request->validated(), ['captcha-response']));
    event(new ReviewCreated($review));

    Mail::to($review->email)->send(new ReviewMailable($review));
    Mail::to(setting("client::company.email"))->send(new AdminReviewMailable($review));

    return redirect()
        ->to(config('review.success_url'))
        ->success(trans('review::frontend.store_success'));
}
```

### View Composer Template

```php
namespace Bongo\Review\Http\ViewComposers;

use Bongo\Review\Models\Review;
use Illuminate\Support\Facades\Cache;
use Illuminate\View\View;

class YourComposer
{
    public function compose(View $view): void
    {
        $data = Cache::remember('cache_key', config('settings.cache_default'), function () {
            return Review::active()->get();
        });

        $view->with(compact('data'));
    }
}
```

### Event Listener Template

```php
namespace Bongo\Review\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;

class YourListener implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        // Handle event
    }
}
```

### Migration Template

```php
use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class AddFieldToReviewsTable extends Migration
{
    public function up()
    {
        Schema::table('reviews', function (Blueprint $table) {
            $table->string('new_field')->nullable()->after('existing_field');
        });
    }

    public function down()
    {
        Schema::table('reviews', function (Blueprint $table) {
            $table->dropColumn('new_field');
        });
    }
}
```

## Common Patterns

### Query Active Reviews

```php
// Paginated active reviews
$reviews = Review::active()->orderBy('date', 'DESC')->paginate(8);

// Latest active review
$latest = Review::active()->whereNotNull('date')->latest('date')->first();

// Average rating of active reviews
$avgRating = ceil(Review::active()->average('rating'));

// Count active reviews
$count = Review::active()->count();
```

### Fire Events After Model Changes

```php
$review = Review::create($data);
event(new ReviewCreated($review));

$review->update($data);
event(new ReviewUpdated($review));

$review->delete();
event(new ReviewDeleted($review));
```

### Send Notification Emails

```php
Mail::to($review->email)->send(new ReviewMailable($review));
Mail::to(setting("client::company.email"))->send(new AdminReviewMailable($review));
```

### Cache Statistics

```php
$statistic = Cache::remember('cache_key', config('settings.cache_default'), function () {
    return Review::active()->count();
});
```

Cache keys used by this package:
- `no_of_reviews` - Total count of active reviews
- `latest_review` - Most recent active review
- `average_rating` - Average rating of active reviews

### Translation Usage

```php
// Backend translations
trans('review::backend.store_success')
trans('review::backend.update_success')
trans('review::backend.delete_success')
trans('review::backend.delete_failed')

// Frontend translations
trans('review::frontend.store_success')
```

### Redirect Responses

```php
// Success redirect (backend)
return redirect()
    ->route('backend.review.show', $review->id)
    ->success(trans('review::backend.store_success'));

// Success redirect (frontend)
return redirect()
    ->to(config('review.success_url'))
    ->success(trans('review::frontend.store_success'));

// Error redirect
return back()->error(trans('review::backend.delete_failed'));
```

### View Rendering

```php
// Backend views (use review ID)
return view('review::backend.show', compact('review'));
return view('review::backend.edit', compact('review'));

// Frontend views (use review UUID)
return view('review::frontend.show', compact('review'));
```

## Routes Structure

### Backend Routes (Admin)
- **Prefix:** `/admin/reviews` (configurable via `config('review.prefix')`)
- **Named:** `backend.review.*`
- **Middleware:** `auth`, `employee`

```
GET    /admin/reviews              → backend.review.index
GET    /admin/reviews/create       → backend.review.create
POST   /admin/reviews/store        → backend.review.store
GET    /admin/reviews/datatable    → backend.review.datatable
GET    /admin/reviews/{review}     → backend.review.show
GET    /admin/reviews/{review}/edit → backend.review.edit
POST   /admin/reviews/{review}/update → backend.review.update
DELETE /admin/reviews/{review}/delete → backend.review.destroy
```

### Frontend Routes (Public)
- **Prefix:** `/reviews` (configurable via `config('review.prefix')`)
- **Named:** `frontend.review.*`
- **Middleware:** `web`, honeypot on store

```
GET  /reviews          → frontend.review.index
GET  /reviews/{uuid}   → frontend.review.show
POST /reviews/store    → frontend.review.store
```

## Configuration

```php
// config/review.php
return [
    'prefix' => 'reviews',                       // Route prefix
    'success_url' => '/thank-you-for-your-review', // Post-submission redirect
    'recaptcha' => [
        'enabled' => true,                       // Package-level toggle
        'min_score' => 0.5,                      // Score threshold (0.1 - 1.0)
    ],
];
```

## Database Schema

```php
reviews table:
- id (increments)
- uuid (uuid, indexed)
- name (string)
- email (string)
- title (string)
- content (text, nullable)
- date (date, nullable)
- rating (unsignedTinyInteger, default: 5, indexed)
- status (enum: pending|active|inactive, default: pending)
- created_by, updated_by, deleted_by (unsignedInteger, nullable, indexed)
- timestamps
- deleted_at (soft deletes)
```

## Service Provider Configuration

```php
class ReviewServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'review';

    protected array $commands = [ExportReviewsCommand::class];

    protected array $composers = [
        NumberOfReviewsComposer::class => ['review::frontend.partials.summary'],
        LatestReviewComposer::class => ['review::frontend.partials.summary'],
        AverageRatingComposer::class => ['review::frontend.partials.summary'],
    ];

    protected array $listeners = [
        ReviewCreated::class => [ClearReviewCache::class],
        ReviewUpdated::class => [ClearReviewCache::class, UpdateSitemap::class],
    ];
}
```

## Common Development Tasks

### Add a New Field to Reviews

1. Create migration in `src/Migrations/`
2. Add field to `$fillable` in `Review` model
3. Update form requests: `StoreReviewRequest`, `StoreFrontendReviewRequest`, `UpdateReviewRequest`
4. Update form views: `backend/partials/form/details.blade.php`, `frontend/partials/form.blade.php`
5. Update display views: `backend/show.blade.php`, `frontend/show.blade.php`
6. Update export template: `exports/reviews.blade.php`

### Add a New Status Type

1. Add constant to `Review` model: `const NEW_STATUS = 'new_status';`
2. Create migration to update enum: `ALTER TABLE reviews MODIFY COLUMN status ENUM(...)`
3. Add scope to model: `public function scopeNewStatus($query) { return $query->where('status', self::NEW_STATUS); }`

### Customize Email Templates

Edit templates in `src/Views/mail/`:
- `review.blade.php` - Customer HTML email
- `review_plain.blade.php` - Customer plain text email
- `admin_review.blade.php` - Admin HTML email
- `admin_review_plain.blade.php` - Admin plain text email

### Add New Cached Statistic

1. Create view composer in `src/Http/ViewComposers/`
2. Register in `ReviewServiceProvider::$composers`
3. Add cache key to `ClearReviewCache::handle()`

## Testing Commands

```bash
# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Export reviews
php artisan review:export
```
