# Architecture Documentation - Bongo Review Package

## Overview

The Bongo Review package is a Laravel package for managing customer reviews and testimonials. It follows a modular architecture with clear separation between backend (admin) and frontend (public) concerns.

**Key Capabilities:**
- Customer review submission with spam protection (reCAPTCHA + Honeypot)
- Admin review moderation and management
- Star rating system (1-5 stars)
- Email notifications (customer confirmation + admin notification)
- Cached statistics (count, average rating, latest review)
- Excel export functionality
- UUID-based public URLs for privacy
- Event-driven cache invalidation

## Directory Structure

```
src/
├── Config/
│   └── review.php                           # Package configuration
│
├── Http/
│   ├── Controllers/
│   │   ├── Backend/
│   │   │   ├── ReviewController.php         # Admin CRUD operations
│   │   │   └── ReviewDatatableController.php # Datatable API endpoint
│   │   └── Frontend/
│   │       └── ReviewController.php         # Public review submission/listing
│   ├── Requests/
│   │   ├── StoreFrontendReviewRequest.php   # Frontend validation (with reCAPTCHA)
│   │   ├── StoreReviewRequest.php           # Backend store validation
│   │   └── UpdateReviewRequest.php          # Backend update validation
│   └── ViewComposers/
│       ├── AverageRatingComposer.php        # Injects average rating
│       ├── LatestReviewComposer.php         # Injects latest review
│       └── NumberOfReviewsComposer.php      # Injects review count
│
├── Models/
│   └── Review.php                           # Review eloquent model
│
├── Events/
│   ├── ReviewCreated.php                    # Dispatched on review creation
│   ├── ReviewUpdated.php                    # Dispatched on review update
│   └── ReviewDeleted.php                    # Dispatched on review deletion
│
├── Listeners/
│   └── ClearReviewCache.php                 # Queued listener to clear cached stats
│
├── Mailables/
│   ├── ReviewMailable.php                   # Email sent to customer
│   └── AdminReviewMailable.php              # Email sent to admin
│
├── Commands/
│   └── ExportReviewsCommand.php             # Artisan command: review:export
│
├── Exports/
│   └── ReviewExport.php                     # Excel export handler (Laravel Excel)
│
├── Migrations/
│   ├── 2021_01_01_000001_create_reviews_table.php
│   ├── 2021_01_01_000002_add_title_column_to_reviews_table.php
│   ├── 2021_01_01_000003_add_email_column_to_reviews_table.php
│   └── 2021_01_01_000004_add_date_column_to_reviews_table.php
│
├── Views/
│   ├── backend/
│   │   ├── index.blade.php                  # Admin review listing
│   │   ├── create.blade.php                 # Admin create form
│   │   ├── edit.blade.php                   # Admin edit form
│   │   ├── show.blade.php                   # Admin review detail
│   │   └── partials/form/details.blade.php  # Form fields partial
│   ├── frontend/
│   │   ├── index.blade.php                  # Public review listing
│   │   ├── show.blade.php                   # Public review detail
│   │   └── partials/
│   │       ├── form.blade.php               # Public submission form
│   │       ├── review.blade.php             # Single review display
│   │       └── summary.blade.php            # Statistics summary
│   ├── mail/
│   │   ├── review.blade.php                 # Customer email (HTML)
│   │   ├── review_plain.blade.php           # Customer email (plain)
│   │   ├── admin_review.blade.php           # Admin email (HTML)
│   │   └── admin_review_plain.blade.php     # Admin email (plain)
│   └── exports/
│       └── reviews.blade.php                # Excel export template
│
├── Routes/
│   ├── backend.php                          # Admin routes
│   └── frontend.php                         # Public routes
│
├── Seeders/
│   └── PackageSeeder.php                    # Seeds package metadata
│
├── Translations/
│   └── en/
│       ├── backend.php                      # Admin translations
│       └── frontend.php                     # Frontend translations
│
└── ReviewServiceProvider.php                # Service provider
```

## Class Diagram

```
┌─────────────────────────────────────────────────────────────────────────┐
│                         ReviewServiceProvider                           │
│                    (extends AbstractServiceProvider)                    │
├─────────────────────────────────────────────────────────────────────────┤
│ - $module: string = 'review'                                            │
│ - $commands: array = [ExportReviewsCommand]                             │
│ - $composers: array = [NumberOf..., Latest..., AverageRating...]        │
│ - $listeners: array = [ReviewCreated => [...], ReviewUpdated => [...]]  │
└─────────────────────────────────────────────────────────────────────────┘
                                    │
                                    │ bootstraps
                                    ▼
        ┌───────────────────────────────────────────────────────┐
        │                    Review Model                       │
        │              (extends AbstractModel)                  │
        ├───────────────────────────────────────────────────────┤
        │ Traits: HasContent, HasStatus, HasUUID, SoftDeletes   │
        ├───────────────────────────────────────────────────────┤
        │ Constants:                                            │
        │   PENDING = 'pending'                                 │
        │   ACTIVE = 'active'                                   │
        │   INACTIVE = 'inactive'                               │
        ├───────────────────────────────────────────────────────┤
        │ Fields:                                               │
        │   id, uuid, name, email, title, content, date,        │
        │   rating, status, created_by, updated_by, deleted_by, │
        │   created_at, updated_at, deleted_at                  │
        ├───────────────────────────────────────────────────────┤
        │ Scopes (from HasStatus):                              │
        │   + scopeActive($query)                               │
        │   + scopePending($query)                              │
        │   + scopeInactive($query)                             │
        │   + isActive(): bool                                  │
        └───────────────────────────────────────────────────────┘
                    │                           │
                    │                           │
        ┌───────────▼─────────┐     ┌──────────▼──────────┐
        │ Backend Controllers │     │ Frontend Controller │
        ├─────────────────────┤     ├─────────────────────┤
        │ ReviewController    │     │ ReviewController    │
        │ - index()           │     │ - index()           │
        │ - create()          │     │ - show(uuid)        │
        │ - store()           │     │ - store()           │
        │ - show(review)      │     │   (with emails)     │
        │ - edit(review)      │     └─────────────────────┘
        │ - update(review)    │
        │ - destroy(review)   │
        │                     │
        │ ReviewDatatable     │
        │   Controller        │
        │ - index()           │
        └─────────────────────┘

┌──────────────────────────────────────────────────────────────────────┐
│                         Event Flow                                   │
├──────────────────────────────────────────────────────────────────────┤
│  ReviewCreated ──┬──> ClearReviewCache (queued)                      │
│                  │                                                    │
│  ReviewUpdated ──┼──> ClearReviewCache (queued)                      │
│                  └──> UpdateSitemap (from bongo/sitemap)             │
│                                                                       │
│  ReviewDeleted ──> (no listeners)                                    │
└──────────────────────────────────────────────────────────────────────┘

┌──────────────────────────────────────────────────────────────────────┐
│                      View Composers                                  │
├──────────────────────────────────────────────────────────────────────┤
│  NumberOfReviewsComposer ──┐                                         │
│  LatestReviewComposer      ├──> review::frontend.partials.summary    │
│  AverageRatingComposer  ───┘                                         │
│                                                                       │
│  All inject cached data into the summary partial                     │
└──────────────────────────────────────────────────────────────────────┘
```

## Request/Response Flow Diagrams

### Frontend Review Submission Flow

```
┌─────────┐                ┌──────────────────────┐                ┌──────────────┐
│ Browser │                │ ReviewController     │                │  Review      │
│         │                │ (Frontend)           │                │  Model       │
└────┬────┘                └──────────┬───────────┘                └──────┬───────┘
     │                                │                                   │
     │ POST /reviews/store            │                                   │
     │ (with honeypot + reCAPTCHA)    │                                   │
     ├───────────────────────────────>│                                   │
     │                                │                                   │
     │                                │ StoreFrontendReviewRequest        │
     │                                │ validates data + captcha          │
     │                                │                                   │
     │                                │ create($data)                     │
     │                                ├──────────────────────────────────>│
     │                                │                                   │
     │                                │                         Review created
     │                                │                         (status: pending)
     │                                │                                   │
     │                                │ dispatch(ReviewCreated)           │
     │                                │────┐                              │
     │                                │    │                              │
     │                                │<───┘                              │
     │                                │                                   │
     │                                │ Mail::to($review->email)          │
     │                                │   ->send(ReviewMailable)          │
     │                                │────┐                              │
     │                                │    │                              │
     │                                │<───┘                              │
     │                                │                                   │
     │                                │ Mail::to(admin)                   │
     │                                │   ->send(AdminReviewMailable)     │
     │                                │────┐                              │
     │                                │    │                              │
     │                                │<───┘                              │
     │                                │                                   │
     │ redirect('/thank-you...')      │                                   │
     │<───────────────────────────────┤                                   │
     │                                │                                   │

                    ┌──────────────────────────────┐
                    │  ClearReviewCache Listener   │
                    │  (queued, runs async)        │
                    └──────────┬───────────────────┘
                               │
                               │ Cache::forget('no_of_reviews')
                               │ Cache::forget('latest_review')
                               │ Cache::forget('average_rating')
                               │
```

### Backend Review Management Flow

```
┌─────────┐            ┌──────────────────────┐            ┌──────────────┐
│  Admin  │            │ ReviewController     │            │  Review      │
│ Browser │            │ (Backend)            │            │  Model       │
└────┬────┘            └──────────┬───────────┘            └──────┬───────┘
     │                            │                               │
     │ GET /admin/reviews         │                               │
     ├───────────────────────────>│                               │
     │                            │                               │
     │ index.blade.php            │                               │
     │ (with datatable)           │                               │
     │<───────────────────────────┤                               │
     │                            │                               │
     │ GET /admin/reviews/datatable│                              │
     ├───────────────────────────>│                               │
     │                            │                               │
     │                            │ Review::query()               │
     │                            ├──────────────────────────────>│
     │                            │                               │
     │ JSON datatable response    │                               │
     │<───────────────────────────┤                               │
     │                            │                               │
     │ POST /admin/reviews/{id}/update                            │
     ├───────────────────────────>│                               │
     │                            │                               │
     │                            │ UpdateReviewRequest validates │
     │                            │                               │
     │                            │ update($data)                 │
     │                            ├──────────────────────────────>│
     │                            │                               │
     │                            │                      Review updated
     │                            │                               │
     │                            │ dispatch(ReviewUpdated)       │
     │                            │────┐                          │
     │                            │    │                          │
     │                            │<───┘                          │
     │                            │                               │
     │ redirect to show page      │                               │
     │<───────────────────────────┤                               │
     │                            │                               │
```

### View Composer Data Injection Flow

```
┌─────────────────────────────────────────────────────────────────────────┐
│  When rendering: review::frontend.partials.summary                      │
└─────────────────────────────┬───────────────────────────────────────────┘
                              │
          ┌───────────────────┼───────────────────┐
          │                   │                   │
┌─────────▼──────────┐ ┌──────▼─────────┐ ┌──────▼──────────────┐
│ NumberOfReviews    │ │ LatestReview   │ │ AverageRating       │
│ Composer           │ │ Composer       │ │ Composer            │
└─────────┬──────────┘ └────────┬───────┘ └──────┬──────────────┘
          │                     │                 │
          │ Cache::remember()   │ Cache::remember() │ Cache::remember()
          │ 'no_of_reviews'     │ 'latest_review'   │ 'average_rating'
          │                     │                   │
          ▼                     ▼                   ▼
    $noOfReviews          $latestReview      $averageRating
                                                $averageRatingText
          │                     │                   │
          └───────────────┬─────┴───────────────────┘
                          │
                          ▼
              View receives all variables:
              - $noOfReviews
              - $latestReview
              - $averageRating
              - $averageRatingText
```

## Trait/Interface Reference

### Model Traits

| Trait | Source | Purpose | Methods/Properties Added |
|-------|--------|---------|--------------------------|
| `HasContent` | `bongo/framework` | Content field management | Content field handling |
| `HasStatus` | `bongo/framework` | Status-based query scopes | `scopeActive()`, `scopePending()`, `scopeInactive()`, `isActive()`, `isPending()`, `isInactive()` |
| `HasUUID` | `bongo/framework` | Auto-generate UUID on creation | Automatically populates `uuid` field on model creation |
| `SoftDeletes` | Laravel | Soft deletion support | `deleted_at` timestamp, `trashed()`, `withTrashed()`, `onlyTrashed()`, `restore()`, `forceDelete()` |

### Listener Interfaces

| Interface | Implemented By | Purpose |
|-----------|----------------|---------|
| `ShouldQueue` | `ClearReviewCache` | Listener runs asynchronously on queue |

### Export Interfaces

| Interface | Implemented By | Purpose |
|-----------|----------------|---------|
| `FromView` | `ReviewExport` | Export data using Blade view |
| `WithStyles` | `ReviewExport` | Apply styles to Excel export |

## Database Schema

### reviews Table

| Column | Type | Constraints | Description |
|--------|------|-------------|-------------|
| `id` | increments | PRIMARY KEY | Auto-incrementing ID |
| `uuid` | uuid | INDEXED | Public identifier for review URLs |
| `name` | string | NOT NULL | Reviewer name |
| `email` | string | | Reviewer email address |
| `title` | string | | Review title |
| `content` | text | NULLABLE | Review content/body |
| `date` | date | NULLABLE | Review date (cast to Date class) |
| `rating` | unsignedTinyInteger | DEFAULT 5, INDEXED | Star rating (1-5) |
| `status` | enum | DEFAULT 'pending' | pending \| active \| inactive |
| `created_by` | unsignedInteger | NULLABLE, INDEXED | User ID who created |
| `updated_by` | unsignedInteger | NULLABLE, INDEXED | User ID who last updated |
| `deleted_by` | unsignedInteger | NULLABLE, INDEXED | User ID who deleted |
| `created_at` | timestamp | | Creation timestamp |
| `updated_at` | timestamp | | Last update timestamp |
| `deleted_at` | timestamp | NULLABLE | Soft delete timestamp |

**Indexes:**
- Primary key on `id`
- Index on `uuid`
- Index on `rating`
- Index on `created_by`, `updated_by`, `deleted_by`

## Route Registration

Routes are automatically registered by `AbstractServiceProvider` based on files in `src/Routes/`:

### Backend Routes (`backend.php`)

**Middleware Applied:** `auth`, `employee` (from AbstractServiceProvider)
**Prefix:** `/admin/{config('review.prefix')}`
**Name Prefix:** `backend.review.`

| Method | URI | Name | Controller Method |
|--------|-----|------|-------------------|
| GET | `/admin/reviews` | `backend.review.index` | `ReviewController@index` |
| GET | `/admin/reviews/create` | `backend.review.create` | `ReviewController@create` |
| POST | `/admin/reviews/store` | `backend.review.store` | `ReviewController@store` |
| GET | `/admin/reviews/datatable` | `backend.review.datatable` | `ReviewDatatableController@index` |
| GET | `/admin/reviews/{review}` | `backend.review.show` | `ReviewController@show` |
| GET | `/admin/reviews/{review}/edit` | `backend.review.edit` | `ReviewController@edit` |
| POST | `/admin/reviews/{review}/update` | `backend.review.update` | `ReviewController@update` |
| DELETE | `/admin/reviews/{review}/delete` | `backend.review.destroy` | `ReviewController@destroy` |

### Frontend Routes (`frontend.php`)

**Middleware Applied:** `web` (default), `ProtectAgainstSpam` (on store)
**Prefix:** `/{config('review.prefix')}`
**Name Prefix:** `frontend.review.`

| Method | URI | Name | Controller Method | Middleware |
|--------|-----|------|-------------------|------------|
| GET | `/reviews` | `frontend.review.index` | `ReviewController@index` | web |
| GET | `/reviews/{uuid}` | `frontend.review.show` | `ReviewController@show` | web |
| POST | `/reviews/store` | `frontend.review.store` | `ReviewController@store` | web, ProtectAgainstSpam |

## Events and Listeners

### Event Lifecycle

```
┌─────────────────────────────────────────────────────────────────────────┐
│                           ReviewCreated                                 │
├─────────────────────────────────────────────────────────────────────────┤
│ Fired When: Review created (backend or frontend)                        │
│ Listeners:                                                              │
│   - ClearReviewCache (queued)                                           │
│     • Clears 'no_of_reviews' cache                                      │
│     • Clears 'latest_review' cache                                      │
│     • Clears 'average_rating' cache                                     │
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│                           ReviewUpdated                                 │
├─────────────────────────────────────────────────────────────────────────┤
│ Fired When: Review updated in backend                                   │
│ Listeners:                                                              │
│   - ClearReviewCache (queued)                                           │
│     • Clears 'no_of_reviews' cache                                      │
│     • Clears 'latest_review' cache                                      │
│     • Clears 'average_rating' cache                                     │
│   - UpdateSitemap (from bongo/sitemap package)                          │
│     • Updates XML sitemap with latest reviews                           │
└─────────────────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────────────────┐
│                           ReviewDeleted                                 │
├─────────────────────────────────────────────────────────────────────────┤
│ Fired When: Review deleted in backend                                   │
│ Listeners: None (cache is already stale, will refresh on next access)   │
└─────────────────────────────────────────────────────────────────────────┘
```

## Caching Strategy

### Cache Keys and Data

| Cache Key | Data Type | Query | TTL |
|-----------|-----------|-------|-----|
| `no_of_reviews` | int | `Review::active()->count()` | `config('settings.cache_default')` |
| `latest_review` | Review model | `Review::active()->whereNotNull('date')->latest('date')->first()` | `config('settings.cache_default')` |
| `average_rating` | int (ceiled) | `ceil(Review::active()->average('rating'))` | `config('settings.cache_default')` |

### Cache Invalidation

Cache is cleared by `ClearReviewCache` listener when:
- Review is created (`ReviewCreated` event)
- Review is updated (`ReviewUpdated` event)

The listener implements `ShouldQueue` with 3 retry attempts, ensuring cache invalidation doesn't block the request.

## Extension Points

### 1. Adding Custom Review Fields

**Steps:**
1. Create migration in `src/Migrations/`
2. Add field to `$fillable` in `Review` model
3. Update form requests: `StoreReviewRequest`, `StoreFrontendReviewRequest`, `UpdateReviewRequest`
4. Update backend form: `src/Views/backend/partials/form/details.blade.php`
5. Update frontend form: `src/Views/frontend/partials/form.blade.php`
6. Update display views: `src/Views/backend/show.blade.php`, `src/Views/frontend/show.blade.php`
7. Update export template: `src/Views/exports/reviews.blade.php`

**Example:**
```php
// Migration
Schema::table('reviews', function (Blueprint $table) {
    $table->string('location')->nullable()->after('email');
});

// Model
protected $fillable = [..., 'location'];

// Request
public function rules(): array {
    return [..., 'location' => 'nullable|string|max:100'];
}
```

### 2. Adding Custom Status Types

**Steps:**
1. Add constant to `Review` model
2. Create migration to update enum column
3. Add scope method to model

**Example:**
```php
// Model constant
const FEATURED = 'featured';

// Migration
DB::statement("ALTER TABLE reviews MODIFY COLUMN status ENUM('pending', 'active', 'inactive', 'featured')");

// Model scope
public function scopeFeatured($query)
{
    return $query->where('status', self::FEATURED);
}
```

### 3. Adding New Cached Statistics

**Steps:**
1. Create view composer in `src/Http/ViewComposers/`
2. Register composer in `ReviewServiceProvider::$composers`
3. Add cache key to `ClearReviewCache::handle()`

**Example:**
```php
// src/Http/ViewComposers/FiveStarCountComposer.php
class FiveStarCountComposer
{
    public function compose(View $view): void
    {
        $fiveStarCount = Cache::remember('five_star_count', config('settings.cache_default'), function () {
            return Review::active()->where('rating', 5)->count();
        });

        $view->with(compact('fiveStarCount'));
    }
}

// ReviewServiceProvider
protected array $composers = [
    ...,
    FiveStarCountComposer::class => ['review::frontend.partials.summary'],
];

// ClearReviewCache listener
Cache::forget('five_star_count');
```

### 4. Custom Event Listeners

**Steps:**
1. Create listener in `src/Listeners/`
2. Register in `ReviewServiceProvider::$listeners`

**Example:**
```php
// src/Listeners/NotifySlackOnReview.php
class NotifySlackOnReview implements ShouldQueue
{
    public function handle(ReviewCreated $event): void
    {
        // Send Slack notification
    }
}

// ReviewServiceProvider
protected array $listeners = [
    ReviewCreated::class => [
        ClearReviewCache::class,
        NotifySlackOnReview::class,
    ],
];
```

### 5. Customizing Email Templates

Edit Blade templates in `src/Views/mail/`:
- `review.blade.php` - Customer HTML email
- `review_plain.blade.php` - Customer plain text email
- `admin_review.blade.php` - Admin HTML email
- `admin_review_plain.blade.php` - Admin plain text email

Variables available: `$review` (Review model instance)

### 6. Overriding Controllers

Create controller in your application that extends package controller:

```php
namespace App\Http\Controllers\Frontend;

use Bongo\Review\Http\Controllers\Frontend\ReviewController as BaseController;

class ReviewController extends BaseController
{
    public function store(StoreFrontendReviewRequest $request): RedirectResponse
    {
        // Custom logic
        return parent::store($request);
    }
}
```

Update routes in your application's route files to use custom controller.

## How to Add New Features

### Adding Review Ratings Filter (e.g., "Show only 5-star reviews")

1. **Add scope to model:**
```php
// src/Models/Review.php
public function scopeWithRating($query, int $rating)
{
    return $query->where('rating', $rating);
}
```

2. **Update frontend controller:**
```php
// src/Http/Controllers/Frontend/ReviewController.php
public function index(Request $request): View
{
    $query = Review::active()->orderBy('date', 'DESC');

    if ($request->has('rating')) {
        $query->withRating($request->integer('rating'));
    }

    return view('review::frontend.index', [
        'reviews' => $query->paginate(8),
    ]);
}
```

3. **Update view to add filter:**
```blade
<!-- src/Views/frontend/index.blade.php -->
<form method="GET">
    <select name="rating">
        <option value="">All Ratings</option>
        <option value="5">5 Stars</option>
        <option value="4">4 Stars</option>
        <!-- etc -->
    </select>
    <button type="submit">Filter</button>
</form>
```

### Adding Review Approval Workflow

1. **Update frontend controller to set status:**
```php
// src/Http/Controllers/Frontend/ReviewController.php
public function store(StoreFrontendReviewRequest $request): RedirectResponse
{
    $review = Review::create(array_merge(
        Arr::except($request->validated(), ['captcha-response']),
        ['status' => Review::PENDING] // Explicitly set pending
    ));

    // Rest of logic...
}
```

2. **Add approval method to backend controller:**
```php
// src/Http/Controllers/Backend/ReviewController.php
public function approve(Review $review): RedirectResponse
{
    $review->update(['status' => Review::ACTIVE]);
    event(new ReviewUpdated($review));

    return back()->success(trans('review::backend.approve_success'));
}
```

3. **Add route:**
```php
// src/Routes/backend.php
Route::post('{review}/approve', [ReviewController::class, 'approve'])
    ->name('approve');
```

4. **Update backend index view to show approve button**

### Adding Review Verification (Verified Purchase)

1. **Create migration:**
```php
Schema::table('reviews', function (Blueprint $table) {
    $table->boolean('verified')->default(false)->after('status');
});
```

2. **Add to fillable:**
```php
protected $fillable = [..., 'verified'];
```

3. **Add scope:**
```php
public function scopeVerified($query)
{
    return $query->where('verified', true);
}
```

4. **Update views to show verification badge**

## Dependencies and Integration

### Framework Dependencies

- **bongo/framework**: Provides base classes and traits
  - `AbstractServiceProvider` - Auto-bootstrapping
  - `AbstractModel` - Base model with audit fields
  - `AbstractController` - Base controller
  - `HasContent`, `HasStatus`, `HasUUID` traits
  - `Date` cast

- **bongo/captcha**: Provides reCAPTCHA validation
  - `Captcha` validation rule

### External Package Dependencies

- **spatie/laravel-honeypot**: Spam protection on frontend submission
- **maatwebsite/laravel-excel**: Excel export functionality
- **Laravel Mail**: Email notifications
- **Laravel Cache**: Statistics caching
- **Laravel Events**: Event-driven architecture

### Optional Integration

- **bongo/sitemap**: Automatically updates XML sitemap when reviews are updated (listener registered in service provider)
