# Bongo Review Package

A Laravel package for managing customer reviews and testimonials with both admin and public interfaces.

[![Latest Version](https://img.shields.io/badge/version-3.0-blue.svg)](https://designtecpackages.co.uk)
[![PHP Version](https://img.shields.io/badge/php-8.2%2B-blue.svg)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-10%2B-red.svg)](https://laravel.com)

## Features

- 📝 Customer review submission with spam protection (reCAPTCHA + Honeypot)
- 👨‍💼 Admin review moderation and CRUD management
- ⭐ Star rating system (1-5 stars)
- 📧 Email notifications (customer confirmation + admin notification)
- 📊 Cached statistics (review count, average rating, latest review)
- 📤 Excel export functionality via artisan command
- 🔒 UUID-based public URLs for privacy
- 🚀 Event-driven cache invalidation
- ✅ Review approval workflow (pending/active/inactive)

## Requirements

- PHP 8.2 or higher
- Laravel 10 or higher
- Composer

## Installation

### Step 1: Install via Composer

```bash
composer require bongo/review
```

### Step 2: Run Migrations

```bash
php artisan migrate
```

The package will automatically register its service provider and migrations.

### Step 3: Publish Configuration (Optional)

```bash
php artisan vendor:publish --provider="Bongo\Review\ReviewServiceProvider"
```

This will publish the configuration file to `config/review.php`.

## Configuration

Edit `config/review.php` to customize the package:

```php
return [
    // Route prefix for review URLs
    'prefix' => 'reviews',

    // Redirect URL after successful frontend submission
    'success_url' => '/thank-you-for-your-review',

    // reCAPTCHA settings
    'recaptcha' => [
        'enabled' => true,    // Enable/disable reCAPTCHA validation
        'min_score' => 0.5,   // Minimum score required (0.1 - 1.0)
    ],
];
```

## Usage

### Admin Interface

Access the admin interface at `/admin/reviews` (requires authentication):

- **List Reviews**: View all reviews in a datatable
- **Create Review**: Manually create reviews
- **Edit Review**: Modify existing reviews
- **Moderate**: Change review status (pending/active/inactive)
- **Delete**: Soft delete reviews

### Frontend Interface

Public review pages:

- **Review List**: `/reviews` - Display all active reviews with pagination
- **Single Review**: `/reviews/{uuid}` - View individual review by UUID
- **Submit Review**: POST to `/reviews/store` - Submit new review

### Displaying Reviews

```blade
{{-- Display paginated reviews --}}
@foreach($reviews as $review)
    <div class="review">
        <h3>{{ $review->title }}</h3>
        <p>Rating: {{ $review->rating }}/5</p>
        <p>{{ $review->content }}</p>
        <p>By {{ $review->name }} on {{ $review->date }}</p>
    </div>
@endforeach

{{ $reviews->links() }}
```

### Review Statistics

The package provides cached statistics via view composers:

```blade
{{-- This data is automatically injected into review::frontend.partials.summary --}}
<div class="review-stats">
    <p>Total Reviews: {{ $noOfReviews }}</p>
    <p>Average Rating: {{ $averageRating }}/5 ({{ $averageRatingText }})</p>
    @if($latestReview)
        <p>Latest: {{ $latestReview->title }}</p>
    @endif
</div>
```

### Querying Reviews

```php
use Bongo\Review\Models\Review;

// Get active reviews
$activeReviews = Review::active()->get();

// Get pending reviews (awaiting moderation)
$pendingReviews = Review::pending()->get();

// Get reviews by rating
$fiveStarReviews = Review::active()->where('rating', 5)->get();

// Get latest review
$latest = Review::active()->latest('date')->first();

// Get average rating
$avgRating = ceil(Review::active()->average('rating'));
```

### Review Submission Flow

1. Customer submits review via frontend form
2. Validation performed (including reCAPTCHA if enabled)
3. Review created with 'pending' status
4. Two emails sent:
   - Confirmation email to customer
   - Notification email to admin
5. Customer redirected to thank you page
6. Admin moderates review (approve/reject)
7. Cache automatically cleared when review is created/updated

## Artisan Commands

### Export Reviews to CSV

```bash
php artisan review:export
```

Exports all reviews to `storage/app/reviews.csv`.

## Events

The package fires the following events:

- `Bongo\Review\Events\ReviewCreated` - When a review is created
- `Bongo\Review\Events\ReviewUpdated` - When a review is updated
- `Bongo\Review\Events\ReviewDeleted` - When a review is deleted

### Listening to Events

```php
// In your EventServiceProvider
use Bongo\Review\Events\ReviewCreated;

protected $listen = [
    ReviewCreated::class => [
        YourCustomListener::class,
    ],
];
```

## Email Templates

The package includes email templates for:

- **Customer Confirmation**: `review::mail.review` (HTML) and `review::mail.review_plain` (plain text)
- **Admin Notification**: `review::mail.admin_review` (HTML) and `review::mail.admin_review_plain` (plain text)

Customize templates by publishing views:

```bash
php artisan vendor:publish --provider="Bongo\Review\ReviewServiceProvider" --tag="views"
```

## Caching

The package caches the following statistics:

- `no_of_reviews` - Total count of active reviews
- `latest_review` - Most recent active review
- `average_rating` - Average rating of active reviews

Cache is automatically invalidated when reviews are created or updated.

**Cache TTL**: Configured via `config('settings.cache_default')`

## Security Features

- **reCAPTCHA v3**: Prevents automated spam submissions
- **Honeypot**: Additional spam protection (Spatie package)
- **Email Validation**: RFC and DNS validation
- **Admin Middleware**: Admin routes protected by authentication
- **Moderation**: Reviews default to 'pending' status
- **UUID URLs**: Public reviews use UUID instead of sequential IDs

## Database Schema

The package creates a `reviews` table with the following columns:

| Column | Type | Description |
|--------|------|-------------|
| id | increments | Primary key |
| uuid | uuid | Public identifier (indexed) |
| name | string | Reviewer name |
| email | string | Reviewer email |
| title | string | Review title |
| content | text | Review content |
| date | date | Review date |
| rating | unsignedTinyInteger | Star rating (1-5) |
| status | enum | pending \| active \| inactive |
| created_by | unsignedInteger | User who created |
| updated_by | unsignedInteger | User who last updated |
| deleted_by | unsignedInteger | User who deleted |
| created_at | timestamp | Creation timestamp |
| updated_at | timestamp | Update timestamp |
| deleted_at | timestamp | Soft delete timestamp |

## API Routes

### Backend (Admin)

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/admin/reviews` | `backend.review.index` | List all reviews |
| GET | `/admin/reviews/create` | `backend.review.create` | Show create form |
| POST | `/admin/reviews/store` | `backend.review.store` | Store new review |
| GET | `/admin/reviews/datatable` | `backend.review.datatable` | Datatable API |
| GET | `/admin/reviews/{id}` | `backend.review.show` | Show review detail |
| GET | `/admin/reviews/{id}/edit` | `backend.review.edit` | Show edit form |
| POST | `/admin/reviews/{id}/update` | `backend.review.update` | Update review |
| DELETE | `/admin/reviews/{id}/delete` | `backend.review.destroy` | Delete review |

### Frontend (Public)

| Method | URI | Name | Description |
|--------|-----|------|-------------|
| GET | `/reviews` | `frontend.review.index` | List active reviews |
| GET | `/reviews/{uuid}` | `frontend.review.show` | Show review by UUID |
| POST | `/reviews/store` | `frontend.review.store` | Submit new review |

## Development

### Running Tests

```bash
vendor/bin/phpunit
```

### Code Style

```bash
# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

### Static Analysis

```bash
vendor/bin/phpstan analyse
```

## Documentation

- **Architecture**: See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed architecture documentation
- **Development Guide**: See [.cursorrules](.cursorrules) for development guidelines
- **Code Templates**: See [.github/copilot-instructions.md](.github/copilot-instructions.md) for code templates

## Dependencies

This package depends on:

- `bongo/framework` ^3.0 - Provides base classes, traits, and helpers
- `bongo/captcha` - Provides reCAPTCHA validation
- Laravel Mail - Email notifications
- Laravel Cache - Statistics caching
- Spatie Honeypot - Spam protection
- Maatwebsite Excel - Export functionality

## Changelog

Please see [CHANGELOG.md](CHANGELOG.md) for version history and changes.

## License

This package is licensed under the MIT License. See [LICENSE](LICENSE) for details.

## Credits

Developed by [Stuart Elliott](https://bespokeuk.com) at [Bespoke UK](https://bespokeuk.com).

## Support

For issues, questions, or feature requests, please contact the development team or create an issue in the project repository.
