<?php

namespace Bongo\Review\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Review\Events\ReviewCreated;
use Bongo\Review\Events\ReviewDeleted;
use Bongo\Review\Events\ReviewUpdated;
use Bongo\Review\Http\Requests\StoreReviewRequest;
use Bongo\Review\Http\Requests\UpdateReviewRequest;
use Bongo\Review\Models\Review;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

/**
 * Class ReviewController.
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Review\Http\Controllers\Backend
 */
class ReviewController extends AbstractController
{
    /**
     * @var Review
     */
    protected $review;

    /**
     * ReviewController constructor.
     *
     * @param Review $review
     */
    public function __construct(Review $review)
    {
        $this->review = $review;
    }

    /**
     * Show the review list view.
     *
     * @return View
     */
    public function index(): View
    {
        return view('review::backend.index');
    }

    /**
     * Show the review create view.
     *
     * @return View
     */
    public function create(): View
    {
        $review = $this->review;

        return view('review::backend.create', compact('review'));
    }

    /**
     * Store a new review.
     *
     * @param StoreReviewRequest $request
     *
     * @return RedirectResponse
     * @throws Exception
     */
    public function store(StoreReviewRequest $request): RedirectResponse
    {
        $review = $this->review->create($request->all());
        event(new ReviewCreated($review));

        return redirect()
            ->route('backend.review.show', $review->id)
            ->success(trans('review::backend.store_success'));
    }

    /**
     * Show the review view.
     *
     * @param Review $review
     *
     * @return View
     */
    public function show(Review $review): View
    {
        return view('review::backend.show', compact('review'));
    }

    /**
     * Show the review edit view.
     *
     * @param Review $review
     *
     * @return View
     */
    public function edit(Review $review): View
    {
        return view('review::backend.edit', compact('review'));
    }

    /**
     * Update the given review.
     *
     * @param UpdateReviewRequest $request
     * @param Review              $review
     *
     * @return RedirectResponse
     * @throws Exception
     */
    public function update(UpdateReviewRequest $request, Review $review): RedirectResponse
    {
        $review->update($request->all());
        event(new ReviewUpdated($review));

        return redirect()
            ->route('backend.review.show', $review->id)
            ->success(trans('review::backend.update_success'));
    }

    /**
     * Destroy a review.
     *
     * @param Review $review
     *
     * @return mixed
     * @throws Exception
     */
    public function destroy(Review $review)
    {
        if ($review->delete()) {
            event(new ReviewDeleted($review));
            $redirect = back();

            // If the review was deleted from their detail view,
            // we'll redirect back to the review list view.
            if (route('backend.review.show', $review) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.review.index');
            }

            return $redirect->success(trans('review::backend.delete_success'));
        }

        return back()->error(trans('review::backend.delete_failed'));
    }
}
