<?php

namespace Bongo\Review\Http\Controllers\Frontend;

use Bongo\Review\Events\ReviewCreated;
use Bongo\Review\Http\Requests\StoreFrontendReviewRequest;
use Bongo\Review\Mailables\AdminReviewMailable;
use Bongo\Review\Mailables\ReviewMailable;
use Bongo\Review\Models\Review;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Mail;
use Illuminate\View\View;

class ReviewController extends AbstractController
{
    public function index(): View
    {
        return view('review::frontend.index', [
            'reviews' => Review::active()->orderBy('date', 'DESC')->paginate(8),
        ]);
    }

    /**
     * @throws BindingResolutionException
     */
    public function store(StoreFrontendReviewRequest $request): RedirectResponse
    {
        $review = Review::create(Arr::except($request->validated(), ['g-recaptcha-response']));
        event(new ReviewCreated($review));

        // Send the email to the user
        Mail::to($review->email)
            ->send(new ReviewMailable($review));

        // Send the email to the admin
        Mail::to(setting("client::company.email"))
            ->send(new AdminReviewMailable($review));

        // Redirect the user to the thank-you screen
        return redirect()
            ->to(config('review.success_url'))
            ->success(trans('review::frontend.store_success'));
    }

    public function show(string $uuid): View
    {
        $review = Review::where('uuid', '=', $uuid)->first();
        if (! $review || (! $review->isActive() && ! auth()->user())) {
            abort('404', 'Review could not be found');
        }

        return view('review::frontend.show', compact('review'));
    }
}
