<?php

namespace Bongo\Setting\Http\Controllers\Backend;

use Bongo\Framework\Helpers\CloudFlare;
use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Setting\Actions\CompileCss;
use Bongo\Setting\Events\CssUpdated;
use Bongo\Setting\Events\JsUpdated;
use Bongo\Setting\Events\SettingUpdated;
use Bongo\Setting\Http\Requests\UpdateSettingRequest;
use Bongo\Setting\Models\Setting;
use Bongo\Setting\Services\EnvironmentManager;
use Bongo\Setting\Services\RobotsManager;
use Bongo\Sitemap\Actions\GenerateSitemap;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Artisan;
use Illuminate\View\View;

/**
 * Class SettingController
 *
 * @category  Bespoke_Software
 * @author    Bespoke.ws Ltd <support@bespokeuk.com>
 * @copyright 2015-2020 Bespoke.ws Ltd, All Rights Reserved
 * @license   Proprietary and confidential
 * Unauthorized copying of this file, via any medium is strictly prohibited
 * @link      https://bespokeuk.com
 * @package   Bongo\Setting\Http\Controllers\Backend
 */
class SettingController extends AbstractController
{
    /**
     * Show the settings view.
     *
     * @return View
     */
    public function index(): View
    {
        return view('setting::backend.index')->with([
            'env' => (new EnvironmentManager())->getAsArray()
        ]);
    }

    /**
     * Show the setting edit view.
     *
     * @return View
     */
    public function edit(): View
    {
        return view('setting::backend.edit')->with([
            'env' => (new EnvironmentManager())->getAsArray()
        ]);
    }

    /**
     * Update the given setting.
     *
     * @param UpdateSettingRequest $request
     *
     * @return RedirectResponse
     */
    public function update(UpdateSettingRequest $request): RedirectResponse
    {
        // Get the settings
        $settings = $request->get('settings');

        // For each setting update the value
        foreach ($settings as $id => $value) {
            // Find the setting
            if ($setting = Setting::find($id)) {
                // Check if its been updated
                if ($setting->value!=$value) {
                    $setting->value = $value;
                    $changes        = $setting->getDiff();
                    $setting->save();
                    event(new SettingUpdated($setting, $changes));
                }
            }
        }

        // Update the env settings
        (new RobotsManager())->set($request);
        (new EnvironmentManager())->set($request);

        // reset cache & config
        Artisan::call('cache:clear');
        Artisan::call('config:clear');
        Artisan::call('view:clear');

        // Update the frontend CSS & JS
        event(new CssUpdated($request->get('custom_css')));
        event(new JsUpdated($request->get('custom_js')));

        return redirect()
            ->route('backend.setting.edit')
            ->success(trans('setting::backend.update_success'));
    }

    /**
     * Show the application dashboard.
     *
     * @return RedirectResponse
     */
    public function clearCache(): RedirectResponse
    {
        Artisan::call('config:clear');
        Artisan::call('route:clear');
        Artisan::call('view:clear');
        Artisan::call('cache:clear');

        // Make a http call to the cloudflare api to clear the cache
        if (config('cloudflare.enabled')) {
            $cloudflare = new CloudFlare();
            if ($cloudflare->hasValidCredentials() && $zoneId = $cloudflare->getZoneId()) {
                $cloudflare->clearCache($zoneId);
            }
        }

        return redirect()->back()->success('Application cache cleared!');
    }

    /**
     * @return RedirectResponse
     */
    public function compileCss(): RedirectResponse
    {
        (new CompileCss())->execute();

        return redirect()->back()->success('CSS has been updated!');
    }

    /**
     * @return RedirectResponse
     */
    public function generateSitemap(): RedirectResponse
    {
        (new GenerateSitemap())->execute();

        return redirect()->back()->success('Sitemap has been updated!');
    }
}
