<?php

namespace Bongo\Setting\Services;

use Bongo\Referrer\Services\Referrer;
use Bongo\Setting\Models\Setting;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;

class SettingManager
{
    protected $items;

    public function __construct()
    {
        $this->items = new Collection;

        if (Schema::hasTable('settings')) {
            $this->items = Cache::rememberForever("settings", function () {
                return Setting::orderBy('namespace')
                    ->orderBy('group')
                    ->orderBy('sort_order')
                    ->orderBy('key')
                    ->get();
            });
        }
    }

    public function has(string $key)
    {
        return $this->items->contains('full_key', $key);
    }

    public function get(string $key, $default = null)
    {
        if ($this->has($key)
            && $setting = $this->items->firstWhere('full_key', $key)
        ) {
            return $setting->isCheckbox() ? intval($setting->value) : $setting->value;
        }

        return $default;
    }

    public function all()
    {
        return $this->items;
    }

    public function allByNamespace(): array
    {
        $settings = [];
        foreach ($this->items as $item) {
            $settings["{$item->namespace}"]["{$item->group}"]["{$item->key}"] = $item;
        }

        return $settings;
    }

    public function getCustomCss()
    {
        if (file_exists(resource_path('/sass/custom.scss'))) {
            return file_get_contents(resource_path('/sass/custom.scss'));
        }

        return null;
    }

    public function setCustomCss($css)
    {
        file_put_contents(resource_path('/sass/custom.scss'), $css);
    }

    public function getCustomJs()
    {
        if (file_exists(resource_path('/js/custom.js'))) {
            return file_get_contents(resource_path('/js/custom.js'));
        }

        return null;
    }

    public function setCustomJs($js)
    {
        file_put_contents(resource_path('/js/custom.js'), $js);
        file_put_contents(public_path('/js/custom.js'), $js);
    }

    public function hasTransparentHeader(): bool
    {
        return $this->get("theme::layout.transparent_header") === 1;
    }

    public function hasStickyHeader(): bool
    {
        return $this->get("theme::layout.sticky_header") === 1;
    }

    public function getHeaderClass(): string
    {
        $headerClass = '';

        if ($this->hasTransparentHeader()) {
            $headerClass .= ' is-transparent';
        }
        if ($this->hasStickyHeader()) {
            $headerClass .= ' is-sticky';
        }

        return $headerClass;
    }

    public function getClientPhoneNumber(): ?string
    {
        // Check for Google Analytics
        if (app(Referrer::class)->isGoogleAds() && ! empty($this->get("client::marketing.google_ads_phone"))) {
            return $this->get("client::marketing.google_ads_phone");
        }

        // Check for Facebook
        if (app(Referrer::class)->isFacebookAds() && ! empty($this->get("client::marketing.facebook_ads_phone"))) {
            return $this->get("client::marketing.facebook_ads_phone");
        }

        // Check for LinkedIn
        if (app(Referrer::class)->isLinkedinAds() && ! empty($this->get("client::marketing.linkedin_ads_phone"))) {
            return $this->get("client::marketing.linkedin_ads_phone");
        }

        // Check for normal company number
        if (! empty($this->get("client::company.phone"))) {
            return $this->get("client::company.phone");
        }

        return null;
    }

    public function isNoFollow(): bool
    {
        return $this->get("system::developer.credit_type") === 'nofollow';
    }

    public function getReCaptchaSiteKey(): ?string
    {
        return ! empty($this->get("system::credentials.google_recaptcha_site_key"))
            ? $this->get("system::credentials.google_recaptcha_site_key")
            : config('recaptchav3.sitekey');
    }

    public function getReCaptchaSecretKey(): ?string
    {
        return ! empty($this->get("system::credentials.google_recaptcha_secret_key"))
            ? $this->get("system::credentials.google_recaptcha_secret_key")
            : config('recaptchav3.secret');
    }

    public function reCaptchaEnabled(): bool
    {
        return ! empty($this->getReCaptchaSiteKey())
            && ! empty($this->getReCaptchaSecretKey());
    }
}
