# Architecture Documentation - Setting Package

## Overview

The **bongo/setting** package provides a comprehensive settings management system for Laravel applications. Settings are stored in a database with hierarchical keys, cached for performance, and accessed through a singleton service or global helper function. The package implements an event-driven architecture where setting changes trigger listeners that update module statuses, compile CSS, and clear caches.

**Package Type**: Laravel package in Bongo monorepo (default/setting)
**Namespace**: `Bongo\Setting`
**Dependencies**: bongo/framework (^3.0), scssphp/scssphp (^1.11)

## Directory Structure

```
src/
├── Actions/                          # Executable actions
│   ├── ClearCache.php                # Clears all Laravel caches
│   ├── CompileCss.php                # Compiles SCSS with theme variables
│   └── UpdateSettings.php            # Updates settings and fires events
├── Events/                           # Domain events
│   ├── CssUpdated.php                # Fired when site CSS updated
│   ├── JsUpdated.php                 # Fired when site JS updated
│   ├── SchemaUpdated.php             # Fired when site schema updated
│   └── SettingUpdated.php            # Fired when any setting changes
├── Http/
│   ├── Controllers/Backend/          # Admin settings controllers
│   │   ├── CacheController.php       # Clear cache endpoint
│   │   ├── ClientController.php      # Client settings
│   │   ├── CssController.php         # Compile CSS endpoint
│   │   ├── EnvironmentController.php # Environment settings
│   │   ├── EstimateController.php    # Estimate settings
│   │   ├── PackageController.php     # Package toggles
│   │   ├── SettingController.php     # Main settings router
│   │   ├── SiteCssController.php     # Custom CSS editor
│   │   ├── SiteJsController.php      # Custom JS editor
│   │   ├── SiteSchemaController.php  # Custom schema editor
│   │   ├── SitemapController.php     # Generate sitemap
│   │   ├── SystemController.php      # System settings
│   │   └── ThemeController.php       # Theme settings
│   └── Requests/
│       └── UpdateEnvironment.php     # Environment update validation
├── Interfaces/                       # Constants and contracts
│   ├── DataTypes.php                 # Data type constants (BOOLEAN, STRING, etc.)
│   ├── Groups.php                    # Group constants (COMPANY, SOCIAL, etc.)
│   ├── InputTypes.php                # Input type constants (CHECKBOX, SELECT, etc.)
│   └── Namespaces.php                # Namespace constants (CLIENT, THEME, etc.)
├── Listeners/                        # Event listeners
│   ├── UpdateCustomSchema.php        # Updates custom schema file
│   ├── UpdateEstimateModuleStatus.php # Toggles estimate module
│   ├── UpdateFrontendCss.php         # Updates frontend CSS file
│   ├── UpdateFrontendJs.php          # Updates frontend JS file
│   ├── UpdateGalleryModuleStatus.php # Toggles gallery module
│   ├── UpdateOpenAIModuleStatus.php  # Toggles OpenAI module
│   ├── UpdatePostModuleStatus.php    # Toggles post module
│   ├── UpdateProjectModuleStatus.php # Toggles project module
│   ├── UpdateRealGreenModuleStatus.php # Toggles RealGreen integration
│   └── UpdateReviewModuleStatus.php  # Toggles review module
├── Migrations/                       # Database migrations
│   ├── 2020_01_01_000001_create_settings_table.php
│   ├── 2021_01_01_000002_add_logo_height_settings.php
│   ├── 2021_01_01_000003_add_mobile_logo_height_setting.php
│   └── 2021_01_01_000004_add_additional_types_to_settings_table.php
├── Models/
│   └── Setting.php                   # Setting model with hierarchical keys
├── Routes/
│   └── backend.php                   # Admin routes for settings pages
├── Seeders/
│   └── DataSeeder.php                # Seeds default settings
├── Services/
│   └── SettingManager.php            # Main service for accessing settings
├── Traits/
│   └── HasType.php                   # Type checking methods (isCheckbox, etc.)
├── Translations/en/                  # Language files
├── Views/backend/                    # Blade templates for admin UI
│   ├── client/                       # Client settings views
│   ├── environment/                  # Environment settings views
│   ├── estimate/                     # Estimate settings views
│   ├── package/                      # Package settings views
│   ├── partials/                     # Reusable view components
│   ├── site_css/                     # CSS editor views
│   ├── site_js/                      # JS editor views
│   ├── site_schema/                  # Schema editor views
│   ├── system/                       # System settings views
│   └── theme/                        # Theme settings views
├── helpers.php                       # Global setting() helper
└── SettingServiceProvider.php        # Service provider
```

## Class Diagrams

### Core Classes

```
┌────────────────────────────────────────┐
│   AbstractServiceProvider              │
│   (from bongo/framework)               │
└────────────────┬───────────────────────┘
                 │ extends
                 │
┌────────────────▼───────────────────────┐
│   SettingServiceProvider               │
├────────────────────────────────────────┤
│ + $module: string = 'setting'          │
│ + $listeners: array                    │
├────────────────────────────────────────┤
│ + register(): void                     │
│ + boot(): void                         │
└────────────────────────────────────────┘
         │
         │ registers
         │
         ▼
┌────────────────────────────────────────┐
│   SettingManager (Singleton)           │
├────────────────────────────────────────┤
│ - $items: Collection                   │
├────────────────────────────────────────┤
│ + has(string $key): bool               │
│ + get(string $key, $default): mixed    │
│ + all(): Collection                    │
│ + allByNamespace(): array              │
│ + getByNamespace(string): array        │
│ + getCustomCss(): ?string              │
│ + setCustomCss(?string): void          │
│ + getCustomJs(): ?string               │
│ + setCustomJs(?string): void           │
│ + getCustomSchema(): ?string           │
│ + setCustomSchema(?string): void       │
│ + hasTransparentHeader(): bool         │
│ + hasStickyHeader(): bool              │
│ + getHeaderClass(): string             │
│ + getClientPhoneNumber(): ?string      │
│ + getRegisteredAddress(): ?string      │
│ + getOfficeAddress(): ?string          │
│ + isNoFollow(): bool                   │
│ + getEstimateAreas(): array            │
│ + getOpenAIApiKey(): ?string           │
│ + getReCaptchaSiteKey(): ?string       │
│ + getReCaptchaSecretKey(): ?string     │
│ + reCaptchaEnabled(): bool             │
└────────────────────────────────────────┘
         │ queries
         │
         ▼
┌────────────────────────────────────────┐
│   Setting extends AbstractModel        │
├────────────────────────────────────────┤
│ + namespace: string                    │
│ + group: string                        │
│ + name: string                         │
│ + key: string                          │
│ + value: string                        │
│ + rules: ?string                       │
│ + options: ?array                      │
│ + type: string                         │
│ + data: string                         │
│ + sort_order: ?int                     │
├────────────────────────────────────────┤
│ + getFullKeyAttribute(): string        │
│ + getDynamicOptions(): array           │
│ + isEnabled(): bool                    │
│ + isDisabled(): bool                   │
│ + isCheckbox(): bool                   │
│ + isColorPicker(): bool                │
│ + isDynamic(): bool                    │
│ + isInput(): bool                      │
│ + isSelect(): bool                     │
│ + isText(): bool                       │
└────────────────────────────────────────┘
         │ implements
         │
         ▼
┌─────────────────────────────────────────────────────────┐
│  DataTypes | Groups | InputTypes | Namespaces          │
│  (Constants Interfaces)                                 │
└─────────────────────────────────────────────────────────┘
```

### Action Classes

```
┌──────────────────────┐
│  UpdateSettings      │
├──────────────────────┤
│ + execute(array)     │
└──────────────────────┘
         │ uses
         ▼
┌──────────────────────┐       ┌──────────────────────┐
│  Setting::find()     │──────▶│  SettingUpdated      │
│  Setting::save()     │       │  (Event)             │
└──────────────────────┘       └──────────────────────┘


┌──────────────────────┐
│  ClearCache          │
├──────────────────────┤
│ + execute(): void    │
└──────────────────────┘
         │ calls
         ▼
┌──────────────────────┐
│  Artisan::call()     │
│  - event:clear       │
│  - view:clear        │
│  - cache:clear       │
│  - route:clear       │
│  - config:clear      │
│  - clear-compiled    │
└──────────────────────┘


┌──────────────────────┐
│  CompileCss          │
├──────────────────────┤
│ + execute(): void    │
└──────────────────────┘
         │ uses
         ▼
┌──────────────────────┐       ┌──────────────────────┐
│  SettingManager      │──────▶│  ScssPhp\Compiler    │
│  (get theme vars)    │       │  (compile SCSS)      │
└──────────────────────┘       └──────────────────────┘
                                         │
                                         ▼
                               ┌──────────────────────┐
                               │  public/css/         │
                               │  frontend.css        │
                               └──────────────────────┘
```

## Lifecycle and Flow Diagrams

### Setting Update Flow

```
1. User submits form
       │
       ▼
2. Controller receives request
       │
       ▼
3. UpdateSettings::execute($request->get('settings'))
       │
       ├─────▶ Loop through settings array
       │            │
       │            ▼
       │       Find Setting by ID
       │            │
       │            ▼
       │       Check if value changed
       │            │
       │            ▼ (if changed)
       │       Update value
       │            │
       │            ▼
       │       Save to database
       │            │
       │            ▼
       │       Fire SettingUpdated event
       │            │
       │            └─────▶ Event Listeners:
       │                         - UpdateEstimateModuleStatus
       │                         - UpdateRealGreenModuleStatus
       │                         - UpdateGalleryModuleStatus
       │                         - UpdateOpenAIModuleStatus
       │                         - UpdatePostModuleStatus
       │                         - UpdateProjectModuleStatus
       │                         - UpdateReviewModuleStatus
       │                         - ClearMenuCache
       │
       ▼
4. ClearCache::execute()
       │
       ├─────▶ event:clear
       ├─────▶ view:clear
       ├─────▶ cache:clear (clears settings cache)
       ├─────▶ route:clear
       ├─────▶ config:clear
       └─────▶ clear-compiled
       │
       ▼
5. Redirect with success message
```

### CSS Compilation Flow

```
1. User triggers compile CSS
       │
       ▼
2. CssController invoked
       │
       ▼
3. CompileCss::execute()
       │
       ├─────▶ Get theme variables from settings:
       │            - Logo heights (header, mobile, sticky, footer)
       │            - Colours (primary, secondary, tertiary, quaternary, color-5 to color-8)
       │            - Skin name
       │
       ├─────▶ Initialize ScssPhp\Compiler
       │            - Add variables
       │            - Set import paths (resource_path('sass'))
       │            - Set output style (COMPRESSED or EXPANDED)
       │
       ├─────▶ Compile SCSS in order:
       │            1. skins/{skin}/_variables.scss
       │            2. bootstrap/bootstrap.scss
       │            3. bootstrap/_overrides.scss
       │            4. skins/{skin}/_overrides.scss
       │            5. vendor/index.scss
       │            6. components/index.scss
       │            7. custom.scss
       │
       ├─────▶ Write output to:
       │            public/css/frontend.css
       │            (and public/css/frontend.map if DEBUG_CSS=true)
       │
       └─────▶ Return success or error
```

### Setting Access Flow

```
Helper Function Call: setting('theme::color.primary')
       │
       ▼
1. Check if key provided
       │
       ├─────▶ NO: Return SettingManager instance
       │
       └─────▶ YES: Call SettingManager::get($key, $default)
                     │
                     ▼
              2. Check cache for 'settings' key
                     │
                     ├─────▶ MISS: Query database
                     │            - SELECT * FROM settings
                     │            - ORDER BY namespace, group, sort_order, key
                     │            - Cache::rememberForever('settings', ...)
                     │
                     └─────▶ HIT: Use cached collection
                               │
                               ▼
                     3. Filter collection by full_key
                               │
                               ▼
                     4. Return value (cast for checkboxes)
```

### Module Status Update Flow

```
SettingUpdated Event Fired
       │
       ▼
Listener: UpdateEstimateModuleStatus
       │
       ▼
Check if $setting->full_key === 'package::estimate.estimate_module'
       │
       ├─────▶ NO: Do nothing
       │
       └─────▶ YES: Update Package model
                     │
                     ▼
              Package::where('key', 'estimate')
                ->update(['is_active' => $setting->isEnabled()])
                     │
                     ▼
              Module status synced with setting
```

## Hierarchical Key System

### Key Format

```
namespace::group.key
    │       │     │
    │       │     └─── Unique identifier within group
    │       └───────── Logical grouping of related settings
    └───────────────── Top-level category

Examples:
┌────────────┬────────────┬──────────────────────────────────────┐
│ Namespace  │ Group      │ Example Keys                         │
├────────────┼────────────┼──────────────────────────────────────┤
│ client     │ company    │ name, phone, email, website          │
│ client     │ social     │ facebook, twitter, instagram         │
│ client     │ marketing  │ google_ads_phone, facebook_ads_phone │
│ client     │ office     │ line_1, city, postcode               │
│ client     │ registered │ line_1, city, postcode               │
│ theme      │ logo       │ default, light, dark, header_height  │
│ theme      │ color      │ primary, secondary, tertiary         │
│ theme      │ layout     │ skin, header, footer, transparent    │
│ system     │ credentials│ google_maps_api_key, openai_api_key  │
│ system     │ developer  │ credit_link, credit_text, credit_type│
│ system     │ misc       │ version, site_visibility             │
│ package    │ estimate   │ estimate_module, step_1, sales_email │
│ package    │ gallery    │ gallery_module                       │
│ package    │ openai     │ openai_module                        │
│ package    │ post       │ post_module, related_type            │
│ package    │ project    │ project_module, meta_title           │
│ estimate   │ area_1     │ name, radius_in_miles, from_latitude │
│ estimate   │ area_2     │ name, radius_in_miles, from_latitude │
└────────────┴────────────┴──────────────────────────────────────┘
```

### Key Generation

```php
// In Setting model
public function getFullKeyAttribute(): string
{
    return "{$this->namespace}::{$this->group}.{$this->key}";
}

// Example:
Setting {
    namespace: 'theme',
    group: 'color',
    key: 'primary'
}
→ full_key: 'theme::color.primary'
```

## Trait and Interface Reference

### Traits

| Trait | Source | Purpose | Methods |
|-------|--------|---------|---------|
| `HasKey` | bongo/framework | Generates snake_case key from name | `getKeyAttribute()` |
| `HasType` | src/Traits/HasType.php | Type checking methods | `isCheckbox()`, `isColorPicker()`, `isDynamic()`, `isInput()`, `isSelect()`, `isText()` |

### Interfaces (Constants)

| Interface | File | Purpose | Constants |
|-----------|------|---------|-----------|
| `Namespaces` | src/Interfaces/Namespaces.php | Top-level categories | CLIENT, PACKAGE, ESTIMATE_AREA, THEME, SYSTEM |
| `Groups` | src/Interfaces/Groups.php | Setting groupings | COLOR, COMPANY, CREDENTIALS, DEVELOPER, ESTIMATE, GALLERY, LAYOUT, LOGO, MARKETING, MISC, OFFICE, OPENAI, PAGE, POST, PROJECT, REGISTERED, REVIEW, SOCIAL, AREA_1 to AREA_5 |
| `InputTypes` | src/Interfaces/InputTypes.php | Form input types | CHECKBOX, COLOR_PICKER, DATE_PICKER, DYNAMIC, INPUT, SELECT, TEXT |
| `DataTypes` | src/Interfaces/DataTypes.php | Value data types | BOOLEAN, DECIMAL, INTEGER, JSON, STRING |

## Extension Points

### 1. Adding a New Namespace

```php
// src/Interfaces/Namespaces.php
interface Namespaces
{
    public const CLIENT = 'client';
    public const PACKAGE = 'package';
    public const NEW_NAMESPACE = 'new_namespace';  // Add here

    public const NAMESPACES = [
        self::CLIENT,
        self::PACKAGE,
        self::NEW_NAMESPACE,  // Add to array
    ];
}
```

### 2. Adding a New Group

```php
// src/Interfaces/Groups.php
interface Groups
{
    public const COMPANY = 'company';
    public const NEW_GROUP = 'new_group';  // Add here

    public const GROUPS = [
        self::COMPANY => 'Company',
        self::NEW_GROUP => 'New Group',  // Add to array with label
    ];
}
```

### 3. Adding a New Settings Section

**Step 1**: Create controller
```php
// src/Http/Controllers/Backend/NewController.php
namespace Bongo\Setting\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Setting\Actions\ClearCache;
use Bongo\Setting\Actions\UpdateSettings;

class NewController extends AbstractController
{
    public function index(): View
    {
        return view('setting::backend.new.index');
    }

    public function update(Request $request): RedirectResponse
    {
        UpdateSettings::execute($request->get('settings'));
        ClearCache::execute();
        return redirect()->route('backend.setting.new.index')->success(...);
    }
}
```

**Step 2**: Add routes
```php
// src/Routes/backend.php
Route::as('new.')
    ->middleware('developer')
    ->prefix('new')
    ->group(function () {
        Route::get('/', [NewController::class, 'index'])->name('index');
        Route::post('update', [NewController::class, 'update'])->name('update');
    });
```

**Step 3**: Create view
```php
// src/Views/backend/new/index.blade.php
@extends('framework::backend.layouts.admin')
@section('content')
    <form method="POST" action="{{ route('backend.setting.new.update') }}">
        @csrf
        {{-- Setting form fields --}}
    </form>
@endsection
```

**Step 4**: Seed default settings
```php
// src/Seeders/DataSeeder.php
private function setNew(): void
{
    $rows = [
        ['name' => 'Setting Name', 'value' => 'default'],
    ];
    foreach ($rows as $row) {
        $this->firstOrCreateSetting(array_merge([
            'namespace' => Namespaces::NEW_NAMESPACE,
            'group' => Groups::NEW_GROUP,
        ], $row));
    }
}

public function run(): void
{
    // ... existing calls
    $this->setNew();  // Add call
}
```

### 4. Adding Event Listeners

```php
// src/Listeners/CustomListener.php
namespace Bongo\Setting\Listeners;

use Bongo\Setting\Events\SettingUpdated;

class CustomListener
{
    public function handle(SettingUpdated $event): void
    {
        $setting = $event->setting;
        $changes = $event->changes;

        // Custom logic here
    }
}

// Register in SettingServiceProvider
protected array $listeners = [
    SettingUpdated::class => [
        CustomListener::class,
    ],
];
```

### 5. Extending SettingManager

```php
// src/Services/SettingManager.php
public function getCustomBusinessLogic(): mixed
{
    // Access settings
    $value = $this->get('namespace::group.key');

    // Apply business rules
    if ($this->has('another::setting.key')) {
        // ...
    }

    return $result;
}
```

## Database Schema

### Settings Table

```sql
CREATE TABLE settings (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    namespace VARCHAR(255) NOT NULL,
    group VARCHAR(255) NOT NULL,
    name VARCHAR(255) NOT NULL,
    key VARCHAR(255) NOT NULL,
    value TEXT,
    rules TEXT NULL,
    options JSON NULL,
    type VARCHAR(255) NOT NULL DEFAULT 'input',
    data VARCHAR(255) NOT NULL DEFAULT 'string',
    sort_order INT NULL,
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    INDEX idx_namespace (namespace),
    INDEX idx_group (group),
    UNIQUE KEY unique_setting (namespace, group, key)
);
```

### Migrations History

1. **2020_01_01_000001**: Creates settings table with basic fields
2. **2021_01_01_000002**: Adds logo height settings (header_height, footer_height)
3. **2021_01_01_000003**: Adds mobile_height logo setting
4. **2021_01_01_000004**: Adds additional input types (color_picker, date_picker, dynamic) and data types

## Key Files Reference

| File Path | Purpose | Key Methods/Features |
|-----------|---------|---------------------|
| `src/SettingServiceProvider.php` | Bootstraps package | Registers singleton, event listeners, helpers, config overrides |
| `src/Models/Setting.php` | Setting model | `getFullKeyAttribute()`, `isEnabled()`, `getDynamicOptions()` |
| `src/Services/SettingManager.php` | Settings access service | `get()`, `has()`, `all()`, business logic helpers |
| `src/helpers.php` | Global helper | `setting($key, $default)` function |
| `src/Actions/UpdateSettings.php` | Update action | Saves settings, fires events |
| `src/Actions/ClearCache.php` | Cache clearing | Clears all Laravel caches |
| `src/Actions/CompileCss.php` | CSS compilation | SCSS to CSS with theme variables |
| `src/Seeders/DataSeeder.php` | Default data | Seeds all default settings |
| `src/Routes/backend.php` | Admin routes | All settings management endpoints |
| `src/Interfaces/` | Constants | Namespaces, Groups, InputTypes, DataTypes |

## Common Use Cases

### 1. Get a Setting Value
```php
$phone = setting('client::company.phone');
$primary = setting('theme::color.primary', '#BED62F');
```

### 2. Check if Setting Exists
```php
if (setting()->has('theme::layout.header')) {
    // Setting exists
}
```

### 3. Get All Settings by Namespace
```php
$themeSettings = setting()->getByNamespace('theme');
// Returns: ['color' => [...], 'logo' => [...], 'layout' => [...]]
```

### 4. Update Settings
```php
// In controller
UpdateSettings::execute($request->get('settings'));
ClearCache::execute();
```

### 5. Compile Theme CSS
```php
$compiler = new CompileCss();
$compiler->execute();  // Compiles to public/css/frontend.css
```

### 6. Manage Custom Files
```php
// Get/set custom CSS
$css = setting()->getCustomCss();
setting()->setCustomCss($newCss);

// Get/set custom JS (also writes to public/js/custom.js)
$js = setting()->getCustomJs();
setting()->setCustomJs($newJs);

// Get/set custom schema
$schema = setting()->getCustomSchema();
setting()->setCustomSchema($newSchema);
```

### 7. Business Logic Helpers
```php
// Phone number based on referrer
$phone = setting()->getClientPhoneNumber();  // Returns Google/Facebook/LinkedIn ads phone if applicable

// Addresses
$office = setting()->getOfficeAddress();
$registered = setting()->getRegisteredAddress();

// Theme helpers
$headerClass = setting()->getHeaderClass();  // Returns 'is-transparent is-sticky' etc.
if (setting()->hasTransparentHeader()) { }
if (setting()->hasStickyHeader()) { }

// ReCaptcha
if (setting()->reCaptchaEnabled()) {
    $siteKey = setting()->getReCaptchaSiteKey();
}
```
