<?php

namespace Bongo\Setting\Seeders;

use Bongo\Package\Models\Package;
use Bongo\Setting\Interfaces\DataTypes;
use Bongo\Setting\Interfaces\Groups;
use Bongo\Setting\Interfaces\InputTypes;
use Bongo\Setting\Interfaces\Namespaces;
use Bongo\Setting\Models\Setting;
use Illuminate\Database\Seeder;

class DataSeeder extends Seeder
{
    public function run(): void
    {
        $this->setCompany();
        $this->setSocial();
        $this->setMarketing();
        $this->setOfficeAddress();
        $this->setRegisteredAddress();
        $this->setTheme();
        $this->setCredentials();
        $this->setDeveloper();
        $this->setMisc();
        $this->setPackages();
        $this->setEstimateAreas();

        $this->command->info('Settings have been updated :)');
    }

    private function setCompany(): void
    {
        $rows = [
            ['name' => 'Name', 'sort_order' => 1],
            ['name' => 'Trading Name', 'sort_order' => 2],
            ['name' => 'Contact', 'sort_order' => 3],
            ['name' => 'Email', 'sort_order' => 4],
            ['name' => 'BCC Email', 'sort_order' => 5],
            ['name' => 'Phone', 'sort_order' => 6],
            ['name' => 'Mobile', 'sort_order' => 7],
            ['name' => 'Website', 'sort_order' => 8],
            ['name' => 'Company Number', 'sort_order' => 9],
            ['name' => 'Vat Number', 'sort_order' => 10],
            ['name' => 'Weekday Hours', 'sort_order' => 11],
            ['name' => 'Weekend Hours', 'sort_order' => 12],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::CLIENT,
                'group' => Groups::COMPANY,
            ], $row));
        }
    }

    private function setSocial(): void
    {
        $rows = [
            ['name' => 'Facebook', 'sort_order' => 1],
            ['name' => 'Twitter', 'sort_order' => 2],
            ['name' => 'Instagram', 'sort_order' => 3],
            ['name' => 'Youtube', 'sort_order' => 4],
            ['name' => 'Linkedin', 'sort_order' => 5],
            ['name' => 'Pinterest', 'sort_order' => 6],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::CLIENT,
                'group' => Groups::SOCIAL,
            ], $row));
        }
    }

    private function setMarketing(): void
    {
        $rows = [
            ['name' => 'Google Ads Phone', 'sort_order' => 1],
            ['name' => 'Facebook Ads Phone', 'sort_order' => 2],
            ['name' => 'Linkedin Ads Phone', 'sort_order' => 3],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::CLIENT,
                'group' => Groups::MARKETING,
            ], $row));
        }
    }

    private function setOfficeAddress(): void
    {
        $rows = [
            ['name' => 'Line 1', 'sort_order' => 1],
            ['name' => 'Line 2', 'sort_order' => 2],
            ['name' => 'Line 3', 'sort_order' => 3],
            ['name' => 'City', 'sort_order' => 4],
            ['name' => 'County', 'sort_order' => 5],
            ['name' => 'Postcode', 'sort_order' => 6],
            ['name' => 'Country', 'sort_order' => 7],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::CLIENT,
                'group' => Groups::OFFICE,
            ], $row));
        }
    }

    private function setRegisteredAddress(): void
    {
        $rows = [
            ['name' => 'Line 1', 'sort_order' => 1],
            ['name' => 'Line 2', 'sort_order' => 2],
            ['name' => 'Line 3', 'sort_order' => 3],
            ['name' => 'City', 'sort_order' => 4],
            ['name' => 'County', 'sort_order' => 5],
            ['name' => 'Postcode', 'sort_order' => 6],
            ['name' => 'Country', 'sort_order' => 7],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::CLIENT,
                'group' => Groups::REGISTERED,
            ], $row));
        }
    }

    private function setTheme(): void
    {
        // Logo settings
        $rows = [
            ['name' => 'Default', 'value' => '/images/logo.png', 'sort_order' => 1],
            ['name' => 'Light', 'value' => '/images/logo_light.png', 'sort_order' => 2],
            ['name' => 'Dark', 'value' => '/images/logo_dark.png', 'sort_order' => 3],
            ['name' => 'Header Height', 'value' => 48, 'sort_order' => 4],
            ['name' => 'Mobile Height', 'value' => 30, 'sort_order' => 5],
            ['name' => 'Sticky Height', 'value' => 33, 'sort_order' => 6],
            ['name' => 'Footer Height', 'value' => 33, 'sort_order' => 7],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::THEME,
                'group' => Groups::LOGO,
            ], $row));
        }

        // Color settings
        $rows = [
            ['name' => 'Primary', 'value' => '#BED62F', 'sort_order' => 1],
            ['name' => 'Secondary', 'value' => '#F8931D', 'sort_order' => 2],
            ['name' => 'Tertiary', 'value' => '#1E79BF', 'sort_order' => 3],
            ['name' => 'Quaternary', 'value' => '#D62F6B', 'sort_order' => 4],
            ['name' => 'Color 5', 'value' => '#DC2121', 'sort_order' => 5],
            ['name' => 'Color 6', 'value' => '#9C0F83', 'sort_order' => 6],
            ['name' => 'Color 7', 'value' => '#9C510F', 'sort_order' => 7],
            ['name' => 'Color 8', 'value' => '#F3036B', 'sort_order' => 8],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::THEME,
                'group' => Groups::COLOR,
                'type' => InputTypes::COLOR_PICKER,
            ], $row));
        }

        // Layout settings
        $rows = [
            [
                'name' => 'Skin',
                'value' => 'default',
                'sort_order' => 9,
                'type' => InputTypes::SELECT,
                'options' => [
                    'cerulean',
                    'cosmo',
                    'custom',
                    'cyborg',
                    'darkly',
                    'default',
                    'flatly',
                    'journal',
                    'litera',
                    'lumen',
                    'lux',
                    'materia',
                    'minty',
                    'pulse',
                    'sandstone',
                    'simplex',
                    'sketchy',
                    'slate',
                    'solar',
                    'spacelab',
                    'superhero',
                    'united',
                    'yeti',
                ],
            ],
            [
                'name' => 'Header',
                'value' => 'header-01',
                'sort_order' => 10,
                'type' => InputTypes::DYNAMIC,
            ],
            [
                'name' => 'Sticky Header',
                'value' => Setting::DISABLED,
                'sort_order' => 11,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            [
                'name' => 'Transparent Header',
                'value' => Setting::DISABLED,
                'sort_order' => 12,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            [
                'name' => 'Heading',
                'value' => 'heading-01',
                'sort_order' => 13,
                'type' => InputTypes::DYNAMIC,
            ],
            [
                'name' => 'Footer',
                'value' => 'footer-01',
                'sort_order' => 14,
                'type' => InputTypes::DYNAMIC,
            ],
            [
                'name' => 'Footer Text',
                'sort_order' => 15,
                'type' => InputTypes::TEXT,
            ],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::THEME,
                'group' => Groups::LAYOUT,
            ], $row));
        }
    }

    private function setCredentials(): void
    {
        $rows = [
            ['name' => 'Facebook App Id'],
            ['name' => 'Facebook Pixel Id'],
            ['name' => 'Google Analytics Id'],
            [
                'name' => 'Google Analytics Type',
                'value' => 'analytics',
                'type' => InputTypes::SELECT,
                'options' => ['analytics', 'tag_manager'],
            ],
            ['name' => 'Google Geocoding Api Key'],
            ['name' => 'Google Maps Api Key'],
            ['name' => 'Google ReCaptcha Site Key'],
            ['name' => 'Google ReCaptcha Secret Key'],
            ['name' => 'Mailchimp Url'],
            ['name' => 'OpenAI Api Key'],
            ['name' => 'Real Green Api Key'],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::SYSTEM,
                'group' => Groups::CREDENTIALS,
            ], $row));
        }
    }

    private function setMisc(): void
    {
        $rows = [
            [
                'name' => 'Version',
                'value' => '2.0.0',
                'data' => DataTypes::DECIMAL,
            ],
            [
                'name' => 'Site Visibility',
                'value' => 'noindex',
                'sort_order' => 6,
                'type' => InputTypes::SELECT,
                'options' => ['index', 'noindex'],
            ],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::SYSTEM,
                'group' => Groups::MISC,
            ], $row));
        }
    }

    private function setDeveloper(): void
    {
        $rows = [
            [
                'name' => 'Credit Link',
                'value' => 'https://www.designtec.co.uk/website-design-norwich',
                'type' => InputTypes::INPUT,
                'data' => DataTypes::STRING,
            ],
            [
                'name' => 'Credit Text',
                'value' => 'Website Design Norwich',
                'type' => InputTypes::INPUT,
                'data' => DataTypes::STRING,
            ],
            [
                'name' => 'Credit Type',
                'value' => 'follow',
                'type' => InputTypes::SELECT,
                'options' => ['follow', 'nofollow'],
            ],
        ];

        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::SYSTEM,
                'group' => Groups::DEVELOPER,
            ], $row));
        }
    }

    private function setPackages(): void
    {
        // Estimate settings
        $rows = [
            [
                'name' => 'Estimate Module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            [
                'name' => 'Real Green Module',
                'key' => 'real_green_module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            ['name' => 'Step 1', 'value' => 'Details', 'sort_order' => 2],
            ['name' => 'Step 2', 'value' => 'Measure', 'sort_order' => 3],
            ['name' => 'Step 3', 'value' => 'Quote', 'sort_order' => 4],
            ['name' => 'More Assistance Url', 'value' => '/more-assistance', 'sort_order' => 5],
            ['name' => 'Custom Estimate Url', 'value' => '/custom-estimate', 'sort_order' => 6],
            ['name' => 'Terms Url', 'value' => '/terms-conditions', 'sort_order' => 7],
            ['name' => 'Confirmation Url', 'value' => '/thank-you', 'sort_order' => 8],
            ['name' => 'Sales Email', 'value' => 'sales@designtec.co.uk', 'sort_order' => 9],
            ['name' => 'Postcode Placeholder', 'value' => 'e.g B78 0AB', 'sort_order' => 10],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::ESTIMATE,
            ], $row));
        }

        // Gallery settings
        $rows = [
            [
                'name' => 'Gallery Module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::GALLERY,
            ], $row));
        }

        // OpenAI settings
        $rows = [
            [
                'name' => 'OpenAI Module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::OPENAI,
            ], $row));
        }

        // Post settings
        $rows = [
            [
                'name' => 'Post Module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            [
                'name' => 'Related Type',
                'value' => 'off',
                'sort_order' => 2,
                'type' => InputTypes::SELECT,
                'options' => ['off', 'random', 'next_previous', 'random_next_previous'],
            ],
            [
                'name' => 'Meta Title',
                'sort_order' => 3,
            ],
            [
                'name' => 'Meta Description',
                'sort_order' => 4,
                'type' => InputTypes::TEXT,
            ],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::POST,
            ], $row));
        }

        // Project settings
        $rows = [
            [
                'name' => 'Project Module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            [
                'name' => 'Related Type',
                'value' => 'off',
                'sort_order' => 2,
                'type' => InputTypes::SELECT,
                'options' => ['off', 'random', 'next_previous', 'random_next_previous'],
            ],
            [
                'name' => 'Meta Title',
                'sort_order' => 3,
            ],
            [
                'name' => 'Meta Description',
                'sort_order' => 4,
                'type' => InputTypes::TEXT,
            ],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::PROJECT,
            ], $row));
        }

        // Review settings
        $rows = [
            [
                'name' => 'Review Module',
                'value' => Package::INACTIVE,
                'type' => InputTypes::CHECKBOX,
                'data' => DataTypes::BOOLEAN,
            ],
            [
                'name' => 'Meta Title',
                'sort_order' => 3,
            ],
            [
                'name' => 'Meta Description',
                'sort_order' => 4,
                'type' => InputTypes::TEXT,
            ],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::REVIEW,
            ], $row));
        }

        // Page settings
        $rows = [
            ['name' => 'Number Of Pages', 'value' => 9, 'data' => DataTypes::INTEGER],
        ];
        foreach ($rows as $row) {
            $this->firstOrCreateSetting(array_merge([
                'namespace' => Namespaces::PACKAGE,
                'group' => Groups::PAGE,
            ], $row));
        }
    }

    private function setEstimateAreas(): void
    {
        // Remove the old areas
        Setting::query()
            ->where('namespace', Namespaces::PACKAGE)
            ->where('group', Groups::ESTIMATE)
            ->whereIn('name', [
                'Radius In Miles',
                'From Latitude',
                'From Longitude',
            ])
            ->delete();

        // Create the areas
        $numberOfAreas = config('estimate.number_of_areas', 5);
        for ($i = 1; $i <= $numberOfAreas; $i++) {
            $rows = [
                ['name' => 'Name', 'sort_order' => 1],
                ['name' => 'Radius In Miles', 'sort_order' => 2],
                ['name' => 'From Latitude', 'sort_order' => 3],
                ['name' => 'From Longitude', 'sort_order' => 4],
                ['name' => 'Fill Color', 'sort_order' => 5, 'type' => InputTypes::COLOR_PICKER],
                ['name' => 'Stroke Color', 'sort_order' => 6, 'type' => InputTypes::COLOR_PICKER],
            ];
            foreach ($rows as $row) {
                $this->firstOrCreateSetting(array_merge([
                    'namespace' => Namespaces::ESTIMATE_AREA,
                    'group' => "area_$i",
                ], $row));
            }
        }
    }

    private function firstOrCreateSetting(array $row): void
    {
        // Check if we already have this setting
        $setting = Setting::query()
            ->where('namespace', $row['namespace'])
            ->where('group', $row['group'])
            ->where('name', $row['name'])
            ->first();

        // If we don't then create it
        if (! $setting) {
            Setting::create([
                'namespace' => $row['namespace'],
                'group' => $row['group'],
                'name' => $row['name'],
                'value' => $row['value'] ?? null,
                'rules' => $row['rules'] ?? null,
                'options' => $row['options'] ?? null,
                'type' => $row['type'] ?? InputTypes::INPUT,
                'data' => $row['data'] ?? DataTypes::STRING,
                'sort_order' => $row['sort_order'] ?? null,
            ]);
        }
    }
}
