# GitHub Copilot Instructions - Setting Package

## Package Overview

**bongo/setting** provides database-backed system settings management for Laravel applications. Settings are organised hierarchically (`namespace::group.key`) and accessed via a singleton SettingManager service or the `setting()` helper function. Changes trigger events for module status updates and cache clearing.

**Core Dependencies**: bongo/framework (^3.0), scssphp/scssphp (^1.11)

## Key Classes and Relationships

### Service Provider
```php
// src/SettingServiceProvider.php
class SettingServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'setting';
    protected array $listeners = [
        SettingUpdated::class => [
            UpdateEstimateModuleStatus::class,
            UpdateRealGreenModuleStatus::class,
            // ... more listeners
        ],
    ];

    public function register(): void
    {
        $this->app->singleton('setting_manager', SettingManager::class);
    }
}
```

### Setting Model
```php
// src/Models/Setting.php
class Setting extends AbstractModel implements DataTypes, Groups, InputTypes, Namespaces
{
    // Hierarchical key structure
    getFullKeyAttribute(): string  // Returns "namespace::group.key"

    // Type checking (via HasType trait)
    isCheckbox(): bool
    isColorPicker(): bool
    isDynamic(): bool
    isInput(): bool
    isSelect(): bool
    isText(): bool

    // State helpers
    isEnabled(): bool   // For checkbox settings
    isDisabled(): bool  // For checkbox settings
}
```

### SettingManager Service
```php
// src/Services/SettingManager.php
class SettingManager
{
    // Cached collection of all settings
    protected Collection $items;  // Cache::rememberForever("settings", ...)

    // Core access methods
    public function has(string $key): bool
    public function get(string $key, $default = null): mixed
    public function all(): Collection
    public function allByNamespace(): array
    public function getByNamespace(string $namespace): array

    // File management
    public function getCustomCss(): ?string
    public function setCustomCss(?string $css): void
    public function getCustomJs(): ?string
    public function setCustomJs(?string $js): void  // Also writes public/js/custom.js

    // Business logic helpers
    public function getClientPhoneNumber(): ?string  // Returns phone based on referrer
    public function getEstimateAreas(): array
    public function getReCaptchaSiteKey(): ?string
    public function reCaptchaEnabled(): bool
}
```

### Actions
```php
// src/Actions/UpdateSettings.php
class UpdateSettings
{
    public static function execute(array $settings): void
    {
        foreach ($settings as $id => $value) {
            if ($setting = Setting::find($id)) {
                if ($setting->value != $value) {
                    $setting->value = $value;
                    $changes = $setting->getDiff();
                    $setting->save();
                    event(new SettingUpdated($setting, $changes));
                }
            }
        }
    }
}

// src/Actions/ClearCache.php
class ClearCache
{
    public static function execute(): void
    {
        Artisan::call('event:clear');
        Artisan::call('view:clear');
        Artisan::call('cache:clear');
        // ... more cache clearing
    }
}

// src/Actions/CompileCss.php
class CompileCss
{
    public function execute(): void
    {
        // Extracts theme variables from settings
        $variables = [
            'primary' => setting('theme::color.primary'),
            'secondary' => setting('theme::color.secondary'),
            'skin' => setting('theme::layout.skin'),
            // ...
        ];

        // Compiles SCSS with scssphp
        $scss = new Compiler();
        $scss->addVariables($variables);
        $scss->setImportPaths([resource_path('sass')]);
        $output = $scss->compile("@import 'skins/...'; @import 'bootstrap'; ...");
        file_put_contents(public_path('css/frontend.css'), $output);
    }
}
```

## Architecture Diagram

```
┌─────────────────────────────────────────────────────────────┐
│                    Setting Package Flow                      │
└─────────────────────────────────────────────────────────────┘

┌──────────────┐       ┌─────────────────┐       ┌────────────┐
│   Request    │──────▶│  Controller     │──────▶│  Action    │
│ (POST form)  │       │  (Backend)      │       │ UpdateSet  │
└──────────────┘       └─────────────────┘       │  tings     │
                                                  └─────┬──────┘
                                                        │
                                                        ▼
┌──────────────┐       ┌─────────────────┐       ┌────────────┐
│  Listener    │◀──────│  SettingUpdated │◀──────│  Setting   │
│ UpdateModule │       │     Event       │       │   Model    │
│   Status     │       └─────────────────┘       └────────────┘
└──────────────┘

┌──────────────┐       ┌─────────────────┐
│   Helper     │──────▶│ SettingManager  │
│ setting()    │       │   (Singleton)   │
└──────────────┘       └────────┬────────┘
                                │
                                ▼
                        ┌───────────────┐
                        │  Cache Layer  │
                        │  (Forever)    │
                        └───────────────┘
```

## Setting Key Structure

```
namespace::group.key
    │       │     │
    │       │     └─── Specific setting identifier
    │       └───────── Logical grouping within namespace
    └───────────────── Top-level category

Examples:
- theme::color.primary
- client::company.phone
- system::credentials.openai_api_key
- package::estimate.estimate_module
```

## Code Style Templates

### Controller Pattern
```php
namespace Bongo\Setting\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Setting\Actions\ClearCache;
use Bongo\Setting\Actions\UpdateSettings;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ExampleController extends AbstractController
{
    public function index(): View
    {
        return view('setting::backend.example.index');
    }

    public function update(Request $request): RedirectResponse
    {
        UpdateSettings::execute($request->get('settings'));
        ClearCache::execute();

        return redirect()
            ->route('backend.setting.example.index')
            ->success(trans('setting::backend.update_success'));
    }
}
```

### Event Listener Pattern
```php
namespace Bongo\Setting\Listeners;

use Bongo\Package\Models\Package;
use Bongo\Setting\Events\SettingUpdated;

class UpdateModuleStatus
{
    public function handle(SettingUpdated $event): void
    {
        $setting = $event->setting;

        if ($setting->full_key === 'package::module.module_name') {
            Package::where('key', 'module')->update([
                'is_active' => $setting->isEnabled(),
            ]);
        }
    }
}
```

### Seeder Pattern
```php
private function setExample(): void
{
    $rows = [
        ['name' => 'Setting Name', 'value' => 'default', 'sort_order' => 1],
        [
            'name' => 'Select Setting',
            'value' => 'option1',
            'sort_order' => 2,
            'type' => InputTypes::SELECT,
            'options' => ['option1', 'option2', 'option3'],
        ],
        [
            'name' => 'Enabled Setting',
            'value' => Setting::DISABLED,
            'sort_order' => 3,
            'type' => InputTypes::CHECKBOX,
            'data' => DataTypes::BOOLEAN,
        ],
    ];

    foreach ($rows as $row) {
        $this->firstOrCreateSetting(array_merge([
            'namespace' => Namespaces::SYSTEM,
            'group' => Groups::MISC,
        ], $row));
    }
}
```

### Route Definition Pattern
```php
// src/Routes/backend.php
Route::as('example.')
    ->middleware('developer')
    ->prefix('example')
    ->group(function () {
        Route::get('/', [ExampleController::class, 'index'])
            ->name('index');

        Route::post('update', [ExampleController::class, 'update'])
            ->name('update');
    });
```

### Accessing Settings Pattern
```php
// Get single value
$phone = setting('client::company.phone');
$primary = setting('theme::color.primary', '#BED62F');

// Get manager instance
$manager = setting();
$enabled = $manager->reCaptchaEnabled();

// Check existence
if (setting()->has('theme::layout.header')) {
    // ...
}

// Get by namespace
$themeSettings = setting()->getByNamespace('theme');
foreach ($themeSettings as $group => $settings) {
    foreach ($settings as $key => $setting) {
        echo "{$setting->full_key}: {$setting->value}";
    }
}
```

## Common Patterns

### Input Types and Data Types
```php
// Input types (how the setting is displayed)
InputTypes::CHECKBOX        // Boolean checkbox
InputTypes::COLOR_PICKER    // Colour picker
InputTypes::DATE_PICKER     // Date picker
InputTypes::DYNAMIC         // Dynamic select (populated from database)
InputTypes::INPUT           // Text input
InputTypes::SELECT          // Static select dropdown
InputTypes::TEXT            // Textarea

// Data types (how the value is stored)
DataTypes::BOOLEAN          // 0 or 1
DataTypes::DECIMAL          // Decimal number
DataTypes::INTEGER          // Integer number
DataTypes::JSON             // JSON string
DataTypes::STRING           // Text (default)
```

### Namespaces and Groups
```php
// Namespaces
Namespaces::CLIENT          // Client/company information
Namespaces::PACKAGE         // Package module toggles
Namespaces::ESTIMATE_AREA   // Estimate area configuration
Namespaces::THEME           // Theme and design settings
Namespaces::SYSTEM          // System configuration

// Common Groups
Groups::COMPANY             // Company details
Groups::SOCIAL              // Social media links
Groups::MARKETING           // Marketing phone numbers
Groups::OFFICE              // Office address
Groups::REGISTERED          // Registered address
Groups::LAYOUT              // Layout settings
Groups::COLOR               // Colour scheme
Groups::LOGO                // Logo settings
Groups::CREDENTIALS         // API keys and credentials
Groups::DEVELOPER           // Developer credit
```

## Key Interfaces

```php
// src/Interfaces/Namespaces.php
interface Namespaces
{
    public const CLIENT = 'client';
    public const PACKAGE = 'package';
    public const ESTIMATE_AREA = 'estimate';
    public const THEME = 'theme';
    public const SYSTEM = 'system';
}

// src/Interfaces/Groups.php
interface Groups
{
    public const COMPANY = 'company';
    public const SOCIAL = 'social';
    // ... 20+ more constants
}

// src/Interfaces/InputTypes.php
interface InputTypes
{
    public const CHECKBOX = 'checkbox';
    public const COLOR_PICKER = 'color_picker';
    // ... more input types
}

// src/Interfaces/DataTypes.php
interface DataTypes
{
    public const BOOLEAN = 'boolean';
    public const STRING = 'string';
    // ... more data types
}
```

## Testing Patterns

```php
use Bongo\Setting\Models\Setting;
use Orchestra\Testbench\TestCase;

class SettingTest extends TestCase
{
    /** @test */
    public function it_can_get_setting_value()
    {
        Setting::factory()->create([
            'namespace' => 'theme',
            'group' => 'color',
            'key' => 'primary',
            'value' => '#BED62F',
        ]);

        $this->assertEquals('#BED62F', setting('theme::color.primary'));
    }
}
```

## Important Notes

1. **Settings are cached forever** - Always call `ClearCache::execute()` after updates
2. **Events fire on changes** - Listeners update related modules automatically
3. **Hierarchical keys** - Always use `namespace::group.key` format
4. **Dynamic options** - Some selects fetch options from database (headers, footers via LayoutFacade)
5. **ReCaptcha override** - Settings override default captcha config on boot
6. **Custom files** - CSS/JS/Schema methods write to resource and public paths
