# CLAUDE.md - Setting Package

This file provides guidance to Claude Code (claude.ai/code) when working with the **bongo/setting** package.

## Overview

The **bongo/setting** package provides comprehensive system settings management for Laravel applications. Settings are stored in a database with hierarchical keys (`namespace::group.key`), cached for performance, and accessed through a singleton service or the `setting()` helper function.

**Key Features**:
- Database-backed settings with hierarchical organisation
- Event-driven architecture for module status updates
- SCSS compilation with theme variables
- Custom CSS/JS/Schema file management
- Cached settings for performance (forever cache)
- Role-based access (developer/manager middleware)

**Related Documentation**:
- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture, class diagrams, and flow diagrams
- [.cursorrules](.cursorrules) - Development guidelines and patterns
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - Code style templates

## Commands

From `composer.json`:

```bash
# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Static analysis (if configured)
vendor/bin/phpstan analyse
```

## Quick Architecture Reference

### Extends Bongo Framework

This package extends `Bongo\Framework\Providers\AbstractServiceProvider` which provides:
- **Automatic bootstrapping** via `boot()` method
- **Config registration** from `src/Config/{module}.php`
- **Route registration** from `src/Routes/`:
  - `backend.php` → `backend.*` prefix with `auth` + `employee` middleware
- **View registration** from `src/Views/{module}/`
- **Migration registration** from `src/Migrations/`
- **Translation registration** from `src/Translations/`
- **Event listener registration** via `$listeners` array

The `$module = 'setting'` property enables all automatic registrations.

### Core Components

```
SettingServiceProvider (extends AbstractServiceProvider)
    ├─ Registers: SettingManager singleton as 'setting_manager'
    ├─ Registers: Setting model alias
    ├─ Loads: helpers.php (setting() function)
    ├─ Overrides: ReCaptcha config from settings
    └─ Listeners: 10 listeners for SettingUpdated, CssUpdated, JsUpdated, SchemaUpdated events

Setting Model (extends AbstractModel)
    ├─ Uses: HasKey trait (from bongo/framework)
    ├─ Uses: HasType trait (type checking methods)
    ├─ Implements: DataTypes, Groups, InputTypes, Namespaces interfaces
    └─ Key Method: getFullKeyAttribute() → "namespace::group.key"

SettingManager Service (Singleton)
    ├─ Cached Collection: All settings (Cache::rememberForever)
    ├─ Access Methods: has(), get(), all(), allByNamespace()
    ├─ File Methods: getCustomCss(), setCustomJs(), setCustomSchema()
    └─ Business Logic: getClientPhoneNumber(), reCaptchaEnabled(), etc.

Actions
    ├─ UpdateSettings::execute(array) → Updates settings, fires events
    ├─ ClearCache::execute() → Clears all Laravel caches
    └─ CompileCss->execute() → Compiles SCSS with theme variables

Events & Listeners
    ├─ SettingUpdated → 8 module status update listeners + ClearMenuCache
    ├─ CssUpdated → UpdateFrontendCss
    ├─ JsUpdated → UpdateFrontendJs
    └─ SchemaUpdated → UpdateCustomSchema
```

### Hierarchical Key System

```
Format: namespace::group.key

Examples:
- theme::color.primary
- client::company.phone
- system::credentials.openai_api_key
- package::estimate.estimate_module

Namespaces:
  client     → Client/company information
  package    → Package module toggles
  estimate   → Estimate area configurations
  theme      → Theme and design settings
  system     → System configuration

Groups (25+):
  company, social, marketing, office, registered,
  logo, color, layout, credentials, developer, misc,
  estimate, gallery, openai, post, project, review, page,
  area_1, area_2, area_3, area_4, area_5
```

### Helper Function

```php
setting()                           // Returns SettingManager instance
setting('key')                      // Returns setting value
setting('key', 'default')           // Returns value or default

// Examples
$phone = setting('client::company.phone');
$primary = setting('theme::color.primary', '#BED62F');
$manager = setting();
$enabled = $manager->reCaptchaEnabled();
```

## Key Files

| File | Purpose | Key Methods |
|------|---------|-------------|
| `src/SettingServiceProvider.php` | Bootstraps package | `register()`, `boot()`, `$listeners` array |
| `src/Models/Setting.php` | Setting model | `getFullKeyAttribute()`, `isEnabled()`, `getDynamicOptions()` |
| `src/Services/SettingManager.php` | Singleton service | `get()`, `has()`, `getClientPhoneNumber()`, `reCaptchaEnabled()` |
| `src/helpers.php` | Global helper | `setting($key, $default)` |
| `src/Actions/UpdateSettings.php` | Update action | `execute(array $settings)` |
| `src/Actions/ClearCache.php` | Cache clearing | `execute()` |
| `src/Actions/CompileCss.php` | CSS compilation | `execute()` |
| `src/Seeders/DataSeeder.php` | Default settings | `run()`, `firstOrCreateSetting()` |
| `src/Routes/backend.php` | Admin routes | All settings endpoints |
| `src/Interfaces/Namespaces.php` | Namespace constants | CLIENT, PACKAGE, ESTIMATE_AREA, THEME, SYSTEM |
| `src/Interfaces/Groups.php` | Group constants | COMPANY, SOCIAL, LAYOUT, COLOR, etc. |
| `src/Interfaces/InputTypes.php` | Input type constants | CHECKBOX, SELECT, INPUT, TEXT, etc. |
| `src/Interfaces/DataTypes.php` | Data type constants | BOOLEAN, STRING, INTEGER, DECIMAL, JSON |
| `src/Traits/HasType.php` | Type checking | `isCheckbox()`, `isSelect()`, etc. |

## Common Tasks

### Access a Setting

```php
// Get value with helper
$phone = setting('client::company.phone');
$primary = setting('theme::color.primary', '#default');

// Check existence
if (setting()->has('theme::layout.header')) {
    // ...
}

// Get by namespace
$themeSettings = setting()->getByNamespace('theme');
```

### Update Settings (in Controller)

```php
use Bongo\Setting\Actions\ClearCache;
use Bongo\Setting\Actions\UpdateSettings;

public function update(Request $request): RedirectResponse
{
    UpdateSettings::execute($request->get('settings'));
    ClearCache::execute();

    return redirect()
        ->route('backend.setting.client.index')
        ->success(trans('setting::backend.update_success'));
}
```

### Add a New Setting

1. **Update Seeder** (`src/Seeders/DataSeeder.php`):
```php
private function setNewSettings(): void
{
    $rows = [
        ['name' => 'New Setting', 'value' => 'default', 'sort_order' => 1],
    ];
    foreach ($rows as $row) {
        $this->firstOrCreateSetting(array_merge([
            'namespace' => Namespaces::CLIENT,
            'group' => Groups::COMPANY,
            'type' => InputTypes::INPUT,
            'data' => DataTypes::STRING,
        ], $row));
    }
}

public function run(): void
{
    // ... existing calls
    $this->setNewSettings();
}
```

2. **Run Seeder**: `php artisan db:seed --class=Bongo\\Setting\\Seeders\\DataSeeder`

3. **Add Getter (Optional)** to `SettingManager.php`:
```php
public function getNewSetting(): ?string
{
    return $this->get('client::company.new_setting');
}
```

### Add a Settings Section

1. **Create Controller** (`src/Http/Controllers/Backend/NewController.php`):
```php
namespace Bongo\Setting\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;
use Bongo\Setting\Actions\ClearCache;
use Bongo\Setting\Actions\UpdateSettings;

class NewController extends AbstractController
{
    public function index(): View
    {
        return view('setting::backend.new.index');
    }

    public function update(Request $request): RedirectResponse
    {
        UpdateSettings::execute($request->get('settings'));
        ClearCache::execute();
        return redirect()->route('backend.setting.new.index')->success(...);
    }
}
```

2. **Add Routes** (`src/Routes/backend.php`):
```php
Route::as('new.')
    ->middleware('developer')
    ->prefix('new')
    ->group(function () {
        Route::get('/', [NewController::class, 'index'])->name('index');
        Route::post('update', [NewController::class, 'update'])->name('update');
    });
```

3. **Create View** (`src/Views/backend/new/index.blade.php`)

### Add Event Listener

1. **Create Listener** (`src/Listeners/CustomListener.php`):
```php
namespace Bongo\Setting\Listeners;

use Bongo\Setting\Events\SettingUpdated;

class CustomListener
{
    public function handle(SettingUpdated $event): void
    {
        $setting = $event->setting;

        if ($setting->full_key === 'package::module.setting_key') {
            // Custom logic
        }
    }
}
```

2. **Register in Service Provider**:
```php
protected array $listeners = [
    SettingUpdated::class => [
        CustomListener::class,
    ],
];
```

### Compile CSS with Theme Variables

```php
use Bongo\Setting\Actions\CompileCss;

$compiler = new CompileCss();
$compiler->execute();
// Outputs to: public/css/frontend.css
```

### Manage Custom Files

```php
// Custom CSS (resource/sass/custom.scss)
$css = setting()->getCustomCss();
setting()->setCustomCss($newCss);

// Custom JS (resource/js/custom.js + public/js/custom.js)
$js = setting()->getCustomJs();
setting()->setCustomJs($newJs);

// Custom Schema (resource/schema/custom.json)
$schema = setting()->getCustomSchema();
setting()->setCustomSchema($newSchema);
```

## Code Style Summary

### Controllers Pattern
```php
class ExampleController extends AbstractController
{
    public function index(): View
    {
        return view('setting::backend.example.index');
    }

    public function update(Request $request): RedirectResponse
    {
        UpdateSettings::execute($request->get('settings'));
        ClearCache::execute();
        return redirect()->route('backend.setting.example.index')->success(...);
    }
}
```

### Model Usage
```php
$setting = Setting::where('namespace', 'theme')
    ->where('group', 'color')
    ->where('key', 'primary')
    ->first();

$fullKey = $setting->full_key;  // "theme::color.primary"
$enabled = $setting->isEnabled();  // For checkbox settings
$isCheckbox = $setting->isCheckbox();  // Type check
```

### Route Definitions
```php
Route::as('example.')
    ->middleware('developer')  // or 'manager'
    ->prefix('example')
    ->group(function () {
        Route::get('/', [ExampleController::class, 'index'])->name('index');
        Route::post('update', [ExampleController::class, 'update'])->name('update');
    });
```

### View Namespacing
```php
return view('setting::backend.client.index');
return view('setting::backend.partials.form');
```

## Important Notes

1. **Settings are cached forever** - Always call `ClearCache::execute()` after updates
2. **Event-driven** - SettingUpdated events automatically update module statuses
3. **Hierarchical keys** - Always use `namespace::group.key` format
4. **Dynamic options** - Some selects fetch options from database (headers, footers via LayoutFacade)
5. **ReCaptcha override** - Settings override captcha config on application boot
6. **SCSS compilation** - Theme colours and variables injected into SCSS compilation
7. **Multiple input types** - CHECKBOX, SELECT, INPUT, TEXT, COLOR_PICKER, DATE_PICKER, DYNAMIC
8. **Role-based access** - Developer and manager middleware protect different sections

## Dependencies

- **bongo/framework** (^3.0): AbstractServiceProvider, AbstractModel, AbstractController
- **scssphp/scssphp** (^1.11): SCSS to CSS compilation
- **bongo/menu**: ClearMenuCache listener
- **bongo/package**: Package model for module toggles
- **bongo/referrer**: Referrer service for marketing phone numbers
- **bongo/builder**: LayoutFacade for dynamic header/footer options

## Testing

```bash
# Run all tests
vendor/bin/phpunit

# Run specific test
vendor/bin/phpunit --filter SettingTest

# Code coverage (if configured)
vendor/bin/phpunit --coverage-html coverage
```

## Code Quality

```bash
# Check code style issues
vendor/bin/pint --test

# Fix code style issues
vendor/bin/pint

# Static analysis (if configured)
vendor/bin/phpstan analyse
```

## Extension Points

1. **Add new namespaces**: Update `src/Interfaces/Namespaces.php`
2. **Add new groups**: Update `src/Interfaces/Groups.php`
3. **Add new input types**: Update `src/Interfaces/InputTypes.php` and `src/Traits/HasType.php`
4. **Listen to changes**: Add listeners to `SettingServiceProvider::$listeners`
5. **Business logic**: Extend `SettingManager` with new methods
6. **New settings pages**: Create controller, add routes, create views
7. **Custom actions**: Create new action classes following UpdateSettings pattern
