# Setting Package

[![Latest Version](https://img.shields.io/badge/version-3.0-blue.svg)](https://github.com/designtec/setting)
[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-brightgreen.svg)](https://php.net)
[![Laravel Version](https://img.shields.io/badge/laravel-10%2B-orange.svg)](https://laravel.com)

Comprehensive database-backed settings management for Laravel applications with hierarchical organisation, event-driven updates, and theme variable compilation.

## Features

- **Hierarchical Settings** - Organised by namespace, group, and key (`namespace::group.key`)
- **Database Storage** - Persistent settings with automatic caching
- **Event-Driven Architecture** - Automatic module status updates on setting changes
- **Theme Management** - SCSS compilation with dynamic colour and layout variables
- **Custom File Management** - Manage custom CSS, JavaScript, and JSON schema files
- **Role-Based Access** - Developer and manager middleware protection
- **Multiple Input Types** - Support for checkbox, select, colour picker, date picker, and more
- **Global Helper** - Easy access via `setting()` helper function
- **Forever Caching** - Settings cached for optimal performance

## Requirements

- PHP 8.2+
- Laravel 10+
- Composer
- bongo/framework ^3.0
- scssphp/scssphp ^1.11

## Installation

### Via Composer

```bash
composer require bongo/setting
```

### Laravel Auto-Discovery

Laravel will automatically discover and register the `SettingServiceProvider`.

For older Laravel versions, manually add to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Setting\SettingServiceProvider::class,
],
```

### Publish and Run Migrations

```bash
php artisan migrate
```

### Seed Default Settings

```bash
php artisan db:seed --class=Bongo\\Setting\\Seeders\\DataSeeder
```

## Usage

### Basic Usage

```php
// Get a setting value
$phone = setting('client::company.phone');

// Get with default value
$primary = setting('theme::color.primary', '#BED62F');

// Check if setting exists
if (setting()->has('theme::layout.header')) {
    // Setting exists
}

// Get SettingManager instance
$manager = setting();
```

### Access Settings

```php
// Get all settings
$all = setting()->all();

// Get settings by namespace
$themeSettings = setting()->getByNamespace('theme');
// Returns: ['color' => [...], 'logo' => [...], 'layout' => [...]]

// Get all settings organised by namespace
$organised = setting()->allByNamespace();
```

### Business Logic Helpers

```php
// Get client phone (returns marketing phone based on referrer)
$phone = setting()->getClientPhoneNumber();

// Get formatted addresses
$office = setting()->getOfficeAddress();
$registered = setting()->getRegisteredAddress();

// Theme helpers
if (setting()->hasTransparentHeader()) {
    // Header is transparent
}

$headerClass = setting()->getHeaderClass();
// Returns: 'is-transparent is-sticky' etc.

// ReCaptcha
if (setting()->reCaptchaEnabled()) {
    $siteKey = setting()->getReCaptchaSiteKey();
    $secretKey = setting()->getReCaptchaSecretKey();
}

// OpenAI
$apiKey = setting()->getOpenAIApiKey();

// Estimate areas
$areas = setting()->getEstimateAreas();
```

### Update Settings

```php
use Bongo\Setting\Actions\UpdateSettings;
use Bongo\Setting\Actions\ClearCache;

// Update settings (typically in a controller)
UpdateSettings::execute($request->get('settings'));
ClearCache::execute();
```

### Manage Custom Files

```php
// Custom CSS (resource/sass/custom.scss)
$css = setting()->getCustomCss();
setting()->setCustomCss($newCss);

// Custom JS (resource/js/custom.js + public/js/custom.js)
$js = setting()->getCustomJs();
setting()->setCustomJs($newJs);

// Custom Schema (resource/schema/custom.json)
$schema = setting()->getCustomSchema();
setting()->setCustomSchema($newSchema);
```

### Compile Theme CSS

```php
use Bongo\Setting\Actions\CompileCss;

$compiler = new CompileCss();
$compiler->execute();
// Compiles SCSS with theme variables to public/css/frontend.css
```

### Clear Caches

```php
use Bongo\Setting\Actions\ClearCache;

ClearCache::execute();
// Clears: event, view, cache, route, config, compiled
```

## Hierarchical Key Structure

Settings are organised hierarchically:

```
namespace::group.key
```

### Examples

```php
theme::color.primary              // Primary theme colour
client::company.phone             // Company phone number
system::credentials.openai_api_key // OpenAI API key
package::estimate.estimate_module  // Estimate module toggle
```

### Available Namespaces

- `client` - Client/company information
- `package` - Package module toggles
- `estimate` - Estimate area configurations
- `theme` - Theme and design settings
- `system` - System configuration

### Available Groups

Company details, social links, marketing phones, addresses (office/registered), theme (logo/colour/layout), credentials, developer credit, package modules (estimate/gallery/openai/post/project/review), and more.

## Setting Types

### Input Types

- `checkbox` - Boolean checkbox
- `color_picker` - Colour picker
- `date_picker` - Date picker
- `dynamic` - Dynamic select (options from database)
- `input` - Text input
- `select` - Static select dropdown
- `text` - Textarea

### Data Types

- `boolean` - 0 or 1
- `decimal` - Decimal number
- `integer` - Integer number
- `json` - JSON string
- `string` - Text (default)

## Events and Listeners

### SettingUpdated Event

Fired when any setting changes. Triggers listeners:

- `UpdateEstimateModuleStatus` - Toggles estimate module
- `UpdateRealGreenModuleStatus` - Toggles RealGreen integration
- `UpdateGalleryModuleStatus` - Toggles gallery module
- `UpdateOpenAIModuleStatus` - Toggles OpenAI module
- `UpdatePostModuleStatus` - Toggles post module
- `UpdateProjectModuleStatus` - Toggles project module
- `UpdateReviewModuleStatus` - Toggles review module
- `ClearMenuCache` - Clears menu cache (from bongo/menu)

### Other Events

- `CssUpdated` → UpdateFrontendCss listener
- `JsUpdated` → UpdateFrontendJs listener
- `SchemaUpdated` → UpdateCustomSchema listener

## Admin Routes

All routes prefixed with `/admin/settings` and named `backend.setting.*`:

```
GET  /admin/settings                  → Redirects to environment or client
GET  /admin/settings/client           → Client settings (manager+)
GET  /admin/settings/environment      → Environment settings (developer)
GET  /admin/settings/package          → Package toggles (developer)
GET  /admin/settings/estimate         → Estimate settings (developer)
GET  /admin/settings/system           → System settings (developer)
GET  /admin/settings/theme            → Theme settings (developer)
GET  /admin/settings/site-css         → Custom CSS editor (developer)
GET  /admin/settings/site-js          → Custom JS editor (developer)
GET  /admin/settings/site-schema      → Custom schema editor (developer)
GET  /admin/settings/clear-cache      → Clear all caches (developer)
GET  /admin/settings/compile-css      → Compile theme CSS (developer)
GET  /admin/settings/generate-sitemap → Generate sitemap (developer)
```

## Model Reference

### Setting Model

```php
use Bongo\Setting\Models\Setting;

// Get full hierarchical key
$setting->full_key;  // "namespace::group.key"

// Check state
$setting->isEnabled();   // For checkbox settings
$setting->isDisabled();  // For checkbox settings

// Check type
$setting->isCheckbox();
$setting->isColorPicker();
$setting->isDynamic();
$setting->isInput();
$setting->isSelect();
$setting->isText();

// Dynamic options (for dynamic selects)
$options = $setting->getDynamicOptions();
```

## Extending the Package

### Add a New Setting

1. Update `src/Seeders/DataSeeder.php`:

```php
private function setNewSettings(): void
{
    $rows = [
        ['name' => 'New Setting', 'value' => 'default', 'sort_order' => 1],
    ];

    foreach ($rows as $row) {
        $this->firstOrCreateSetting(array_merge([
            'namespace' => Namespaces::CLIENT,
            'group' => Groups::COMPANY,
            'type' => InputTypes::INPUT,
            'data' => DataTypes::STRING,
        ], $row));
    }
}
```

2. Run the seeder

### Add a Settings Section

1. Create controller extending `AbstractController`
2. Add routes to `src/Routes/backend.php`
3. Create views in `src/Views/backend/`

### Add Event Listener

1. Create listener class in `src/Listeners/`
2. Register in `SettingServiceProvider::$listeners`

See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed extension guides.

## Testing

```bash
# Run tests
vendor/bin/phpunit

# Run with coverage
vendor/bin/phpunit --coverage-html coverage
```

## Code Style

```bash
# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint
```

## Documentation

- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture documentation with class diagrams and flow diagrams
- [.cursorrules](.cursorrules) - Development guidelines and patterns for Cursor AI
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - Code style templates for GitHub Copilot
- [CLAUDE.md](CLAUDE.md) - Quick reference guide for Claude Code

## Dependencies

- [bongo/framework](https://bitbucket.org/designtec/framework) - Base framework classes
- [scssphp/scssphp](https://github.com/scssphp/scssphp) - SCSS compilation
- bongo/menu - Menu cache clearing
- bongo/package - Package module management
- bongo/referrer - Referrer detection
- bongo/builder - Layout management

## License

MIT License. See LICENSE file for details.

## Authors

- **Stuart Elliott** - [Bespoke UK](https://bespokeuk.com)

## Support

For issues and feature requests, please use the issue tracker at:
https://bitbucket.org/designtec/setting/issues

## Changelog

See [CHANGELOG.md](CHANGELOG.md) for version history and changes.
