<?php

namespace Bongo\Setting\Services;

use Bongo\Referrer\Services\Referrer;
use Bongo\Setting\Models\Setting;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;

class SettingManager
{
    protected Collection $items;

    public function __construct()
    {
        $config = app()->make('config');
        $this->items = new Collection();

        if ($config->get('settings.install_complete')) {
            $this->items = Cache::rememberForever("settings", function () {
                return Setting::orderBy('namespace')
                    ->orderBy('group')
                    ->orderBy('sort_order')
                    ->orderBy('key')
                    ->get();
            });
        }
    }

    public function has(string $key): bool
    {
        return $this->items->contains('full_key', $key);
    }

    public function get(string $key, $default = null): mixed
    {
        if ($this->has($key)
            && $setting = $this->items->firstWhere('full_key', $key)
        ) {
            return $setting->isCheckbox() ? intval($setting->value) : $setting->value;
        }

        return $default;
    }

    public function all(): Collection
    {
        return $this->items;
    }

    public function allByNamespace(): array
    {
        $settings = [];

        foreach ($this->items as $item) {
            $settings["{$item->namespace}"]["{$item->group}"]["{$item->key}"] = $item;
        }

        return $settings;
    }

    public function getByNamespace(string $namespace): array
    {
        $settings = [];

        if ($this->items->contains('namespace', $namespace)
            && $items = $this->items->where('namespace', $namespace)->all()
        ) {
            foreach ($items as $item) {
                $settings["{$item->group}"]["{$item->key}"] = $item;
            }
        }

        return $settings;
    }

    public function getCustomCss(): false|string|null
    {
        if (file_exists(resource_path('/sass/custom.scss'))) {
            return file_get_contents(resource_path('/sass/custom.scss'));
        }

        return null;
    }

    public function setCustomCss(?string $css): void
    {
        file_put_contents(resource_path('/sass/custom.scss'), $css);
    }

    public function getCustomJs(): false|string|null
    {
        if (file_exists(resource_path('/js/custom.js'))) {
            return file_get_contents(resource_path('/js/custom.js'));
        }

        return null;
    }

    public function setCustomJs(?string $js): void
    {
        file_put_contents(resource_path('/js/custom.js'), $js);
        file_put_contents(public_path('/js/custom.js'), $js);
    }

    public function getCustomSchema(): false|string|null
    {
        if (file_exists(resource_path('/schema/custom.json'))) {
            return file_get_contents(resource_path('/schema/custom.json'));
        }

        return null;
    }

    public function setCustomSchema(?string $schema): void
    {
        file_put_contents(resource_path('/schema/custom.json'), $schema);
    }

    public function hasTransparentHeader(): bool
    {
        return $this->get("theme::layout.transparent_header") === 1;
    }

    public function hasStickyHeader(): bool
    {
        return $this->get("theme::layout.sticky_header") === 1;
    }

    public function getHeaderClass(): string
    {
        $headerClass = '';

        if ($this->hasTransparentHeader()) {
            $headerClass .= ' is-transparent';
        }
        if ($this->hasStickyHeader()) {
            $headerClass .= ' is-sticky';
        }

        return ltrim($headerClass, ' ');
    }

    public function getClientPhoneNumber(): ?string
    {
        // Check for Google Analytics
        if (app(Referrer::class)->isGoogleAds() && ! empty($this->get("client::marketing.google_ads_phone"))) {
            return $this->get("client::marketing.google_ads_phone");
        }

        // Check for Facebook
        if (app(Referrer::class)->isFacebookAds() && ! empty($this->get("client::marketing.facebook_ads_phone"))) {
            return $this->get("client::marketing.facebook_ads_phone");
        }

        // Check for LinkedIn
        if (app(Referrer::class)->isLinkedinAds() && ! empty($this->get("client::marketing.linkedin_ads_phone"))) {
            return $this->get("client::marketing.linkedin_ads_phone");
        }

        // Check for normal company number
        if (! empty($this->get("client::company.phone"))) {
            return $this->get("client::company.phone");
        }

        return null;
    }

    public function getRegisteredAddress(): ?string
    {
        $address = '';

        if (! empty($this->get("client::registered.line_1"))) {
            $address .= $this->get("client::registered.line_1");
        }
        if (! empty($this->get("client::registered.line_2"))) {
            $address .= ', '.$this->get("client::registered.line_2");
        }
        if (! empty($this->get("client::registered.line_3"))) {
            $address .= ', '.$this->get("client::registered.line_3");
        }
        if (! empty($this->get("client::registered.city"))) {
            $address .= ', '.$this->get("client::registered.city");
        }
        if (! empty($this->get("client::registered.county"))) {
            $address .= ', '.$this->get("client::registered.county");
        }
        if (! empty($this->get("client::registered.postcode"))) {
            $address .= ', '.$this->get("client::registered.postcode");
        }

        return rtrim($address, ', ');
    }

    public function getOfficeAddress(): ?string
    {
        $address = '';

        if (! empty($this->get("client::office.line_1"))) {
            $address .= $this->get("client::office.line_1");
        }
        if (! empty($this->get("client::office.line_2"))) {
            $address .= ', '.$this->get("client::office.line_2");
        }
        if (! empty($this->get("client::office.line_3"))) {
            $address .= ', '.$this->get("client::office.line_3");
        }
        if (! empty($this->get("client::office.city"))) {
            $address .= ', '.$this->get("client::office.city");
        }
        if (! empty($this->get("client::office.county"))) {
            $address .= ', '.$this->get("client::office.county");
        }
        if (! empty($this->get("client::office.postcode"))) {
            $address .= ', '.$this->get("client::office.postcode");
        }

        return rtrim($address, ', ');
    }

    public function isNoFollow(): bool
    {
        return $this->get("system::developer.credit_type") === 'nofollow';
    }

    public function getEstimateAreas(): array
    {
        $areas = [];
        $numberOfAreas = config('estimate.number_of_areas', 5);

        for ($i = 1; $i <= $numberOfAreas; $i++) {
            if ($this->get("estimate::area_{$i}.from_latitude")
                && $this->get("estimate::area_{$i}.from_longitude")
                && $this->get("estimate::area_{$i}.radius_in_miles")
            ) {
                $areas[] = [
                    'name' => $this->get("estimate::area_{$i}.name"),
                    'latitude' => $this->get("estimate::area_{$i}.from_latitude"),
                    'longitude' => $this->get("estimate::area_{$i}.from_longitude"),
                    'fill' => $this->get("estimate::area_{$i}.fill_color"),
                    'stroke' => $this->get("estimate::area_{$i}.stroke_color"),
                    'radius' => $this->get("estimate::area_{$i}.radius_in_miles"),
                ];
            }
        }

        return $areas;
    }

    public function getOpenAIApiKey(): ?string
    {
        return $this->get("system::credentials.openai_api_key");
    }

    public function getReCaptchaSiteKey(): ?string
    {
        return ! empty($this->get("system::credentials.google_recaptcha_site_key"))
            ? $this->get("system::credentials.google_recaptcha_site_key")
            : config('recaptchav3.sitekey');
    }

    public function getReCaptchaSecretKey(): ?string
    {
        return ! empty($this->get("system::credentials.google_recaptcha_secret_key"))
            ? $this->get("system::credentials.google_recaptcha_secret_key")
            : config('recaptchav3.secret');
    }

    public function reCaptchaEnabled(): bool
    {
        return ! empty($this->getReCaptchaSiteKey())
            && ! empty($this->getReCaptchaSecretKey());
    }
}
