# Bongo Sitemap Package - Architecture Documentation

## Overview

The `bongo/sitemap` package provides automated XML sitemap generation for Laravel applications built on the Bongo CMS framework. It crawls active, SEO-indexed content from multiple CMS packages and generates search engine-friendly sitemaps with image tags and priority rankings.

**Key Capabilities:**
- Automatic daily sitemap generation via scheduled tasks
- Event-driven regeneration when content changes
- Multi-package content aggregation (pages, posts, projects, reviews)
- Image sitemap support with eager loading optimization
- SEO-aware filtering (meta_index flags, site visibility settings)
- Priority-based URL ranking for search engine guidance

## Directory Structure

```
bongo/sitemap/
├── .github/
│   └── copilot-instructions.md          # GitHub Copilot AI guidance
├── src/
│   ├── Actions/
│   │   └── GenerateSitemap.php          # Core sitemap generation logic
│   ├── Commands/
│   │   └── GenerateSitemapCommand.php   # Artisan command: sitemap:generate
│   ├── Listeners/
│   │   └── UpdateSitemap.php            # Queued event listener for content updates
│   └── SitemapServiceProvider.php       # Service provider (extends AbstractServiceProvider)
├── tests/
│   └── TestCase.php                     # Base test case with package registration
├── .cursorrules                         # Cursor AI rules
├── .editorconfig                        # Editor configuration
├── .gitignore                           # Git ignore patterns
├── ARCHITECTURE.md                      # This file
├── CLAUDE.md                            # Claude Code guidance
├── composer.json                        # Package dependencies & autoload
├── phpunit.xml                          # PHPUnit configuration
└── README.md                            # Package overview & installation

Output:
public/sitemap.xml                       # Generated XML sitemap (Laravel app)
storage/logs/sitemap.log                 # Scheduled task log output (Laravel app)
```

## Class Diagram

```
┌─────────────────────────────────────────────────────────────┐
│ AbstractServiceProvider (bongo/framework)                   │
│ ─────────────────────────────────────────────────────────   │
│ + boot(): void                                              │
│ + register(): void                                          │
│ # bootCronSchedule(Schedule $schedule): void                │
└─────────────────────────────────────────────────────────────┘
                          △
                          │ extends
                          │
┌─────────────────────────────────────────────────────────────┐
│ SitemapServiceProvider                                      │
│ ─────────────────────────────────────────────────────────   │
│ # $module: string = 'sitemap'                               │
│ # $commands: array = [GenerateSitemapCommand::class]        │
│ ─────────────────────────────────────────────────────────   │
│ # bootCronSchedule(Schedule $schedule): void                │
│   - Schedules daily sitemap generation in production        │
│   - Runs in background with no overlapping                  │
│   - Logs to storage/logs/sitemap.log                        │
└─────────────────────────────────────────────────────────────┘
                          │ registers
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│ GenerateSitemapCommand extends Command                      │
│ ─────────────────────────────────────────────────────────   │
│ # $signature: string = 'sitemap:generate'                   │
│ # $description: string = 'Generate the sitemap.'            │
│ ─────────────────────────────────────────────────────────   │
│ + handle(): void                                            │
│   - Invokes GenerateSitemap action                          │
│   - Displays success message via console_print()            │
└─────────────────────────────────────────────────────────────┘
                          │ invokes
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│ GenerateSitemap (Action)                                    │
│ ─────────────────────────────────────────────────────────   │
│ + execute(): void                                           │
│   1. Check site_visibility === 'index'                      │
│   2. Create Spatie Sitemap instance                         │
│   3. Query & add active Pages (priority 0.9-1.0)            │
│   4. Query & add active Posts/Categories (0.7-0.8)          │
│   5. Query & add active Projects/Categories (0.7-0.8)       │
│   6. Query & add active Reviews (0.6)                       │
│   7. Add image tags for models with images                  │
│   8. Write XML to public/sitemap.xml                        │
└─────────────────────────────────────────────────────────────┘
                          △
                          │ invokes
                          │
┌─────────────────────────────────────────────────────────────┐
│ UpdateSitemap (Listener) implements ShouldQueue            │
│ ─────────────────────────────────────────────────────────   │
│ + $tries: int = 3                                           │
│ ─────────────────────────────────────────────────────────   │
│ + handle($event): void                                      │
│   - Invokes GenerateSitemap action                          │
│   - Runs asynchronously via queue                           │
└─────────────────────────────────────────────────────────────┘
                          △
                          │ listens to
                          │
┌─────────────────────────────────────────────────────────────┐
│ Content Events (registered in other packages)              │
│ ─────────────────────────────────────────────────────────   │
│ - PostCreated / PostUpdated / PostDeleted                   │
│ - PageCreated / PageUpdated / PageDeleted                   │
│ - ProjectCreated / ProjectUpdated / ProjectDeleted          │
│ - etc.                                                      │
└─────────────────────────────────────────────────────────────┘

External Dependencies:
┌─────────────────────────────────────────────────────────────┐
│ Spatie\Sitemap\Sitemap                                      │
│ ─────────────────────────────────────────────────────────   │
│ + create(): Sitemap                                         │
│ + add(Url $url): Sitemap                                    │
│ + writeToFile(string $path): void                           │
└─────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────┐
│ Spatie\Sitemap\Tags\Url                                     │
│ ─────────────────────────────────────────────────────────   │
│ + __construct(string $url)                                  │
│ + setPriority(float $priority): Url                         │
│ + addImage(string $url): Url                                │
└─────────────────────────────────────────────────────────────┘
```

## Component Interactions

### Execution Flow Diagram

```
┌─────────────────────────────────────────────────────────────────────────┐
│                         SCHEDULED EXECUTION                             │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ Daily at midnight (production only)
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ Laravel Task Scheduler                                                  │
│ (bootCronSchedule in SitemapServiceProvider)                            │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ php artisan sitemap:generate
    │ - runInBackground()
    │ - withoutOverlapping()
    │ - appendOutputTo(storage/logs/sitemap.log)
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ GenerateSitemapCommand::handle()                                        │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ (new GenerateSitemap())->execute()
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ GenerateSitemap::execute()                                              │
└─────────────────────────────────────────────────────────────────────────┘
    │
    ├─> Check: setting('system::misc.site_visibility') === 'index'?
    │   └─> NO: return (exit early)
    │   └─> YES: continue
    │
    ├─> Create Spatie Sitemap instance
    │
    ├─> Query Pages
    │   ├─> Page::query()->with('images')->active()
    │   │       ->where('meta_index', Page::INDEX)->get()
    │   │
    │   └─> For each page:
    │       ├─> Create URL (priority: 1.0 for home, 0.9 for others)
    │       ├─> Add images (if hasImages())
    │       └─> Add to sitemap
    │
    ├─> Query Posts (if package()->isEnabled('post'))
    │   ├─> Post::query()->with('images')->active()
    │   │       ->where('meta_index', Post::INDEX)->get()
    │   │
    │   └─> For each post:
    │       ├─> Create URL (priority: 0.8)
    │       ├─> Add images (if hasImages())
    │       └─> Add to sitemap
    │
    ├─> Query Post Categories (if package()->isEnabled('post'))
    │   ├─> PostCategory::query()->with('images')->active()
    │   │       ->where('meta_index', PostCategory::INDEX)->get()
    │   │
    │   └─> For each category:
    │       ├─> Create URL (priority: 0.7)
    │       ├─> Add images (if hasImages())
    │       └─> Add to sitemap
    │
    ├─> Query Projects (if package()->isEnabled('project'))
    │   └─> [Similar pattern: priority 0.8]
    │
    ├─> Query Project Categories (if package()->isEnabled('project'))
    │   └─> [Similar pattern: priority 0.7]
    │
    ├─> Query Reviews (if package()->isEnabled('review'))
    │   ├─> Review::query()->active()->get()
    │   │
    │   └─> For each review:
    │       ├─> Create URL (priority: 0.6)
    │       └─> Add to sitemap
    │
    └─> Write XML: $siteMap->writeToFile(public_path('sitemap.xml'))

┌─────────────────────────────────────────────────────────────────────────┐
│ Output: public/sitemap.xml                                              │
│ <?xml version="1.0" encoding="UTF-8"?>                                  │
│ <urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"            │
│         xmlns:image="http://www.google.com/schemas/sitemap-image/1.1"> │
│   <url>                                                                 │
│     <loc>https://example.com/</loc>                                     │
│     <priority>1.0</priority>                                            │
│     <image:image>                                                       │
│       <image:loc>https://example.com/storage/image.jpg</image:loc>     │
│     </image:image>                                                      │
│   </url>                                                                │
│   ...                                                                   │
│ </urlset>                                                               │
└─────────────────────────────────────────────────────────────────────────┘
```

### Event-Driven Flow Diagram

```
┌─────────────────────────────────────────────────────────────────────────┐
│                        EVENT-DRIVEN EXECUTION                           │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ User updates content in CMS
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ Content Model Event Fired                                               │
│ (PostCreated, PageUpdated, ProjectDeleted, etc.)                        │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ Event dispatched by Laravel
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ UpdateSitemap Listener::handle($event)                                  │
│ (implements ShouldQueue - runs asynchronously)                          │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ Job dispatched to queue
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ Queue Worker picks up job                                               │
│ (3 retry attempts if failure)                                           │
└─────────────────────────────────────────────────────────────────────────┘
    │
    │ (new GenerateSitemap())->execute()
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ GenerateSitemap::execute()                                              │
│ [Same flow as scheduled execution - see diagram above]                  │
└─────────────────────────────────────────────────────────────────────────┘
    │
    ▼
┌─────────────────────────────────────────────────────────────────────────┐
│ Output: public/sitemap.xml (updated)                                    │
└─────────────────────────────────────────────────────────────────────────┘
```

## Data Flow & Query Strategy

### Query Optimization Pattern

The package uses eager loading to prevent N+1 queries:

```
┌────────────────────────────────────────────────────────────────┐
│ Query Pattern for Each Content Type                           │
└────────────────────────────────────────────────────────────────┘
    │
    ├─> Model::query()
    │       │
    │       ├─> ->with('images')         [Eager load relationship]
    │       ├─> ->active()                [Apply active scope]
    │       ├─> ->where('meta_index', Model::INDEX)  [SEO filter]
    │       └─> ->get()                   [Execute query]
    │
    └─> ->each(function (Model $model) use ($siteMap) {
            │
            ├─> Create URL with config-based prefix
            │   └─> config('model.prefix').'/'.$model->slug
            │
            ├─> Set priority based on content type
            │   └─> (new Url($url))->setPriority(0.8)
            │
            ├─> Check if model has images
            │   └─> if ($model->hasImages())
            │
            ├─> Iterate pre-loaded images (no additional query)
            │   └─> foreach ($model->images as $image)
            │       └─> $url->addImage(url(config('image.prefix').'/'.$image->name))
            │
            └─> Add URL to sitemap
                └─> $siteMap->add($url)
        })
```

### Priority Hierarchy

```
Priority  | Content Type          | Example URL
----------|-----------------------|------------------------------------------
1.0       | Homepage              | /
0.9       | Pages                 | /about, /contact, /services
0.8       | Posts                 | /blog/post-slug
0.8       | Projects              | /projects/project-slug
0.7       | Post Categories       | /blog/category/category-slug
0.7       | Project Categories    | /projects/category/category-slug
0.6       | Reviews               | /reviews/uuid
```

## Extension Points

### 1. Adding New Content Types

To add a new content type (e.g., events) to the sitemap:

**Step 1:** Add query logic in `GenerateSitemap::execute()`:

```php
// In src/Actions/GenerateSitemap.php

public function execute(): void
{
    // ... existing code ...

    // Add events
    if (package()->isEnabled('event')) {
        Event::query()
            ->with('images')
            ->active()
            ->where('meta_index', Event::INDEX)
            ->get()
            ->each(function (Event $event) use ($siteMap) {
                $url = (new Url(config('event.prefix').'/'.$event->slug))
                    ->setPriority(0.75); // Set appropriate priority

                if ($event->hasImages()) {
                    foreach ($event->images as $image) {
                        $url->addImage(url(config('image.prefix').'/'.$image->name));
                    }
                }

                $siteMap->add($url);
            });
    }

    // ... existing code ...
}
```

**Step 2:** Register listener in Event package's service provider:

```php
// In EventServiceProvider

use Bongo\Sitemap\Listeners\UpdateSitemap;

protected array $listeners = [
    EventCreated::class => [UpdateSitemap::class],
    EventUpdated::class => [UpdateSitemap::class],
    EventDeleted::class => [UpdateSitemap::class],
];
```

### 2. Custom URL Generation

Create a custom action extending GenerateSitemap:

```php
namespace App\Actions;

use Bongo\Sitemap\Actions\GenerateSitemap as BaseGenerateSitemap;
use Spatie\Sitemap\Tags\Url;

class CustomGenerateSitemap extends BaseGenerateSitemap
{
    public function execute(): void
    {
        parent::execute();

        // Add custom URLs
        $customUrls = [
            '/custom-page-1' => 0.8,
            '/custom-page-2' => 0.7,
        ];

        $siteMap = Sitemap::create();

        foreach ($customUrls as $url => $priority) {
            $siteMap->add(
                (new Url($url))->setPriority($priority)
            );
        }

        $siteMap->writeToFile(public_path('sitemap.xml'));
    }
}
```

Then swap in your custom action:

```php
// In GenerateSitemapCommand

use App\Actions\CustomGenerateSitemap;

public function handle()
{
    (new CustomGenerateSitemap())->execute();
    console_print('Sitemaps generated :)');
}
```

### 3. Custom Scheduling

Override `bootCronSchedule()` in your application's service provider:

```php
// In AppServiceProvider or custom provider

use Illuminate\Console\Scheduling\Schedule;

protected function bootCronSchedule(Schedule $schedule): void
{
    parent::bootCronSchedule($schedule);

    // Override default schedule
    $schedule->command('sitemap:generate')
        ->hourly()  // Change frequency
        ->environments(['production', 'staging'])  // Add staging
        ->emailOutputOnFailure('admin@example.com');  // Add alerting
}
```

### 4. Custom Sitemap Filters

Add custom filtering logic in GenerateSitemap:

```php
public function execute(): void
{
    // ... existing code ...

    Page::query()
        ->with('images')
        ->active()
        ->where('meta_index', Page::INDEX)
        ->where('published_at', '<=', now())  // Custom filter
        ->whereNull('archived_at')             // Custom filter
        ->get()
        ->each(function (Page $page) use ($siteMap) {
            // ... existing code ...
        });

    // ... existing code ...
}
```

## Service Provider Integration

### AbstractServiceProvider Features Used

The SitemapServiceProvider leverages these AbstractServiceProvider capabilities:

```
┌────────────────────────────────────────────────────────────────┐
│ AbstractServiceProvider (bongo/framework)                      │
└────────────────────────────────────────────────────────────────┘
    │
    ├─> Automatic Bootstrapping
    │   ├─> Config registration from src/Config/{module}.php
    │   │   └─> Not used (no config file)
    │   │
    │   ├─> Route registration from src/Routes/
    │   │   └─> Not used (no routes)
    │   │
    │   ├─> View registration from src/Views/{module}/
    │   │   └─> Not used (no views)
    │   │
    │   ├─> Migration registration from src/Migrations/
    │   │   └─> Not used (no migrations)
    │   │
    │   └─> Translation registration from src/Translations/
    │       └─> Not used (no translations)
    │
    ├─> Command Registration
    │   └─> $commands array property
    │       └─> [GenerateSitemapCommand::class]
    │
    └─> Schedule Registration
        └─> bootCronSchedule(Schedule $schedule) method
            └─> Custom implementation: daily sitemap generation
```

### Properties Reference

| Property | Type | Value | Purpose |
|----------|------|-------|---------|
| `$module` | `string` | `'sitemap'` | Module identifier (required by AbstractServiceProvider) |
| `$commands` | `array` | `[GenerateSitemapCommand::class]` | Auto-registers artisan commands |
| `$middlewares` | `array` | `[]` | Not used (no routes) |
| `$listeners` | `array` | `[]` | Not used (listeners registered in content packages) |
| `$subscribers` | `array` | `[]` | Not used |
| `$composers` | `array` | `[]` | Not used (no views) |

## Dependencies & Integration

### Required Dependencies

```
bongo/framework (^3.0)
├─> AbstractServiceProvider
├─> Helper functions:
│   ├─> setting($key)
│   ├─> package()->isEnabled($name)
│   └─> console_print($message)
└─> Bongo Package ecosystem integration

spatie/laravel-sitemap (^6.2)
├─> Sitemap class
│   ├─> create(): Sitemap
│   ├─> add(Url $url): Sitemap
│   └─> writeToFile(string $path): void
└─> Url class
    ├─> __construct(string $url)
    ├─> setPriority(float $priority): Url
    └─> addImage(string $url): Url
```

### Optional Dependencies (Runtime Checked)

```
bongo/page (optional)
└─> Page model with images relationship

bongo/post (optional)
└─> Post, PostCategory models with images relationships

bongo/project (optional)
└─> Project, ProjectCategory models with images relationships

bongo/review (optional)
└─> Review model (no images)

bongo/image (optional)
└─> Image model & config('image.prefix')
```

### Integration Points

```
┌────────────────────────────────────────────────────────────────┐
│ Content Packages (Page, Post, Project, Review)                │
└────────────────────────────────────────────────────────────────┘
    │
    ├─> Models provide:
    │   ├─> active() scope (filters published/enabled content)
    │   ├─> meta_index constant (INDEX = should index in search)
    │   ├─> images() relationship (polymorphic hasMany)
    │   └─> hasImages() method (check if images exist)
    │
    ├─> Config files provide:
    │   ├─> prefix (URL prefix for content type)
    │   └─> category_prefix (URL prefix for categories)
    │
    └─> Events fire on:
        ├─> ModelCreated
        ├─> ModelUpdated
        └─> ModelDeleted

┌────────────────────────────────────────────────────────────────┐
│ System Settings (bongo/framework)                             │
└────────────────────────────────────────────────────────────────┘
    │
    └─> setting('system::misc.site_visibility')
        ├─> 'index' = public site, generate sitemap
        └─> 'noindex' = private site, skip sitemap
```

## Testing Strategy

### Test Structure

```
tests/
└── TestCase.php           # Base test case
    ├─> Extends Orchestra\Testbench\TestCase
    ├─> Registers SitemapServiceProvider
    └─> Configures test environment

Future test files should follow:
tests/
├── Unit/
│   ├── Actions/
│   │   └── GenerateSitemapTest.php
│   └── Listeners/
│       └── UpdateSitemapTest.php
└── Feature/
    └── Commands/
        └── GenerateSitemapCommandTest.php
```

### Testing Patterns

**Unit Test Example:**
```php
namespace Bongo\Sitemap\Tests\Unit\Actions;

use Bongo\Sitemap\Actions\GenerateSitemap;
use Bongo\Sitemap\Tests\TestCase;
use Illuminate\Support\Facades\File;

class GenerateSitemapTest extends TestCase
{
    public function test_generates_sitemap_xml_file()
    {
        // Arrange
        $sitemapPath = public_path('sitemap.xml');
        File::delete($sitemapPath);

        // Act
        (new GenerateSitemap())->execute();

        // Assert
        $this->assertFileExists($sitemapPath);
    }

    public function test_skips_generation_when_site_not_public()
    {
        // Arrange
        setting()->set('system::misc.site_visibility', 'noindex');
        $sitemapPath = public_path('sitemap.xml');
        File::delete($sitemapPath);

        // Act
        (new GenerateSitemap())->execute();

        // Assert
        $this->assertFileDoesNotExist($sitemapPath);
    }
}
```

**Feature Test Example:**
```php
namespace Bongo\Sitemap\Tests\Feature\Commands;

use Bongo\Sitemap\Tests\TestCase;

class GenerateSitemapCommandTest extends TestCase
{
    public function test_command_executes_successfully()
    {
        // Act
        $this->artisan('sitemap:generate')
            ->expectsOutput('Sitemaps generated :)')
            ->assertExitCode(0);

        // Assert
        $this->assertFileExists(public_path('sitemap.xml'));
    }
}
```

## Configuration Reference

### Schedule Configuration

```php
// Location: SitemapServiceProvider::bootCronSchedule()

$schedule->command('sitemap:generate')
    ->daily()                          // Runs at midnight daily
    ->environments(['production'])     // Production only
    ->runInBackground()                // Non-blocking execution
    ->withoutOverlapping()             // Prevents concurrent runs
    ->appendOutputTo(                  // Log output
        base_path('storage/logs/sitemap.log')
    );
```

### Queue Configuration

```php
// Location: UpdateSitemap listener

class UpdateSitemap implements ShouldQueue
{
    public int $tries = 3;  // Retry failed jobs 3 times

    // Uses default queue connection & queue name
    // Configure in config/queue.php
}
```

### URL Priority Configuration

```php
// Location: GenerateSitemap::execute()

// Priorities are hardcoded but can be extracted to config:
// config('sitemap.priorities.homepage') => 1.0
// config('sitemap.priorities.page') => 0.9
// config('sitemap.priorities.post') => 0.8
// config('sitemap.priorities.category') => 0.7
// config('sitemap.priorities.review') => 0.6
```

## Performance Considerations

### Query Optimization

1. **Eager Loading**: All relationships loaded upfront via `->with('images')`
2. **Scoped Queries**: Uses `active()` scope to filter at database level
3. **Indexed Columns**: Queries filter on `meta_index` (should be indexed)

### Execution Time

Estimated generation time based on content volume:

```
Small site (< 100 URLs):   < 1 second
Medium site (100-1000):    1-5 seconds
Large site (1000-10000):   5-30 seconds
Very large (> 10000):      30+ seconds
```

**Recommendations:**
- Run in background: `->runInBackground()` (already implemented)
- Avoid overlapping: `->withoutOverlapping()` (already implemented)
- Queue listener: `implements ShouldQueue` (already implemented)

### Memory Usage

Memory scales with:
- Number of URLs (each URL object ~1KB)
- Number of images per URL (each image tag ~500 bytes)

**Mitigation:**
- Uses chunking via `->each()` on collections
- Writes to file immediately (not accumulated in memory)
- Spatie library handles streaming efficiently

## Security Considerations

### Access Control

- No authentication required (sitemap.xml is public)
- Respects `meta_index` flags (only includes indexable content)
- Checks `site_visibility` setting (skips private sites)

### Data Exposure

**Safe:**
- URLs are already public (accessible via frontend routes)
- Images are already public (accessible via direct URLs)
- No sensitive data exposed (IDs, emails, etc.)

**Caution:**
- Review models use UUIDs in URLs (good: no sequential ID guessing)
- Ensure `active()` scope filters unpublished content
- Ensure `meta_index` flag respects `NOINDEX` setting

### File Permissions

- Writes to `public/sitemap.xml` (world-readable)
- Appends to `storage/logs/sitemap.log` (application-writable)
- No user input processed (no injection risk)

## Troubleshooting

### Common Issues

**Issue:** Sitemap not generated
- Check: `setting('system::misc.site_visibility') === 'index'`
- Check: Queue workers running (for event-driven updates)
- Check: File permissions on `public/` directory

**Issue:** Missing content in sitemap
- Check: Content has `active()` scope applied
- Check: Content has `meta_index` set to `Model::INDEX`
- Check: Package is enabled via `package()->isEnabled('package')`

**Issue:** N+1 query performance problems
- Verify: `->with('images')` is present in query
- Verify: `hasImages()` check before iterating images

**Issue:** Scheduled task not running
- Check: `php artisan schedule:list` shows the task
- Check: Cron job configured: `* * * * * php artisan schedule:run`
- Check: Environment is 'production'

### Debug Commands

```bash
# Manually generate sitemap
php artisan sitemap:generate

# View scheduled tasks
php artisan schedule:list

# Run scheduled tasks manually
php artisan schedule:run

# Check queue workers
php artisan queue:work --once

# View sitemap output
cat public/sitemap.xml

# View log output
tail -f storage/logs/sitemap.log

# Check site visibility setting
php artisan tinker
>>> setting('system::misc.site_visibility')

# Check enabled packages
php artisan tinker
>>> package()->isEnabled('post')
```

## Changelog & Version History

For detailed version history, see `CHANGELOG.md` (if available).

**Current Version:** 3.0.x (PHP 8.2+, Laravel 10+)

**Major Changes:**
- PHP 8.2+ requirement (constructor property promotion, strict types)
- Laravel 10+ requirement
- Spatie Laravel Sitemap 6.2+
- Bongo Framework 3.0 integration

## Further Reading

- [Spatie Laravel Sitemap Documentation](https://github.com/spatie/laravel-sitemap)
- [Sitemap Protocol Specification](https://www.sitemaps.org/protocol.html)
- [Google Sitemap Guidelines](https://developers.google.com/search/docs/advanced/sitemaps/overview)
- [Laravel Task Scheduling](https://laravel.com/docs/10.x/scheduling)
- [Laravel Queues & Jobs](https://laravel.com/docs/10.x/queues)
- [Bongo Framework Documentation](../../default/framework/README.md)
