# Bongo Sitemap Package - Claude Code Guidance

## Overview

The `bongo/sitemap` package provides automated XML sitemap generation for Laravel applications built with Bongo CMS. It crawls active, SEO-indexed content from multiple CMS packages and generates search engine-friendly sitemaps.

**Key Features:**
- Automatic daily sitemap generation via scheduled tasks
- Event-driven regeneration when content changes
- Multi-package content aggregation (pages, posts, projects, reviews)
- Image sitemap support with eager loading optimization
- SEO-aware filtering (meta_index, site visibility)
- Priority-based URL ranking

## Documentation

- **ARCHITECTURE.md** - Detailed architecture, class diagrams, data flow, extension points
- **.cursorrules** - Cursor AI rules with coding conventions and common tasks
- **.github/copilot-instructions.md** - GitHub Copilot patterns and templates
- **README.md** - Package overview and installation instructions

## Requirements

- PHP 8.2+
- Laravel 10+
- Bongo Framework 3.0+
- Spatie Laravel Sitemap 6.2+

## Commands

```bash
# Generate sitemap manually
php artisan sitemap:generate

# View scheduled tasks
php artisan schedule:list

# Run scheduled tasks (development)
php artisan schedule:run

# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse

# Update dependencies
composer update -W
```

## Architecture Quick Reference

### Service Provider Pattern

Extends `Bongo\Framework\Providers\AbstractServiceProvider`:

```php
class SitemapServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'sitemap';
    protected array $commands = [GenerateSitemapCommand::class];

    protected function bootCronSchedule(Schedule $schedule): void
    {
        $schedule->command('sitemap:generate')
            ->daily()
            ->environments(['production'])
            ->runInBackground()
            ->withoutOverlapping()
            ->appendOutputTo(base_path('storage/logs/sitemap.log'));
    }
}
```

**Features Used:**
- Command registration via `$commands` array
- Custom schedule via `bootCronSchedule()` method
- No routes, migrations, views, or config files

### Action-Command-Listener Pattern

```
GenerateSitemapCommand (CLI)
        ↓
GenerateSitemap (Action) ← UpdateSitemap (Listener)
        ↓
Spatie\Sitemap
        ↓
public/sitemap.xml
```

**Key Classes:**
- `GenerateSitemap::execute()` - Core business logic
- `GenerateSitemapCommand::handle()` - CLI wrapper
- `UpdateSitemap::handle($event)` - Queued listener (3 retries)

### URL Priority Hierarchy

| Priority | Content Type | Example |
|----------|--------------|---------|
| 1.0 | Homepage | `/` |
| 0.9 | Pages | `/about`, `/contact` |
| 0.8 | Posts/Projects | `/blog/post-slug` |
| 0.7 | Categories | `/blog/category/slug` |
| 0.6 | Reviews | `/reviews/uuid` |

## Key Files

| File | Purpose | LOC | Key Methods |
|------|---------|-----|-------------|
| `src/SitemapServiceProvider.php` | Registers commands & schedule | 28 | `bootCronSchedule()` |
| `src/Actions/GenerateSitemap.php` | Core sitemap generation logic | 126 | `execute()` |
| `src/Commands/GenerateSitemapCommand.php` | Artisan command interface | 20 | `handle()` |
| `src/Listeners/UpdateSitemap.php` | Event-driven regeneration | 16 | `handle($event)` |
| `tests/TestCase.php` | Base test case setup | 26 | `getPackageProviders()` |

## Code Style Summary

### PHP Standards

- **PHP Version:** 8.2+ with strict types
- **Style Guide:** Laravel Pint (PSR-12 based)
- **Indentation:** 4 spaces
- **Line Length:** 120 characters
- **Return Types:** Always declare return types

### Naming Conventions

```php
// Actions: {Verb}{Noun}
GenerateSitemap

// Commands: {Verb}{Noun}Command
GenerateSitemapCommand

// Listeners: {Verb}{Noun}
UpdateSitemap

// Signatures: {module}:{action}
sitemap:generate
```

### Query Patterns

**Always eager load relationships:**
```php
// ✅ Correct
Page::query()
    ->with('images')
    ->active()
    ->where('meta_index', Page::INDEX)
    ->get()
    ->each(function (Page $page) use ($siteMap) {
        // ...
    });

// ❌ Avoid - N+1 queries
Page::query()->active()->get()->each(function (Page $page) {
    foreach ($page->images as $image) { // N+1
        // ...
    }
});
```

**Check package availability:**
```php
if (package()->isEnabled('post')) {
    Post::query()->active()->get()->each(...);
}
```

## Common Development Tasks

### Adding a New Content Type

1. Add query block in `GenerateSitemap::execute()`:

```php
if (package()->isEnabled('event')) {
    Event::query()
        ->with('images')
        ->active()
        ->where('meta_index', Event::INDEX)
        ->get()
        ->each(function (Event $event) use ($siteMap) {
            $url = (new Url(config('event.prefix').'/'.$event->slug))
                ->setPriority(0.75);

            if ($event->hasImages()) {
                foreach ($event->images as $image) {
                    $url->addImage(url(config('image.prefix').'/'.$image->name));
                }
            }

            $siteMap->add($url);
        });
}
```

2. Register listener in Event package's service provider:

```php
protected array $listeners = [
    EventCreated::class => [
        \Bongo\Sitemap\Listeners\UpdateSitemap::class,
    ],
    EventUpdated::class => [
        \Bongo\Sitemap\Listeners\UpdateSitemap::class,
    ],
];
```

### Modifying URL Priorities

Edit priority values in `GenerateSitemap::execute()`:

```php
// Location: src/Actions/GenerateSitemap.php

(new Url($page->slug))->setPriority(0.9);      // Pages
(new Url($post->url))->setPriority(0.8);       // Posts
(new Url($category->url))->setPriority(0.7);   // Categories
```

### Testing Sitemap Generation

```bash
# Generate manually
php artisan sitemap:generate

# Verify output
cat public/sitemap.xml

# Check schedule
php artisan schedule:list

# Run schedule (dev)
php artisan schedule:run
```

## Integration with Bongo Framework

### AbstractServiceProvider Features

- **Commands:** Registered via `$commands` array
- **Schedule:** Custom `bootCronSchedule()` for daily task
- **Logging:** Appends to `storage/logs/sitemap.log`

### Helper Functions

```php
// Check site visibility
setting('system::misc.site_visibility') === 'index'

// Check package availability
package()->isEnabled('post')

// CLI output
console_print('Sitemaps generated :)')

// Generate URLs
url(config('image.prefix').'/'.$image->name)
```

### Dependencies

**Required:**
- `bongo/framework` - Base provider & helpers
- `spatie/laravel-sitemap` - XML generation

**Optional (runtime checked):**
- `bongo/page` - Page models
- `bongo/post` - Post/category models
- `bongo/project` - Project/category models
- `bongo/review` - Review models
- `bongo/image` - Image relationships

## Execution Flows

### Scheduled Execution

```
Laravel Scheduler (daily, production)
    ↓
php artisan sitemap:generate (background, no overlap)
    ↓
GenerateSitemapCommand::handle()
    ↓
GenerateSitemap::execute()
    ↓
1. Check site_visibility === 'index'
2. Create Sitemap instance
3. Query Pages (0.9-1.0 priority)
4. Query Posts/Categories (0.7-0.8 priority)
5. Query Projects/Categories (0.7-0.8 priority)
6. Query Reviews (0.6 priority)
7. Add image tags where available
8. Write to public/sitemap.xml
    ↓
Log to storage/logs/sitemap.log
```

### Event-Driven Execution

```
Content Updated (PostCreated, PageUpdated, etc.)
    ↓
UpdateSitemap Listener (queued, 3 retries)
    ↓
Queue Worker
    ↓
GenerateSitemap::execute()
    ↓
[Same flow as scheduled execution]
    ↓
public/sitemap.xml (updated)
```

## Extension Points

### 1. Custom URL Generation

Extend `GenerateSitemap` and override `execute()`:

```php
namespace App\Actions;

use Bongo\Sitemap\Actions\GenerateSitemap as BaseGenerateSitemap;

class CustomGenerateSitemap extends BaseGenerateSitemap
{
    public function execute(): void
    {
        parent::execute();

        // Add custom URLs
        $siteMap = Sitemap::create();
        $siteMap->add((new Url('/custom-page'))->setPriority(0.8));
        $siteMap->writeToFile(public_path('sitemap.xml'));
    }
}
```

### 2. Custom Scheduling

Override `bootCronSchedule()` in your service provider:

```php
protected function bootCronSchedule(Schedule $schedule): void
{
    parent::bootCronSchedule($schedule);

    $schedule->command('sitemap:generate')
        ->hourly()  // Change frequency
        ->environments(['production', 'staging']);
}
```

### 3. Custom Filtering

Add filters in `GenerateSitemap::execute()`:

```php
Page::query()
    ->with('images')
    ->active()
    ->where('meta_index', Page::INDEX)
    ->where('published_at', '<=', now())  // Custom filter
    ->whereNull('archived_at')             // Custom filter
    ->get()
    ->each(...);
```

## Troubleshooting

### Sitemap Not Generated

**Check:**
- Site visibility: `setting('system::misc.site_visibility') === 'index'`
- Queue workers running (for event-driven updates)
- File permissions on `public/` directory

**Debug:**
```bash
php artisan sitemap:generate
cat public/sitemap.xml
tail -f storage/logs/sitemap.log
```

### Missing Content

**Check:**
- Content has `active()` scope applied
- Content has `meta_index` set to `Model::INDEX`
- Package enabled: `package()->isEnabled('package')`

**Debug:**
```bash
php artisan tinker
>>> Page::query()->active()->where('meta_index', 1)->count()
>>> package()->isEnabled('post')
```

### Performance Issues

**Check:**
- Eager loading: `->with('images')` present
- N+1 queries: Use Laravel Debugbar or Telescope

**Optimize:**
- Run in background: `->runInBackground()` (already implemented)
- Prevent overlap: `->withoutOverlapping()` (already implemented)
- Queue listener: `implements ShouldQueue` (already implemented)

### Scheduled Task Not Running

**Check:**
- Cron configured: `* * * * * php artisan schedule:run`
- Environment is 'production'
- Schedule list: `php artisan schedule:list`

**Debug:**
```bash
php artisan schedule:run
php artisan schedule:work
```

## Testing

### Running Tests

```bash
# All tests
vendor/bin/phpunit

# Specific test
vendor/bin/phpunit --filter test_name

# With coverage
vendor/bin/phpunit --coverage-html coverage
```

### Test Structure

```php
namespace Bongo\Sitemap\Tests;

use Bongo\Sitemap\SitemapServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [SitemapServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app)
    {
        config()->set('database.default', 'testing');
    }
}
```

## Code Quality

### Running Pint

```bash
# Check issues
vendor/bin/pint --test

# Fix issues
vendor/bin/pint

# Specific file
vendor/bin/pint src/Actions/GenerateSitemap.php
```

### Running Larastan

```bash
# Analyse all files
vendor/bin/phpstan analyse

# Specific level
vendor/bin/phpstan analyse --level=5
```

## Performance Considerations

### Query Optimization

- **Eager loading:** `->with('images')` prevents N+1 queries
- **Scoped queries:** `->active()` filters at database level
- **Indexed columns:** `meta_index` should be indexed

### Execution Time

Estimated generation time:

```
< 100 URLs:      < 1 second
100-1000 URLs:   1-5 seconds
1000-10000 URLs: 5-30 seconds
> 10000 URLs:    30+ seconds
```

### Memory Usage

Memory scales with:
- Number of URLs (~1KB per URL)
- Number of images (~500 bytes per image tag)

**Mitigation:**
- Uses `->each()` chunking on collections
- Streams to file (not accumulated in memory)

## Security Considerations

### Access Control

- Sitemap is public (no authentication required)
- Respects `meta_index` flags (only indexable content)
- Checks `site_visibility` setting (skips private sites)

### Data Exposure

**Safe:**
- URLs already public (accessible via routes)
- Images already public (direct URLs)
- No sensitive data (IDs, emails, etc.)

**Caution:**
- Ensure `active()` filters unpublished content
- Ensure `meta_index` respects NOINDEX setting
- Review URLs use UUIDs (good: no ID guessing)

## Common Pitfalls

1. **N+1 Queries:** Always use `->with('images')`
2. **Missing Package Checks:** Wrap queries in `package()->isEnabled()`
3. **Ignoring SEO Settings:** Filter by `active()` and `meta_index`
4. **Hardcoded URLs:** Use `config('post.prefix')`
5. **Blocking Listener:** Ensure queue workers run
6. **Missing Visibility Check:** Check `site_visibility === 'index'`
7. **Overlapping Runs:** Don't remove `withoutOverlapping()`

## External Resources

- [Spatie Laravel Sitemap](https://github.com/spatie/laravel-sitemap)
- [Sitemap Protocol](https://www.sitemaps.org/protocol.html)
- [Google Sitemap Guidelines](https://developers.google.com/search/docs/advanced/sitemaps/overview)
- [Laravel Task Scheduling](https://laravel.com/docs/10.x/scheduling)
- [Laravel Queues](https://laravel.com/docs/10.x/queues)

## Package Information

- **Name:** bongo/sitemap
- **Version:** 3.0.x
- **License:** MIT
- **Author:** Stuart Elliott <stuart.elliott@bespokeuk.com>
- **Repository:** https://bitbucket.org/designtec/sitemap
- **Private Registry:** https://designtecpackages.co.uk

## Quick Reference

### File Locations

```
src/SitemapServiceProvider.php       - Service provider
src/Actions/GenerateSitemap.php      - Core logic
src/Commands/GenerateSitemapCommand.php - CLI command
src/Listeners/UpdateSitemap.php      - Event listener
tests/TestCase.php                   - Test base class

Output:
public/sitemap.xml                   - Generated sitemap
storage/logs/sitemap.log             - Scheduled task log
```

### Key Methods

```php
// Service Provider
SitemapServiceProvider::bootCronSchedule(Schedule $schedule): void

// Action
GenerateSitemap::execute(): void

// Command
GenerateSitemapCommand::handle(): void

// Listener
UpdateSitemap::handle($event): void
```

### Configuration

```php
// Schedule (SitemapServiceProvider)
->daily()                    // Frequency
->environments(['production']) // Environments
->runInBackground()          // Non-blocking
->withoutOverlapping()       // Prevent concurrent runs

// Listener (UpdateSitemap)
public int $tries = 3;       // Retry attempts
implements ShouldQueue       // Queue async
```

## Next Steps

1. **Read ARCHITECTURE.md** for detailed design patterns and data flow
2. **Check .cursorrules** for Cursor AI coding conventions
3. **Review .github/copilot-instructions.md** for GitHub Copilot patterns
4. **Run tests** to verify package functionality
5. **Generate sitemap** manually to test integration
