# Bongo Sitemap Package

Automated XML sitemap generation for Laravel applications built with Bongo CMS. This package crawls active, SEO-indexed content from multiple CMS packages and generates search engine-friendly sitemaps with image tags and priority rankings.

[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-8892BF.svg)](https://php.net/)
[![Laravel Version](https://img.shields.io/badge/laravel-%5E10.0-FF2D20.svg)](https://laravel.com/)
[![License](https://img.shields.io/badge/license-MIT-green.svg)](LICENSE)

## Features

- ✅ **Automatic Daily Generation** - Scheduled task runs at midnight in production
- ✅ **Event-Driven Updates** - Regenerates when content changes (queued)
- ✅ **Multi-Package Support** - Aggregates pages, posts, projects, and reviews
- ✅ **Image Sitemap** - Includes image tags with eager loading optimization
- ✅ **SEO-Aware** - Respects `meta_index` flags and site visibility settings
- ✅ **Priority Ranking** - Assigns priorities based on content importance
- ✅ **Zero Configuration** - Works out of the box with sensible defaults

## Requirements

- PHP 8.2 or higher
- Laravel 10.0 or higher
- Bongo Framework 3.0 or higher

## Installation

### Via Composer

```bash
composer require bongo/sitemap
```

### Laravel Auto-Discovery

The package uses Laravel's auto-discovery feature. The service provider will be automatically registered.

For Laravel 5.x (legacy), manually register the service provider in `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Sitemap\SitemapServiceProvider::class,
],
```

## Usage

### Manual Generation

Generate the sitemap manually via artisan command:

```bash
php artisan sitemap:generate
```

This creates `public/sitemap.xml` with all active, indexable content.

### Automatic Generation

The package automatically schedules daily sitemap generation in production environments:

```php
// Runs at midnight daily (production only)
// Logs to storage/logs/sitemap.log
// Runs in background with no overlapping
```

Ensure your cron is configured:

```bash
* * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
```

### Event-Driven Updates

The package can regenerate the sitemap when content changes. Register the listener in your content package's service provider:

```php
use Bongo\Sitemap\Listeners\UpdateSitemap;

protected array $listeners = [
    PostCreated::class => [UpdateSitemap::class],
    PostUpdated::class => [UpdateSitemap::class],
    PostDeleted::class => [UpdateSitemap::class],
];
```

**Note:** Ensure queue workers are running in production:

```bash
php artisan queue:work
```

## What Gets Included?

The sitemap includes:

### Pages (Priority: 0.9-1.0)
- All active pages with `meta_index === INDEX`
- Homepage gets priority 1.0, other pages 0.9
- Includes page images

### Posts & Categories (Priority: 0.7-0.8)
- Active posts: priority 0.8
- Active categories: priority 0.7
- Requires `bongo/post` package

### Projects & Categories (Priority: 0.7-0.8)
- Active projects: priority 0.8
- Active categories: priority 0.7
- Requires `bongo/project` package

### Reviews (Priority: 0.6)
- Active reviews with UUID-based URLs
- Requires `bongo/review` package

### Images
- All images associated with content items
- Uses image sitemap protocol

## Configuration

### Site Visibility

The package respects the system setting `system::misc.site_visibility`:

```php
// Site must be public to generate sitemap
setting('system::misc.site_visibility') === 'index'
```

If set to `noindex`, sitemap generation is skipped.

### URL Prefixes

URL prefixes are read from each package's config:

```php
config('post.prefix')              // e.g., 'blog'
config('post.category_prefix')     // e.g., 'blog/category'
config('project.prefix')           // e.g., 'projects'
config('project.category_prefix')  // e.g., 'projects/category'
config('review.prefix')            // e.g., 'reviews'
config('image.prefix')             // e.g., 'storage'
```

### Schedule Customisation

Override the schedule in your service provider:

```php
use Illuminate\Console\Scheduling\Schedule;

protected function bootCronSchedule(Schedule $schedule): void
{
    parent::bootCronSchedule($schedule);

    $schedule->command('sitemap:generate')
        ->hourly()  // Change frequency
        ->environments(['production', 'staging']);
}
```

## Package Structure

```
src/
├── Actions/
│   └── GenerateSitemap.php          # Core sitemap generation logic
├── Commands/
│   └── GenerateSitemapCommand.php   # Artisan command: sitemap:generate
├── Listeners/
│   └── UpdateSitemap.php            # Queued event listener
└── SitemapServiceProvider.php       # Service provider

tests/
└── TestCase.php                     # Base test case
```

## Testing

Run the test suite:

```bash
vendor/bin/phpunit
```

Run code style checks:

```bash
vendor/bin/pint --test
```

Fix code style issues:

```bash
vendor/bin/pint
```

## How It Works

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider`:

- Registers the `sitemap:generate` command
- Schedules daily execution in production
- Logs output to `storage/logs/sitemap.log`

### Action Class

`GenerateSitemap::execute()` performs the following:

1. Checks if site is public (`site_visibility === 'index'`)
2. Creates a Spatie Sitemap instance
3. Queries active, indexable content from each package
4. Adds URLs with appropriate priorities
5. Includes image tags where available
6. Writes XML to `public/sitemap.xml`

### Queued Listener

`UpdateSitemap` implements `ShouldQueue`:

- Listens to content change events
- Regenerates sitemap asynchronously
- Retries 3 times on failure

## Example Output

```xml
<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
        xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">
  <url>
    <loc>https://example.com/</loc>
    <priority>1.0</priority>
    <image:image>
      <image:loc>https://example.com/storage/hero.jpg</image:loc>
    </image:image>
  </url>
  <url>
    <loc>https://example.com/about</loc>
    <priority>0.9</priority>
  </url>
  <url>
    <loc>https://example.com/blog/my-post</loc>
    <priority>0.8</priority>
  </url>
</urlset>
```

## Troubleshooting

### Sitemap Not Generated

**Cause:** Site visibility is set to `noindex`

**Solution:**
```bash
php artisan tinker
>>> setting('system::misc.site_visibility')
```

### Missing Content

**Cause:** Content is not active or `meta_index` is set to `NOINDEX`

**Solution:** Check your content's SEO settings:
```bash
php artisan tinker
>>> Page::query()->active()->where('meta_index', 1)->count()
```

### Scheduled Task Not Running

**Cause:** Cron not configured or environment is not production

**Solution:**
```bash
# Check scheduled tasks
php artisan schedule:list

# Run manually
php artisan schedule:run
```

### Performance Issues

**Cause:** N+1 queries due to missing eager loading

**Solution:** The package already uses `->with('images')` for optimal performance. If issues persist, check your database indices:

```sql
-- Ensure these columns are indexed
ALTER TABLE pages ADD INDEX idx_active_meta (status, meta_index);
ALTER TABLE posts ADD INDEX idx_active_meta (status, meta_index);
```

## Advanced Usage

### Adding Custom Content Types

Extend `GenerateSitemap` and override `execute()`:

```php
namespace App\Actions;

use Bongo\Sitemap\Actions\GenerateSitemap as BaseGenerateSitemap;
use Spatie\Sitemap\Tags\Url;

class CustomGenerateSitemap extends BaseGenerateSitemap
{
    public function execute(): void
    {
        parent::execute();

        // Add custom URLs
        if (package()->isEnabled('event')) {
            Event::query()
                ->with('images')
                ->active()
                ->where('meta_index', Event::INDEX)
                ->get()
                ->each(function (Event $event) use ($siteMap) {
                    $url = (new Url(config('event.prefix').'/'.$event->slug))
                        ->setPriority(0.75);

                    if ($event->hasImages()) {
                        foreach ($event->images as $image) {
                            $url->addImage(url(config('image.prefix').'/'.$image->name));
                        }
                    }

                    $siteMap->add($url);
                });
        }
    }
}
```

Then swap in your custom action in the command.

## Dependencies

### Required

- [`bongo/framework`](https://designtecpackages.co.uk) - Base service provider and helper functions
- [`spatie/laravel-sitemap`](https://github.com/spatie/laravel-sitemap) - XML sitemap generation library

### Optional (Runtime Checked)

- `bongo/page` - Page models and routes
- `bongo/post` - Post/category models and routes
- `bongo/project` - Project/category models and routes
- `bongo/review` - Review models and routes
- `bongo/image` - Image relationships and URL generation

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture, class diagrams, data flow
- **[CLAUDE.md](CLAUDE.md)** - Claude Code guidance and quick reference
- **[.cursorrules](.cursorrules)** - Cursor AI rules and coding conventions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot patterns

## Contributing

This package is part of the Bongo CMS monorepo. For development guidelines:

1. Clone the repository
2. Install dependencies: `composer install`
3. Run tests: `vendor/bin/phpunit`
4. Fix code style: `vendor/bin/pint`
5. Submit changes via pull request

## Code Style

This package uses Laravel Pint for code style enforcement:

```bash
# Check code style
vendor/bin/pint --test

# Fix code style issues
vendor/bin/pint
```

**Standards:**
- PSR-12 compliant
- PHP 8.2+ with strict types
- 4 spaces indentation
- 120 character line length
- Return type declarations required

## Security

If you discover any security-related issues, please email stuart.elliott@bespokeuk.com instead of using the issue tracker.

## License

The MIT License (MIT). Please see [LICENSE](LICENSE) for more information.

## Credits

- **Author:** Stuart Elliott <stuart.elliott@bespokeuk.com>
- **Company:** [Bespoke UK](https://bespokeuk.com)
- **Package:** Part of the Bongo CMS ecosystem

## Links

- **Repository:** https://bitbucket.org/designtec/sitemap
- **Private Composer Registry:** https://designtecpackages.co.uk
- **Documentation:** See `ARCHITECTURE.md` and `CLAUDE.md`
- **Spatie Sitemap:** https://github.com/spatie/laravel-sitemap

## Version History

See [CHANGELOG.md](CHANGELOG.md) for version history and release notes.

**Current Version:** 3.0.x

**Requirements:**
- PHP 8.2+
- Laravel 10+
- Bongo Framework 3.0+

## Support

For support, please:

1. Check the documentation (ARCHITECTURE.md, CLAUDE.md)
2. Review troubleshooting section above
3. Contact: stuart.elliott@bespokeuk.com

## FAQ

**Q: How often does the sitemap update?**
A: Daily at midnight in production (scheduled), or immediately on content changes (if event listeners are configured and queue workers are running).

**Q: Does this affect site performance?**
A: No. Generation runs in the background with no overlapping, and uses optimised queries with eager loading.

**Q: Can I customise which content is included?**
A: Yes. Extend the `GenerateSitemap` action class and override the `execute()` method.

**Q: Why isn't my content showing in the sitemap?**
A: Check that content is active (`status === 'published'`) and has `meta_index` set to `INDEX` (not `NOINDEX`).

**Q: Can I generate multiple sitemaps?**
A: The package generates a single `sitemap.xml`. For multiple sitemaps or sitemap indices, extend the action class or use Spatie's sitemap index feature.

**Q: Is this compatible with Laravel 11?**
A: The package targets Laravel 10, but should work with Laravel 11. Test thoroughly before using in production.

## Roadmap

Potential future enhancements:

- [ ] Sitemap index support for large sites (50,000+ URLs)
- [ ] Configurable priorities via config file
- [ ] Support for additional content types (products, courses, etc.)
- [ ] Video sitemap support
- [ ] News sitemap support
- [ ] Multi-language sitemap support
- [ ] Sitemap ping to search engines after generation

Contributions are welcome!
