# GitHub Copilot Instructions - Bongo Sitemap Package

## Project Overview

The `bongo/sitemap` package provides automatic XML sitemap generation for Laravel applications using the Bongo CMS framework. It crawls active, indexable content from multiple CMS packages (pages, posts, projects, reviews) and generates an SEO-friendly sitemap with image tags and priority rankings.

**Technology Stack:**
- PHP 8.2+
- Laravel 10+
- Spatie Laravel Sitemap
- Bongo Framework (AbstractServiceProvider pattern)

## Key Classes & Relationships

### Class Hierarchy

```
AbstractServiceProvider (bongo/framework)
    ↑
    └── SitemapServiceProvider
            ├── registers → GenerateSitemapCommand
            └── schedules → sitemap:generate (daily)
                                ↓
                         invokes → GenerateSitemap::execute()
                                        ↓
                                   Spatie\Sitemap
                                        ↓
                                  public/sitemap.xml

Event System Flow:
    Content Events (PostCreated, PageUpdated, etc.)
            ↓
    UpdateSitemap (Queued Listener)
            ↓
    GenerateSitemap::execute()
```

### Core Classes

#### 1. SitemapServiceProvider

**Purpose:** Registers commands and schedules daily sitemap generation.

```php
namespace Bongo\Sitemap;

use Bongo\Framework\Providers\AbstractServiceProvider;

class SitemapServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'sitemap';
    protected array $commands = [GenerateSitemapCommand::class];

    protected function bootCronSchedule(Schedule $schedule): void
    {
        $schedule->command('sitemap:generate')
            ->daily()
            ->environments(['production'])
            ->runInBackground()
            ->withoutOverlapping()
            ->appendOutputTo(base_path('storage/logs/sitemap.log'));
    }
}
```

**Key Features:**
- Extends AbstractServiceProvider for automatic bootstrapping
- No routes, migrations, views, or config files
- Schedules daily execution at midnight (production only)
- Logs output to `storage/logs/sitemap.log`

#### 2. GenerateSitemap (Action)

**Purpose:** Contains core business logic for sitemap generation.

```php
namespace Bongo\Sitemap\Actions;

class GenerateSitemap
{
    public function execute(): void
    {
        // 1. Check site visibility
        if (setting('system::misc.site_visibility') !== 'index') {
            return;
        }

        // 2. Create sitemap instance
        $siteMap = Sitemap::create();

        // 3. Add pages with priority 0.9-1.0
        Page::query()
            ->with('images')
            ->active()
            ->where('meta_index', Page::INDEX)
            ->get()
            ->each(function (Page $page) use ($siteMap) {
                $url = $page->is_home_page == 1
                    ? (new Url('/'))->setPriority(1)
                    : (new Url($page->slug))->setPriority(0.9);

                if ($page->hasImages()) {
                    foreach ($page->images as $image) {
                        $url->addImage(url(config('image.prefix').'/'.$image->name));
                    }
                }
                $siteMap->add($url);
            });

        // 4. Conditionally add posts, projects, reviews
        if (package()->isEnabled('post')) {
            // Add posts with priority 0.8
            // Add categories with priority 0.7
        }

        // 5. Write to file
        $siteMap->writeToFile(public_path('sitemap.xml'));
    }
}
```

**Key Responsibilities:**
- Checks if site is public (`site_visibility === 'index'`)
- Queries each content type with eager loading
- Respects SEO settings (`meta_index`, `active()` scope)
- Adds URLs with priority hierarchy
- Includes image sitemap tags
- Writes XML to `public/sitemap.xml`

#### 3. GenerateSitemapCommand

**Purpose:** Artisan command wrapper for CLI execution.

```php
namespace Bongo\Sitemap\Commands;

class GenerateSitemapCommand extends Command
{
    protected $signature = 'sitemap:generate';
    protected $description = 'Generate the sitemap.';

    public function handle()
    {
        (new GenerateSitemap())->execute();
        console_print('Sitemaps generated :)');
    }
}
```

**Usage:**
```bash
php artisan sitemap:generate
```

#### 4. UpdateSitemap (Listener)

**Purpose:** Queued listener for event-driven sitemap regeneration.

```php
namespace Bongo\Sitemap\Listeners;

class UpdateSitemap implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        (new GenerateSitemap())->execute();
    }
}
```

**Integration Example:**
```php
// In PostServiceProvider
protected array $listeners = [
    PostCreated::class => [UpdateSitemap::class],
    PostUpdated::class => [UpdateSitemap::class],
    PostDeleted::class => [UpdateSitemap::class],
];
```

## Code Style Templates

### Action Class Template

```php
<?php

namespace Bongo\Sitemap\Actions;

use Spatie\Sitemap\Sitemap;
use Spatie\Sitemap\Tags\Url;

class GenerateSitemap
{
    public function execute(): void
    {
        // Early return if conditions not met
        if (setting('system::misc.site_visibility') !== 'index') {
            return;
        }

        $siteMap = Sitemap::create();

        // Query pattern: eager load, scope, filter
        Model::query()
            ->with('images')
            ->active()
            ->where('meta_index', Model::INDEX)
            ->get()
            ->each(function (Model $model) use ($siteMap) {
                $url = (new Url($model->url))->setPriority(0.8);

                if ($model->hasImages()) {
                    foreach ($model->images as $image) {
                        $url->addImage(url(config('image.prefix').'/'.$image->name));
                    }
                }

                $siteMap->add($url);
            });

        $siteMap->writeToFile(public_path('sitemap.xml'));
    }
}
```

### Command Template

```php
<?php

namespace Bongo\Sitemap\Commands;

use Bongo\Sitemap\Actions\GenerateSitemap;
use Illuminate\Console\Command;

class GenerateSitemapCommand extends Command
{
    protected $signature = 'sitemap:generate';
    protected $description = 'Generate the sitemap.';

    public function handle()
    {
        (new GenerateSitemap())->execute();
        console_print('Sitemaps generated :)');
    }
}
```

### Queued Listener Template

```php
<?php

namespace Bongo\Sitemap\Listeners;

use Bongo\Sitemap\Actions\GenerateSitemap;
use Illuminate\Contracts\Queue\ShouldQueue;

class UpdateSitemap implements ShouldQueue
{
    public int $tries = 3;

    public function handle($event): void
    {
        (new GenerateSitemap())->execute();
    }
}
```

## Common Patterns

### 1. Package Availability Check

Always check if a package is enabled before querying its models:

```php
if (package()->isEnabled('post')) {
    Post::query()->active()->get()->each(...);
}

if (package()->isEnabled('project')) {
    Project::query()->active()->get()->each(...);
}
```

### 2. Eager Loading Images

Prevent N+1 queries by eager loading relationships:

```php
// ✅ Correct
Page::query()
    ->with('images')
    ->active()
    ->get()
    ->each(function (Page $page) {
        if ($page->hasImages()) {
            foreach ($page->images as $image) {
                // No additional query
            }
        }
    });

// ❌ Avoid
Page::query()
    ->active()
    ->get()
    ->each(function (Page $page) {
        foreach ($page->images as $image) {
            // N+1 query per page
        }
    });
```

### 3. SEO Filtering

Always filter by active status and meta_index flag:

```php
Model::query()
    ->active()
    ->where('meta_index', Model::INDEX)
    ->get();
```

### 4. URL Priority Hierarchy

Assign priorities based on content importance:

```php
// Homepage (most important)
(new Url('/'))->setPriority(1);

// Main pages
(new Url($page->slug))->setPriority(0.9);

// Posts and projects
(new Url(config('post.prefix').'/'.$post->slug))->setPriority(0.8);

// Categories
(new Url(config('post.category_prefix').'/'.$category->slug))->setPriority(0.7);

// Reviews
(new Url(config('review.prefix').'/'.$review->uuid))->setPriority(0.6);
```

### 5. Config-Based URL Generation

Use config values for URL prefixes to support customization:

```php
// ✅ Correct
config('post.prefix').'/'.$post->slug

// ❌ Avoid hardcoding
'/blog/'.$post->slug
```

### 6. Image Sitemap Tags

Add image tags when available:

```php
if ($model->hasImages()) {
    foreach ($model->images as $image) {
        $url->addImage(url(config('image.prefix').'/'.$image->name));
    }
}
```

## Helper Functions Reference

### Bongo Framework Helpers

```php
// Get system settings
setting('system::misc.site_visibility'); // Returns 'index' or 'noindex'

// Check if package is enabled
package()->isEnabled('post'); // Returns bool

// CLI output helper
console_print('Sitemaps generated :)');
```

### Laravel Helpers

```php
// Generate absolute URL
url($path); // Returns full URL with domain

// Get config value
config('post.prefix'); // Returns configured prefix

// Get public path
public_path('sitemap.xml'); // Returns absolute path to public file

// Get base path
base_path('storage/logs/sitemap.log'); // Returns absolute path to base file
```

## Testing Patterns

### Base Test Case Setup

```php
namespace Bongo\Sitemap\Tests;

use Bongo\Sitemap\SitemapServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [
            SitemapServiceProvider::class,
        ];
    }

    protected function getEnvironmentSetUp($app)
    {
        // Configure test environment
        config()->set('database.default', 'testing');
    }
}
```

### Testing Sitemap Generation

```php
use Bongo\Sitemap\Actions\GenerateSitemap;
use Illuminate\Support\Facades\File;

public function test_generates_sitemap_xml()
{
    // Arrange
    $sitemapPath = public_path('sitemap.xml');
    File::delete($sitemapPath);

    // Act
    (new GenerateSitemap())->execute();

    // Assert
    $this->assertFileExists($sitemapPath);
    $content = File::get($sitemapPath);
    $this->assertStringContainsString('<?xml version="1.0" encoding="UTF-8"?>', $content);
    $this->assertStringContainsString('<urlset', $content);
}
```

## Common Development Tasks

### Adding a New Content Type

1. Check package availability
2. Query with eager loading
3. Filter by active and meta_index
4. Create URL with priority
5. Add images if available
6. Add to sitemap

```php
if (package()->isEnabled('event')) {
    Event::query()
        ->with('images')
        ->active()
        ->where('meta_index', Event::INDEX)
        ->get()
        ->each(function (Event $event) use ($siteMap) {
            $url = (new Url(config('event.prefix').'/'.$event->slug))
                ->setPriority(0.75);

            if ($event->hasImages()) {
                foreach ($event->images as $image) {
                    $url->addImage(url(config('image.prefix').'/'.$image->name));
                }
            }

            $siteMap->add($url);
        });
}
```

### Triggering Regeneration on Events

Register the listener in your content package's service provider:

```php
protected array $listeners = [
    EventCreated::class => [
        \Bongo\Sitemap\Listeners\UpdateSitemap::class,
    ],
    EventUpdated::class => [
        \Bongo\Sitemap\Listeners\UpdateSitemap::class,
    ],
    EventDeleted::class => [
        \Bongo\Sitemap\Listeners\UpdateSitemap::class,
    ],
];
```

## Dependencies

### Required Packages

- `bongo/framework` - Base service provider and helper functions
- `spatie/laravel-sitemap` - XML sitemap generation library

### Optional Packages (Runtime Checked)

- `bongo/page` - Page models and routes
- `bongo/post` - Post/category models and routes
- `bongo/project` - Project/category models and routes
- `bongo/review` - Review models and routes
- `bongo/image` - Image relationships and URL generation

## Commands & Scripts

```bash
# Generate sitemap manually
php artisan sitemap:generate

# View scheduled tasks
php artisan schedule:list

# Run scheduled tasks (development)
php artisan schedule:run

# Run tests
vendor/bin/phpunit

# Fix code style
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse
```

## Configuration Notes

### Schedule Configuration

The scheduled task runs:
- **Frequency:** Daily at midnight
- **Environments:** Production only
- **Execution:** Background with no overlapping
- **Logging:** Appends to `storage/logs/sitemap.log`

### Queue Configuration

The UpdateSitemap listener:
- **Queue:** Default queue
- **Retries:** 3 attempts
- **Requirement:** Queue workers must be running in production

### URL Configuration

URL prefixes are read from config files:
- `config('post.prefix')` - Post URL prefix
- `config('post.category_prefix')` - Post category prefix
- `config('project.prefix')` - Project URL prefix
- `config('project.category_prefix')` - Project category prefix
- `config('review.prefix')` - Review URL prefix
- `config('image.prefix')` - Image URL prefix

## Best Practices

1. **Always eager load relationships** to prevent N+1 queries
2. **Check package availability** before querying models
3. **Respect SEO settings** (active scope, meta_index flag)
4. **Use config values** for URL prefixes (don't hardcode)
5. **Assign appropriate priorities** based on content importance
6. **Check hasImages()** before iterating image relationships
7. **Ensure queue workers run** in production for event-driven updates
