# GitHub Copilot Instructions - Bongo User Package

## Project Overview

The `bongo/user` package provides comprehensive user authentication, authorization, and management for Laravel applications using the Bongo framework. It includes:

- Complete authentication flow (login, register, password reset, email verification)
- User management backend with CRUD operations
- Role-based access control (Developer, Manager, Employee)
- User impersonation for admins
- API endpoints with Sanctum authentication
- Image/avatar support via `bongo/image` package

## Key Classes & Relationships

### Core Model
**`Bongo\User\Models\User`** (src/Models/User.php)
- Extends: `Illuminate\Foundation\Auth\User`
- Uses traits: `HasApiTokens`, `Notifiable`, `SoftDeletes`, `HasUUID`, `HasStatus`, `HasImages`, `HasAvatar`, `HasImpersonate`, `HasName`, `HasPhone`, `HasType`
- Constants: `DEVELOPER`, `MANAGER`, `EMPLOYEE` (types); `PENDING`, `ACTIVE`, `INACTIVE` (statuses)
- Key methods: `sendPasswordResetNotification($token)`, `getAvatar()`, `isDeveloper()`, `isManager()`, `isEmployee()`

### Service Provider
**`Bongo\User\UserServiceProvider`** (src/UserServiceProvider.php)
- Extends: `Bongo\Framework\Providers\AbstractServiceProvider`
- Module name: `'user'`
- Registers: middleware (`developer`, `manager`, `employee`), commands (`ResetPasswordCommand`), composers (`UserComposer`), subscribers (`UserEventHandler`)
- Creates alias: `'User'` → `Bongo\User\Models\User`

### Controllers Structure

#### Authentication (src/Http/Controllers/Auth/)
- `AuthenticatedSessionController` - Login/logout
- `RegisteredUserController` - Registration
- `PasswordResetLinkController` - Forgot password
- `NewPasswordController` - Reset password with token
- `EmailVerificationPromptController` - Email verification prompt
- `VerifyEmailController` - Email verification handler
- `PasswordController` - Password update (authenticated)
- `ConfirmablePasswordController` - Password confirmation

#### Backend Admin (src/Http/Controllers/Backend/)
- `UserController` - CRUD operations (index, create, store, show, edit, update, destroy)
- `UserDatatableController` - Datatable data source
- `UserImageController` - Avatar image upload
- `UserImpersonationController` - Start/stop impersonation

#### API (src/Http/Controllers/Api/)
- `UserController` - RESTful API (index returns `UserResource` collection)

### Middleware
1. **`Developer`** (src/Http/Middleware/Developer.php)
   - Requires: `isDeveloper()` only
   - Redirects to `/login` if not authenticated, 404 if not developer

2. **`Manager`** (src/Http/Middleware/Manager.php)
   - Requires: `isManager()` OR `isDeveloper()`
   - Redirects to `/login` if not authenticated, 404 if insufficient permissions

3. **`Employee`** (src/Http/Middleware/Employee.php)
   - Requires: `isEmployee()` OR `isManager()` OR `isDeveloper()`
   - Most permissive staff middleware

### Event Handler
**`Bongo\User\Events\UserEventHandler`** (src/Events/UserEventHandler.php)
- Extends: `Bongo\Framework\Events\AbstractEventHandler`
- Clears `'packages'` cache when user is saved

## Code Style & Templates

### Creating a User (Backend Controller Pattern)
```php
public function store(StoreUserRequest $request)
{
    $user = $this->user;
    $user->fill($request->except('password'));
    $user->password = Hash::make($request->get('password'));
    $user->save();

    return redirect()
        ->route('backend.user.show', $user->id)
        ->success(trans('user::backend.store_success'));
}
```

### Updating a User with Optional Password
```php
public function update(UpdateUserRequest $request, User $user)
{
    $user->fill($request->except('password'));
    if ($request->has('password') && !empty($request->get('password'))) {
        $user->password = Hash::make($request->get('password'));
    }
    $user->save();

    return redirect()
        ->route('backend.user.show', $user->id)
        ->success(trans('user::backend.update_success'));
}
```

### Protecting Developer Accounts
```php
public function show(User $user)
{
    if (!user()->isDeveloper() && $user->isDeveloper()) {
        abort(404);
    }

    return view('user::backend.show', compact('user'));
}
```

### User Type Query Scopes
```php
// Get all managers
$managers = User::manager()->get();

// Get all non-developer users
$staff = User::notDeveloper()->get();

// Get active employees
$employees = User::employee()->active()->get();
```

### API Resource Pattern
```php
public function toArray(Request $request): array
{
    return [
        'id' => $this->id,
        'avatar' => $this->getAvatar(),
        'name' => $this->name,
        'email' => $this->email,
        'type' => $this->type,
        'status' => $this->status,
    ];
}
```

### Middleware Check Pattern
```php
public function handle($request, Closure $next)
{
    if (!user()) {
        return redirect()->to('/login');
    }

    if (!user()->isDeveloper() && !request()->is('admin/users/stop-impersonating')) {
        abort(404);
    }

    return $next($request);
}
```

### User Impersonation Pattern
```php
// Start impersonating (in HasImpersonate trait)
public function startImpersonating($user): void
{
    session()->put('impersonator', user());
    session()->put('impersonating', $user);
    Auth::loginUsingId($user->id);
}

// Stop impersonating
public function stopImpersonating(): void
{
    if (session()->has('impersonator')) {
        Auth::loginUsingId(session()->get('impersonator')->id);
    }
    session()->forget('impersonator');
    session()->forget('impersonating');
}

// Check if impersonating
public function isImpersonating(): bool
{
    return session()->has('impersonating');
}
```

### View Composer Pattern
```php
public function compose(View $view): void
{
    $users = Cache::remember('users', config('settings.cache_default'), function () {
        $users = User::visible()->orderBy('email', 'ASC')->pluck('email', 'id')->toArray();
        return $users ?: [];
    });

    $view->with(compact('users'));
}
```

## Common Patterns

### Route Definitions
- **Backend routes**: Use `backend.user.*` naming with `manager` middleware
- **API routes**: Use `api.user.*` naming with `auth:sanctum` middleware
- **Web routes**: Standard Laravel auth naming (`password.reset`, `login`, etc.)

### View Naming
- Backend views: `user::backend.{action}` (e.g., `user::backend.show`)
- Auth views: `user::auth.{action}` (e.g., `user::auth.login`)
- Partials: `user::backend.partials.{category}.{name}`

### Translation Keys
- Use `user::backend.{key}` for backend messages
- Examples: `user::backend.store_success`, `user::backend.delete_failed`

### Flash Message Pattern
```php
return redirect()->route('backend.user.index')->success('User created');
return back()->error('Delete failed');
```

### Form Request Validation
```php
class StoreUserRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'unique:users'],
            'password' => ['required', 'string', 'min:8'],
            'type' => ['required', 'in:developer,manager,employee'],
        ];
    }
}
```

## Database Schema Reference

### users table
```sql
id (int, primary)
uuid (string, indexed)
first_name, last_name, telephone, mobile (nullable strings)
email (string, indexed)
password (string)
remember_token (string)
email_verified_at (timestamp, nullable)
type (enum: developer, manager, employee)
status (enum: pending, active, inactive)
created_by, updated_by, deleted_by (int, nullable, indexed)
created_at, updated_at (timestamps)
deleted_at (timestamp, nullable) -- soft deletes
```

## Important Rules

1. **Always hash passwords** using `Hash::make()` before saving
2. **Protect developer accounts** - non-developers cannot view/edit/delete them
3. **Prevent self-deletion** - users cannot delete themselves
4. **Clear caches** - UserEventHandler automatically clears `packages` cache on save
5. **Use traits** - User model uses 9 traits for modularity
6. **Type hierarchy** - Developer > Manager > Employee for permissions
7. **Soft deletes** - Never hard-delete users
8. **Session regeneration** - Regenerate session on login/logout
9. **API authentication** - Use Sanctum tokens for API routes
10. **Impersonation safety** - Allow stop-impersonating even for restricted users

## Commands

### Reset Developer Password
```bash
php artisan user:reset_password
```
Generates a 32-character random password for the developer account (uses `config('developer.email')`).

## Testing

Tests are located in `tests/Feature` and `tests/Unit`. Run with:
```bash
vendor/bin/phpunit
```

## Dependencies

- `bongo/framework` - Core framework (AbstractServiceProvider, AbstractController, traits)
- `bongo/image` - Image handling (HasImages trait)
- Laravel 10+
- Laravel Sanctum (API tokens)
- PHP 8.2+
