# Claude Code Guidance - Bongo User Package

## Overview

The `bongo/user` package provides comprehensive user authentication, authorization, and management for Laravel applications built on the Bongo framework. It includes:

- Complete authentication flow (login, register, password reset, email verification)
- Role-based access control (Developer, Manager, Employee)
- User management backend with CRUD operations
- User impersonation for admins
- API endpoints with Sanctum authentication
- Integration with `bongo/framework` and `bongo/image` packages

**For detailed documentation:**
- Architecture and patterns: See [ARCHITECTURE.md](ARCHITECTURE.md)
- AI coding assistance: See [.cursorrules](.cursorrules)
- GitHub Copilot: See [.github/copilot-instructions.md](.github/copilot-instructions.md)

## Quick Links

- **Repository**: https://bitbucket.org/designtec/user
- **Package**: `bongo/user`
- **Namespace**: `Bongo\User`
- **Requirements**: PHP 8.2+, Laravel 10+

## Commands

From the `composer.json` file, available commands:

```bash
# Run tests
vendor/bin/phpunit

# Code style checking
vendor/bin/pint --test    # Check only
vendor/bin/pint           # Fix issues

# Static analysis
vendor/bin/phpstan analyse  # If configured
```

### Package-Specific Artisan Commands

```bash
# Reset developer account password
php artisan user:reset_password
```
This command generates a random 32-character password for the developer account (uses `config('developer.email')`).

## Architecture Quick Reference

### Package Structure

```
src/
├── Commands/           # Artisan commands
├── Events/            # Event handlers (cache management)
├── Factories/         # Model factories
├── Http/
│   ├── Controllers/
│   │   ├── Api/       # API endpoints (auth:sanctum)
│   │   ├── Auth/      # Authentication flow
│   │   └── Backend/   # Admin user management (auth + manager)
│   ├── Middleware/    # Authorization (developer, manager, employee)
│   ├── Requests/      # Form validation
│   ├── Resources/     # API transformers
│   └── ViewComposers/ # View data injection
├── Mailables/         # Email classes
├── Migrations/        # Database migrations
├── Models/            # Eloquent models
├── Notifications/     # Notification classes
├── Routes/            # Route definitions (api, backend, web)
├── Traits/            # Reusable traits (HasAvatar, HasType, etc.)
├── Translations/      # Language files
├── Views/             # Blade templates
└── UserServiceProvider.php
```

### Extends Bongo Framework

This package **extends** `Bongo\Framework\Providers\AbstractServiceProvider`:

- **Module name**: `'user'`
- **Auto-registers routes** from `src/Routes/`:
  - `api.php` → `/api/users` with `auth:sanctum` middleware
  - `backend.php` → `/admin/users` with `auth` + `manager` middleware
  - `web.php` → Authentication routes (login, register, password reset, etc.)
- **Auto-loads**: Config, views (`user::`), migrations, translations
- **Registers**:
  - Middleware: `developer`, `manager`, `employee`
  - Commands: `ResetPasswordCommand`
  - Composers: `UserComposer` (injects cached user list)
  - Subscribers: `UserEventHandler` (cache invalidation)

### User Model (src/Models/User.php)

The `User` model extends `Illuminate\Foundation\Auth\User` and uses 11 traits:

**From Laravel:**
- `HasApiTokens` - Sanctum API authentication
- `Notifiable` - Email/SMS notifications
- `SoftDeletes` - Soft delete support

**From bongo/framework:**
- `HasUUID` - UUID generation
- `HasStatus` - Status management (active/inactive/pending)

**From bongo/image:**
- `HasImages` - Image attachment support

**From this package:**
- `HasAvatar` - Get user avatar image
- `HasImpersonate` - User impersonation functionality
- `HasName` - Name helper methods
- `HasPhone` - Phone number helpers
- `HasType` - User type scopes and checks

### User Types & Hierarchy

```
DEVELOPER  → Full system access (can manage all users including developers)
    ↓
MANAGER    → Can manage users (except developers)
    ↓
EMPLOYEE   → Standard backend access (cannot manage users)
```

**Middleware:**
- `developer` - Only developers
- `manager` - Managers + developers
- `employee` - Employees + managers + developers (most permissive)

### User Status

- `PENDING` - Awaiting activation
- `ACTIVE` - Active user (default)
- `INACTIVE` - Disabled user

## Key Files Reference

| File Path | Purpose | Key Methods/Routes |
|-----------|---------|-------------------|
| `src/UserServiceProvider.php` | Package service provider | Registers middleware, commands, composers, subscribers |
| `src/Models/User.php` | Main user model | `isDeveloper()`, `isManager()`, `isEmployee()`, `getAvatar()` |
| `src/Routes/web.php` | Public auth routes | `/login`, `/register`, `/forgot-password`, `/logout` |
| `src/Routes/backend.php` | Admin user management | `/admin/users/*` with `manager` middleware |
| `src/Routes/api.php` | API endpoints | `/api/users` with `auth:sanctum` |
| `src/Http/Controllers/Backend/UserController.php` | CRUD operations | `index()`, `create()`, `store()`, `show()`, `edit()`, `update()`, `destroy()` |
| `src/Http/Controllers/Auth/AuthenticatedSessionController.php` | Login/logout | `create()`, `store()`, `destroy()` |
| `src/Http/Middleware/Developer.php` | Developer-only access | Checks `isDeveloper()`, redirects/404 |
| `src/Http/Middleware/Manager.php` | Manager+ access | Checks `isManager()` or `isDeveloper()` |
| `src/Http/Middleware/Employee.php` | Employee+ access | Checks `isEmployee()`, `isManager()`, or `isDeveloper()` |
| `src/Traits/HasType.php` | User type helpers | `scopeDeveloper()`, `scopeManager()`, `scopeEmployee()`, type checks |
| `src/Traits/HasImpersonate.php` | Impersonation | `startImpersonating()`, `stopImpersonating()`, `isImpersonating()` |
| `src/Commands/ResetPasswordCommand.php` | Reset password CLI | `php artisan user:reset_password` |
| `src/Events/UserEventHandler.php` | Event handling | Clears `packages` cache on user save |
| `src/Http/ViewComposers/UserComposer.php` | View data injection | Injects cached user list for dropdowns |
| `src/Migrations/2018_01_02_000001_create_users_table.php` | Database schema | Creates `users` table with type, status, audit fields |

## Code Style Summary

### Naming Conventions
- **Controllers**: `{Entity}Controller` (e.g., `UserController`)
- **Backend controllers**: `Backend\{Entity}Controller`
- **Auth controllers**: `Auth\{Action}Controller`
- **Middleware**: Single word (e.g., `Developer`, `Employee`)
- **Traits**: `Has{Feature}` (e.g., `HasAvatar`, `HasType`)
- **Routes**:
  - API: `api.user.{action}`
  - Backend: `backend.user.{action}`
  - Web: Standard Laravel naming (`password.reset`, `login`, etc.)

### Return Types
Always declare return types on public methods:
```php
public function getAvatar(): string
public function isDeveloper(): bool
public function index(): View
public function store(StoreUserRequest $request): RedirectResponse
```

### Password Hashing
Always use `Hash::make()` when setting passwords:
```php
$user->password = Hash::make($request->get('password'));
```

### Flash Messages
Use `->success()` and `->error()` macros:
```php
return redirect()->route('backend.user.show', $user)->success('User updated');
return back()->error('Delete failed');
```

### View Naming
- Backend views: `user::backend.{view}`
- Auth views: `user::auth.{view}`
- Partials: `user::backend.partials.{category}.{name}`

### Translation Keys
- Backend messages: `user::backend.{key}`
- Example: `trans('user::backend.store_success')`

## Common Patterns

### Creating a User
```php
$user = new User();
$user->fill($request->except('password'));
$user->password = Hash::make($request->get('password'));
$user->save();
```

### Querying by Type
```php
User::developer()->get();        // All developers
User::manager()->get();          // All managers
User::notDeveloper()->get();     // Exclude developers
User::employee()->active()->get(); // Active employees
```

### Checking User Type
```php
if (user()->isDeveloper()) { /* ... */ }
if (user()->isManager()) { /* ... */ }
if (user()->isEmployee()) { /* ... */ }
```

### Getting User Avatar
```php
$avatarUrl = $user->getAvatar(); // Returns image preset or '/images/default-avatar.png'
```

### User Impersonation
```php
// Start impersonating
$admin->startImpersonating($targetUser);

// Check if impersonating
if ($user->isImpersonating()) { /* ... */ }

// Stop impersonating
$admin->stopImpersonating();
```

### Protecting Developer Accounts
```php
// In backend controllers
if (!user()->isDeveloper() && $user->isDeveloper()) {
    abort(404);
}
```

### Form Request Validation
```php
public function rules(): array
{
    return [
        'first_name' => ['required', 'string', 'max:255'],
        'last_name' => ['required', 'string', 'max:255'],
        'email' => ['required', 'email', 'unique:users'],
        'password' => ['required', 'string', 'min:8'],
        'type' => ['required', 'in:developer,manager,employee'],
    ];
}
```

## Database Schema Quick Reference

### users Table
```
id, uuid (indexed)
first_name, last_name, telephone, mobile (nullable)
email (indexed), password, remember_token
email_verified_at (nullable)
type (enum: developer, manager, employee) - default: employee
status (enum: pending, active, inactive) - default: active
created_by, updated_by, deleted_by (nullable, indexed)
created_at, updated_at, deleted_at (soft deletes)
```

### password_reset_tokens Table
```
email (primary), token, created_at
```

## Important Rules & Considerations

1. **Developer Protection**: Non-developers cannot view/edit/delete developer accounts (404 error)
2. **Self-Protection**: Users cannot delete themselves
3. **Password Security**: Always use `Hash::make()` when setting passwords
4. **Soft Deletes**: Users are soft-deleted, query with `User::visible()` or include trashed with `withTrashed()`
5. **Cache Management**: `UserEventHandler` clears `packages` cache on user save
6. **View Composer**: `UserComposer` injects cached user list into dropdown views
7. **Audit Fields**: Tracks `created_by`, `updated_by`, `deleted_by` (requires framework support)
8. **User Alias**: Service provider creates `'User'` alias for `Bongo\User\Models\User`
9. **Session Regeneration**: Always regenerate session on login/logout
10. **Middleware Hierarchy**: `employee` middleware is most permissive (allows employee, manager, developer)
11. **Impersonation Safety**: Stop-impersonating route allowed even for restricted users
12. **Type Checking**: Use `isDeveloper()`, `isManager()`, `isEmployee()` methods, not direct type comparison

## Authentication Flow

### Login
1. GET `/login` → `AuthenticatedSessionController@create` (shows login form)
2. POST `/login` → `LoginRequest` validates → `AuthenticatedSessionController@store`
3. Session regenerated → Redirect to `RouteServiceProvider::HOME`

### Password Reset
1. GET `/forgot-password` → Shows request form
2. POST `/forgot-password` → Sends reset email with token
3. GET `/reset-password/{token}` → Shows reset form
4. POST `/reset-password` → Updates password → Redirect to login

### Email Verification
1. GET `/verify-email` → Prompt user to verify
2. Click email link → GET `/verify-email/{id}/{hash}`
3. Email marked as verified → Redirect to dashboard

## Middleware Usage

### In Routes
```php
// Require developer only
Route::get('/admin/system', SystemController::class)->middleware('developer');

// Require manager or higher
Route::get('/admin/users', UserController::class)->middleware('manager');

// Require any staff member
Route::get('/admin/dashboard', DashboardController::class)->middleware('employee');
```

### In Controllers
```php
public function __construct()
{
    $this->middleware('developer')->only(['destroy', 'forceDelete']);
    $this->middleware('manager')->except(['index', 'show']);
}
```

## API Usage

### Authentication
Use Laravel Sanctum for API authentication:
```php
// Create token
$token = $user->createToken('token-name')->plainTextToken;

// Use in requests
curl -H "Authorization: Bearer {token}" https://example.com/api/users
```

### API Endpoints
```
GET /api/users - List users (returns UserResource collection)
```

All API routes require `auth:sanctum` middleware.

## Testing

Run tests from package root:
```bash
vendor/bin/phpunit
```

Example test patterns:
```php
// Test user creation
User::factory()->create(['type' => User::DEVELOPER]);

// Test authentication
$this->actingAs($user)->get('/admin/users')->assertStatus(200);

// Test authorization
$this->actingAs($employee)->get('/admin/users')->assertStatus(404);

// Test impersonation
$admin->startImpersonating($user);
$this->assertTrue($admin->isImpersonating());
```

## Extension Points

### Adding New User Types
1. Update `User` model constants
2. Update migration enum
3. Add scope methods in `HasType` trait
4. Create new middleware if needed
5. Register middleware in `UserServiceProvider`

### Adding New User Fields
1. Create migration
2. Add to `$fillable` array in `User` model
3. Update form requests
4. Update views
5. Update API resource if needed

### Adding New Routes
1. Add routes to appropriate file (`api.php`, `backend.php`, `web.php`)
2. Create controller methods
3. Create form requests for validation
4. Create views if needed

See [ARCHITECTURE.md](ARCHITECTURE.md) for detailed extension examples.

## Dependencies

- **Required**:
  - `bongo/framework` ^3.0 - Core framework
  - PHP 8.2+
  - Laravel 10+
- **Used via traits**:
  - `bongo/image` - For `HasImages` trait
  - `laravel/sanctum` - For API authentication

## Security Features

1. CSRF protection on all forms
2. Password hashing with bcrypt
3. Session regeneration on login
4. Email verification support
5. Password reset with secure tokens
6. Rate limiting on password resets and verification emails
7. Middleware-based authorization
8. Developer account protection
9. User impersonation tracking
10. Soft deletes for data preservation

## Cache Keys Used

- `'users'` - Cached user list for dropdowns (cleared by `UserEventHandler`)
- `'packages'` - Cleared on user save by `UserEventHandler`

Cache duration: `config('settings.cache_default')`

## Events

The package uses Laravel's model events via `UserEventHandler`:
- `saved` - Clears `packages` cache

To listen for user events, add custom listeners to `UserServiceProvider::$listeners` or create event subscribers.

---

For comprehensive architecture details, diagrams, and patterns, refer to [ARCHITECTURE.md](ARCHITECTURE.md).
