# Bongo User Package

A comprehensive Laravel package providing user authentication, authorization, and management for the Bongo framework. Includes role-based access control, user impersonation, and complete authentication flows.

## Features

- **Complete Authentication Flow**: Login, registration, password reset, email verification
- **Role-Based Access Control**: Developer, Manager, and Employee user types with hierarchical permissions
- **User Management Backend**: Full CRUD operations with datatable support
- **User Impersonation**: Admins can impersonate users to test permissions
- **API Support**: RESTful API endpoints with Laravel Sanctum authentication
- **Avatar/Image Support**: Integration with `bongo/image` package
- **Security Features**: Developer account protection, password hashing, CSRF protection, soft deletes
- **Extensible Architecture**: Trait-based design with multiple extension points

## Requirements

- PHP 8.2+
- Laravel 10+
- `bongo/framework` ^3.0

## Installation

### Composer

```bash
composer require bongo/user
```

### Laravel Configuration

The package uses Laravel's auto-discovery feature. The service provider will be automatically registered.

### Publish Assets (Optional)

If you need to customise views, translations, or configuration:

```bash
php artisan vendor:publish --provider="Bongo\User\UserServiceProvider"
```

### Run Migrations

```bash
php artisan migrate
```

This creates:
- `users` table with type, status, and audit fields
- `password_reset_tokens` table

## Quick Start

### User Model

The `User` model is located at `Bongo\User\Models\User` and includes:

```php
use Bongo\User\Models\User;

// Create a user
$user = User::create([
    'first_name' => 'John',
    'last_name' => 'Doe',
    'email' => 'john@example.com',
    'password' => Hash::make('password'),
    'type' => User::EMPLOYEE, // or User::MANAGER, User::DEVELOPER
    'status' => User::ACTIVE,
]);
```

### User Types

Three hierarchical user types:

| Type | Constant | Description | Access Level |
|------|----------|-------------|--------------|
| Developer | `User::DEVELOPER` | Full system access | Highest |
| Manager | `User::MANAGER` | Can manage users (except developers) | Medium |
| Employee | `User::EMPLOYEE` | Standard backend access | Basic |

### User Status

| Status | Constant | Description |
|--------|----------|-------------|
| Pending | `User::PENDING` | Awaiting activation |
| Active | `User::ACTIVE` | Active user (default) |
| Inactive | `User::INACTIVE` | Disabled user |

### Checking User Type

```php
if (auth()->user()->isDeveloper()) {
    // Developer-specific functionality
}

if (auth()->user()->isManager()) {
    // Manager-specific functionality
}

if (auth()->user()->isEmployee()) {
    // Employee-specific functionality
}
```

### Query Scopes

```php
// Get all developers
User::developer()->get();

// Get all managers
User::manager()->get();

// Get active employees
User::employee()->active()->get();

// Exclude developers
User::notDeveloper()->get();
```

## Routes

### Web Authentication Routes

Public authentication routes (no prefix):

- `GET /login` - Login form
- `POST /login` - Login submission
- `GET /register` - Registration form
- `POST /register` - Registration submission
- `GET /forgot-password` - Password reset request form
- `POST /forgot-password` - Send password reset email
- `GET /reset-password/{token}` - Password reset form
- `POST /reset-password` - Reset password
- `GET /verify-email` - Email verification prompt
- `GET /verify-email/{id}/{hash}` - Email verification handler
- `GET /confirm-password` - Password confirmation
- `ANY /logout` - Logout

### Backend Admin Routes

User management routes (prefix: `/admin/users`, requires `manager` middleware):

- `GET /admin/users` - User list
- `GET /admin/users/create` - Create user form
- `POST /admin/users/store` - Store new user
- `GET /admin/users/datatable` - Datatable data source
- `GET /admin/users/{user}` - Show user
- `GET /admin/users/{user}/edit` - Edit user form
- `POST /admin/users/{user}/update` - Update user
- `DELETE /admin/users/{user}/delete` - Delete user
- `POST /admin/users/{user}/image` - Upload user avatar
- `GET /admin/users/{user}/start-impersonating` - Start impersonating user
- `GET /admin/users/stop-impersonating` - Stop impersonating

### API Routes

RESTful API routes (prefix: `/api/users`, requires `auth:sanctum` middleware):

- `GET /api/users` - List users (returns `UserResource` collection)

## Middleware

Three authorisation middleware are provided:

### `developer`
Only allows developers. Redirects to `/login` if not authenticated, `404` if not a developer.

```php
Route::get('/admin/system', SystemController::class)->middleware('developer');
```

### `manager`
Allows managers and developers. Redirects to `/login` if not authenticated, `404` if insufficient permissions.

```php
Route::get('/admin/users', UserController::class)->middleware('manager');
```

### `employee`
Allows employees, managers, and developers (most permissive). Redirects to `/login` if not authenticated, `404` if not staff.

```php
Route::get('/admin/dashboard', DashboardController::class)->middleware('employee');
```

## User Impersonation

Admins can impersonate users to test permissions and see the system as another user:

```php
// Start impersonating
$admin = auth()->user();
$targetUser = User::find($userId);
$admin->startImpersonating($targetUser);

// Check if impersonating
if ($admin->isImpersonating()) {
    // Currently impersonating another user
}

// Stop impersonating
$admin->stopImpersonating();
```

Or use the provided routes:
- `GET /admin/users/{user}/start-impersonating`
- `GET /admin/users/stop-impersonating`

## User Avatars

Users can have avatar images via the `HasAvatar` trait (requires `bongo/image` package):

```php
// Get user avatar URL
$avatarUrl = $user->getAvatar();
// Returns: image preset URL or '/images/default-avatar.png'
```

Upload avatar via backend:
```php
POST /admin/users/{user}/image
```

## Artisan Commands

### Reset Developer Password

Generate a random password for the developer account:

```bash
php artisan user:reset_password
```

Uses the email address from `config('developer.email')` and outputs a 32-character random password.

## API Authentication

The package includes API endpoints that use Laravel Sanctum for authentication:

```php
// Create a token
$token = $user->createToken('api-token')->plainTextToken;

// Use in API requests
curl -H "Authorization: Bearer {token}" https://example.com/api/users
```

## Security Features

1. **Developer Protection**: Non-developer users cannot view, edit, or delete developer accounts (404 error)
2. **Self-Protection**: Users cannot delete themselves
3. **Password Hashing**: Automatic password hashing using `Hash::make()`
4. **Session Regeneration**: Sessions regenerated on login to prevent session fixation
5. **CSRF Protection**: All forms use CSRF tokens
6. **Email Verification**: Optional email verification flow
7. **Password Reset**: Secure token-based password reset
8. **Soft Deletes**: Users are soft-deleted, not permanently removed
9. **Audit Fields**: Tracks `created_by`, `updated_by`, `deleted_by`

## Extending the Package

### Adding New User Fields

1. Create a migration:
```php
Schema::table('users', function (Blueprint $table) {
    $table->string('company')->nullable();
});
```

2. Add to `$fillable` in User model:
```php
protected $fillable = [
    'first_name', 'last_name', 'email', 'password', 'type',
    'company', // New field
];
```

3. Update form requests and views accordingly.

### Adding New User Types

1. Add constant to User model:
```php
public const ADMIN = 'admin';
```

2. Update migration enum:
```php
$table->enum('type', [User::DEVELOPER, User::MANAGER, User::EMPLOYEE, User::ADMIN]);
```

3. Add scope and check methods in `HasType` trait:
```php
public function scopeAdmin($query) { /* ... */ }
public function isAdmin(): bool { /* ... */ }
```

4. Create and register new middleware if needed.

## Database Schema

### users Table

| Column | Type | Description |
|--------|------|-------------|
| `id` | INT | Primary key |
| `uuid` | VARCHAR(36) | UUID (indexed) |
| `first_name` | VARCHAR(255) | First name (nullable) |
| `last_name` | VARCHAR(255) | Last name (nullable) |
| `telephone` | VARCHAR(255) | Landline (nullable) |
| `mobile` | VARCHAR(255) | Mobile (nullable) |
| `email` | VARCHAR(255) | Email (indexed) |
| `password` | VARCHAR(255) | Hashed password |
| `remember_token` | VARCHAR(100) | Remember me token |
| `email_verified_at` | TIMESTAMP | Email verification timestamp |
| `type` | ENUM | User type: developer, manager, employee |
| `status` | ENUM | User status: pending, active, inactive |
| `created_by` | INT | User ID who created (nullable) |
| `updated_by` | INT | User ID who updated (nullable) |
| `deleted_by` | INT | User ID who deleted (nullable) |
| `created_at` | TIMESTAMP | Creation timestamp |
| `updated_at` | TIMESTAMP | Last update timestamp |
| `deleted_at` | TIMESTAMP | Soft delete timestamp (nullable) |

### password_reset_tokens Table

| Column | Type | Description |
|--------|------|-------------|
| `email` | VARCHAR(255) | User email (primary key) |
| `token` | VARCHAR(255) | Reset token |
| `created_at` | TIMESTAMP | Token creation timestamp |

## Testing

Run tests from the package root:

```bash
vendor/bin/phpunit
```

Example test:
```php
public function test_user_can_login()
{
    $user = User::factory()->create([
        'email' => 'test@example.com',
        'password' => Hash::make('password'),
    ]);

    $response = $this->post('/login', [
        'email' => 'test@example.com',
        'password' => 'password',
    ]);

    $response->assertRedirect('/');
    $this->assertAuthenticatedAs($user);
}
```

## Code Style

This package uses Laravel Pint for code formatting:

```bash
# Check code style
vendor/bin/pint --test

# Fix code style issues
vendor/bin/pint
```

## Documentation

For detailed documentation:

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Comprehensive architecture documentation with diagrams and patterns
- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for Claude Code AI assistant
- **[.cursorrules](.cursorrules)** - Cursor AI coding assistant instructions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions

## Dependencies

- **bongo/framework** - Core framework providing base classes and traits
- **bongo/image** - Image handling (used via `HasImages` trait)
- **laravel/sanctum** - API authentication
- **illuminate/contracts** - Laravel contracts

## License

MIT Licence. See [LICENSE](LICENSE) for details.

## Author

**Stuart Elliott**
Email: stuart.elliott@bespokeuk.com
Website: https://bespokeuk.com

## Repository

https://bitbucket.org/designtec/user

## Support

For issues, feature requests, or questions, please submit feedback through the repository.
